"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const log_stream_1 = require("./log-stream");
const logs_generated_1 = require("./logs.generated");
const metric_filter_1 = require("./metric-filter");
const pattern_1 = require("./pattern");
const subscription_filter_1 = require("./subscription-filter");
/**
 * An CloudWatch Log Group
 */
class LogGroupBase extends core_1.Resource {
    /**
     * Create a new Log Stream for this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the LogStream
     */
    addStream(id, props = {}) {
        return new log_stream_1.LogStream(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Create a new Subscription Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the SubscriptionFilter
     */
    addSubscriptionFilter(id, props) {
        return new subscription_filter_1.SubscriptionFilter(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Create a new Metric Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the MetricFilter
     */
    addMetricFilter(id, props) {
        return new metric_filter_1.MetricFilter(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Extract a metric from structured log events in the LogGroup
     *
     * Creates a MetricFilter on this LogGroup that will extract the value
     * of the indicated JSON field in all records where it occurs.
     *
     * The metric will be available in CloudWatch Metrics under the
     * indicated namespace and name.
     *
     * @param jsonField JSON field to extract (example: '$.myfield')
     * @param metricNamespace Namespace to emit the metric under
     * @param metricName Name to emit the metric under
     * @returns A Metric object representing the extracted metric
     */
    extractMetric(jsonField, metricNamespace, metricName) {
        new metric_filter_1.MetricFilter(this, `${metricNamespace}_${metricName}`, {
            logGroup: this,
            metricNamespace,
            metricName,
            filterPattern: pattern_1.FilterPattern.exists(jsonField),
            metricValue: jsonField,
        });
        return new cloudwatch.Metric({ metricName, namespace: metricNamespace }).attachTo(this);
    }
    /**
     * Give permissions to write to create and write to streams in this log group
     */
    grantWrite(grantee) {
        return this.grant(grantee, 'logs:CreateLogStream', 'logs:PutLogEvents');
    }
    /**
     * Give the indicated permissions on this log group and all streams
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            // A LogGroup ARN out of CloudFormation already includes a ':*' at the end to include the log streams under the group.
            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#w2ab1c21c10c63c43c11
            resourceArns: [this.logGroupArn],
            scope: this,
        });
    }
}
/**
 * How long, in days, the log contents will be retained.
 */
var RetentionDays;
(function (RetentionDays) {
    /**
     * 1 day
     */
    RetentionDays[RetentionDays["ONE_DAY"] = 1] = "ONE_DAY";
    /**
     * 3 days
     */
    RetentionDays[RetentionDays["THREE_DAYS"] = 3] = "THREE_DAYS";
    /**
     * 5 days
     */
    RetentionDays[RetentionDays["FIVE_DAYS"] = 5] = "FIVE_DAYS";
    /**
     * 1 week
     */
    RetentionDays[RetentionDays["ONE_WEEK"] = 7] = "ONE_WEEK";
    /**
     * 2 weeks
     */
    RetentionDays[RetentionDays["TWO_WEEKS"] = 14] = "TWO_WEEKS";
    /**
     * 1 month
     */
    RetentionDays[RetentionDays["ONE_MONTH"] = 30] = "ONE_MONTH";
    /**
     * 2 months
     */
    RetentionDays[RetentionDays["TWO_MONTHS"] = 60] = "TWO_MONTHS";
    /**
     * 3 months
     */
    RetentionDays[RetentionDays["THREE_MONTHS"] = 90] = "THREE_MONTHS";
    /**
     * 4 months
     */
    RetentionDays[RetentionDays["FOUR_MONTHS"] = 120] = "FOUR_MONTHS";
    /**
     * 5 months
     */
    RetentionDays[RetentionDays["FIVE_MONTHS"] = 150] = "FIVE_MONTHS";
    /**
     * 6 months
     */
    RetentionDays[RetentionDays["SIX_MONTHS"] = 180] = "SIX_MONTHS";
    /**
     * 1 year
     */
    RetentionDays[RetentionDays["ONE_YEAR"] = 365] = "ONE_YEAR";
    /**
     * 13 months
     */
    RetentionDays[RetentionDays["THIRTEEN_MONTHS"] = 400] = "THIRTEEN_MONTHS";
    /**
     * 18 months
     */
    RetentionDays[RetentionDays["EIGHTEEN_MONTHS"] = 545] = "EIGHTEEN_MONTHS";
    /**
     * 2 years
     */
    RetentionDays[RetentionDays["TWO_YEARS"] = 731] = "TWO_YEARS";
    /**
     * 5 years
     */
    RetentionDays[RetentionDays["FIVE_YEARS"] = 1827] = "FIVE_YEARS";
    /**
     * 10 years
     */
    RetentionDays[RetentionDays["TEN_YEARS"] = 3653] = "TEN_YEARS";
    /**
     * Retain logs forever
     */
    RetentionDays[RetentionDays["INFINITE"] = 9999] = "INFINITE";
})(RetentionDays = exports.RetentionDays || (exports.RetentionDays = {}));
/**
 * Define a CloudWatch Log Group
 */
class LogGroup extends LogGroupBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.logGroupName,
        });
        let retentionInDays = props.retention;
        if (retentionInDays === undefined) {
            retentionInDays = RetentionDays.TWO_YEARS;
        }
        if (retentionInDays === Infinity || retentionInDays === RetentionDays.INFINITE) {
            retentionInDays = undefined;
        }
        if (retentionInDays !== undefined && !core_1.Token.isUnresolved(retentionInDays) && retentionInDays <= 0) {
            throw new Error(`retentionInDays must be positive, got ${retentionInDays}`);
        }
        const resource = new logs_generated_1.CfnLogGroup(this, 'Resource', {
            logGroupName: this.physicalName,
            retentionInDays,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.logGroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'logs',
            resource: 'log-group',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.logGroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import an existing LogGroup given its ARN
     */
    static fromLogGroupArn(scope, id, logGroupArn) {
        class Import extends LogGroupBase {
            constructor() {
                super(...arguments);
                this.logGroupArn = logGroupArn;
                this.logGroupName = core_1.Stack.of(scope).parseArn(logGroupArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing LogGroup given its name
     */
    static fromLogGroupName(scope, id, logGroupName) {
        class Import extends LogGroupBase {
            constructor() {
                super(...arguments);
                this.logGroupName = logGroupName;
                this.logGroupArn = core_1.Stack.of(scope).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    sep: ':',
                    resourceName: logGroupName,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.LogGroup = LogGroup;
//# sourceMappingURL=data:application/json;base64,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