"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'fixed retention'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: lib_1.RetentionDays.ONE_WEEK,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 7,
        }));
        test.done();
    },
    'default retention'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 731,
        }));
        test.done();
    },
    'infinite retention/dont delete log group by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: lib_1.RetentionDays.INFINITE,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: 'AWS::Logs::LogGroup',
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'infinite retention via legacy method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            // Don't know why TypeScript doesn't complain about passing Infinity to
            // something where an enum is expected, but better keep this behavior for
            // existing clients.
            retention: Infinity,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: 'AWS::Logs::LogGroup',
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'unresolved retention'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const parameter = new core_1.CfnParameter(stack, 'RetentionInDays', { default: 30, type: 'Number' });
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: parameter.valueAsNumber,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: {
                Ref: 'RetentionInDays',
            },
        }));
        test.done();
    },
    'will delete log group if asked to'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: Infinity,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: 'AWS::Logs::LogGroup',
                    DeletionPolicy: 'Delete',
                    UpdateReplacePolicy: 'Delete',
                },
            },
        }));
        test.done();
    },
    'import from arn'(test) {
        // GIVEN
        const stack2 = new core_1.Stack();
        // WHEN
        const imported = lib_1.LogGroup.fromLogGroupArn(stack2, 'lg', 'arn:aws:logs:us-east-1:123456789012:log-group:my-log-group');
        imported.addStream('MakeMeAStream');
        // THEN
        test.deepEqual(imported.logGroupName, 'my-log-group');
        test.deepEqual(imported.logGroupArn, 'arn:aws:logs:us-east-1:123456789012:log-group:my-log-group');
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::Logs::LogStream', {
            LogGroupName: 'my-log-group',
        }));
        test.done();
    },
    'import from name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const imported = lib_1.LogGroup.fromLogGroupName(stack, 'lg', 'my-log-group');
        imported.addStream('MakeMeAStream');
        // THEN
        test.deepEqual(imported.logGroupName, 'my-log-group');
        test.ok(/^arn:.+:logs:.+:.+:log-group:my-log-group$/.test(imported.logGroupArn), `LogGroup ARN ${imported.logGroupArn} does not match the expected pattern`);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogStream', {
            LogGroupName: 'my-log-group',
        }));
        test.done();
    },
    'extractMetric'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: '{ $.myField = "*" }',
            LogGroupName: { Ref: 'LogGroupF5B46931' },
            MetricTransformations: [
                {
                    MetricName: 'Field',
                    MetricNamespace: 'MyService',
                    MetricValue: '$.myField',
                },
            ],
        }));
        test.equal(metric.namespace, 'MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'extractMetric allows passing in namespaces with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyNamespace/MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: '{ $.myField = "*" }',
            MetricTransformations: [
                {
                    MetricName: 'Field',
                    MetricNamespace: 'MyNamespace/MyService',
                    MetricValue: '$.myField',
                },
            ],
        }));
        test.equal(metric.namespace, 'MyNamespace/MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'grant'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        const user = new iam.User(stack, 'User');
        // WHEN
        lg.grantWrite(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: ['logs:CreateLogStream', 'logs:PutLogEvents'],
                        Effect: 'Allow',
                        Resource: { 'Fn::GetAtt': ['LogGroupF5B46931', 'Arn'] },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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