"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlane extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.hostedZoneName,
                        dnsNames: props.dnsNames,
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${(_a = props.rds.secret) === null || _a === void 0 ? void 0 : _a.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                // ingressController: {
                //   enabled: true,
                //   installCRDs: false,
                //   image: {
                //     repository: 'kong/kubernetes-ingress-controller',
                //     tag: '1.3.1-alpine',
                //   },
                // },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/ca.crt',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: 'true',
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 8005,
                        containerPort: 8005,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'cluster.kong-cp.internal',
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'telemetry.kong-cp.internal',
                    },
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-cp.internal',
                    },
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    http: {
                        enabled: true,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': 'admin.kong-cp.internal',
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'manager.kong-cp.internal',
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-cp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlane = KongControlPlane;
//# sourceMappingURL=data:application/json;base64,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