"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossAccountDestination = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios.
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object.
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    /**
     *
     */
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    /**
     * Return the properties required to send subscription events to this destination.
     *
     * If necessary, the destination can use the properties of the SubscriptionFilter
     * object itself to configure its permissions to allow the subscription to write
     * to it.
     *
     * The destination may reconfigure its own permissions in response to this
     * function call.
     */
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.stringValue({
            produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,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