"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogRetention = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
const log_group_1 = require("./log-group");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs log group.
 *
 * The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 * Log group can be created in the region that is different from stack region by
 * specifying `logGroupRegion`
 */
class LogRetention extends cdk.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Custom resource provider
        const provider = this.ensureSingletonLogRetentionFunction(props);
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        const retryOptions = props.logRetentionRetryOptions;
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                LogGroupRegion: props.logGroupRegion,
                SdkRetry: retryOptions ? {
                    maxRetries: retryOptions.maxRetries,
                    base: (_a = retryOptions.base) === null || _a === void 0 ? void 0 : _a.toMilliseconds(),
                } : undefined,
                RetentionInDays: props.retention === log_group_1.RetentionDays.INFINITE ? undefined : props.retention,
            },
        });
        const logGroupName = resource.getAtt('LogGroupName').toString();
        // Append ':*' at the end of the ARN to match with how CloudFormation does this for LogGroup ARNs
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values
        this.logGroupArn = cdk.Stack.of(this).formatArn({
            region: props.logGroupRegion,
            service: 'logs',
            resource: 'log-group',
            resourceName: `${logGroupName}:*`,
            sep: ':',
        });
    }
    /**
     * Helper method to ensure that only one instance of LogRetentionFunction resources are in the stack mimicking the
     * behaviour of @aws-cdk/aws-lambda's SingletonFunction to prevent circular dependencies
     */
    ensureSingletonLogRetentionFunction(props) {
        const functionLogicalId = 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8a';
        const existing = cdk.Stack.of(this).node.tryFindChild(functionLogicalId);
        if (existing) {
            return existing;
        }
        return new LogRetentionFunction(cdk.Stack.of(this), functionLogicalId, props);
    }
}
exports.LogRetention = LogRetention;
/**
 * Private provider Lambda function to support the log retention custom resource.
 */
class LogRetentionFunction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Code
        const asset = new s3_assets.Asset(this, 'Code', {
            path: path.join(__dirname, 'log-retention-provider'),
        });
        // Role
        const role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing its name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Lambda function
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'AWS::Lambda::Function',
            properties: {
                Handler: 'index.handler',
                Runtime: 'nodejs10.x',
                Code: {
                    S3Bucket: asset.s3BucketName,
                    S3Key: asset.s3ObjectKey,
                },
                Role: role.roleArn,
            },
        });
        this.functionArn = resource.getAtt('Arn');
        // Function dependencies
        role.node.children.forEach((child) => {
            if (cdk.CfnResource.isCfnResource(child)) {
                resource.addDependsOn(child);
            }
            if (cdk.Construct.isConstruct(child) && child.node.defaultChild && cdk.CfnResource.isCfnResource(child.node.defaultChild)) {
                resource.addDependsOn(child.node.defaultChild);
            }
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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