"""Implementation of the |pydwf.utilities.openDwfDevice| function."""

from typing import Optional, Callable, Dict, Any

from pydwf.core.dwf_library import DwfLibrary
from pydwf.core.dwf_device import DwfDevice
from pydwf.core.auxiliary.enum_types import DwfEnumConfigInfo, DwfEnumFilter
from pydwf.core.auxiliary.exceptions import PyDwfError


def openDwfDevice(
            dwf: DwfLibrary,
            enum_filter: Optional[DwfEnumFilter] = None,
            serial_number: Optional[str] = None,
            score_func: Optional[Callable[[Dict[DwfEnumConfigInfo, Any]], Optional[int]]] = None
        ) -> DwfDevice:
    """Open a device identified by its serial number, optionally selecting a preferred configuration.

    This is a three-step process:

    1. The first step this function performs is to select a device for opening.

       To do this, device enumeration is performed, resulting in a list of all reachable Digilent Waveforms devices.

       For this initial enumeration process the *enum_filter* can be used to only list certain types of devices
       (for example, only Analog Discovery 2 devices, or Digital Discovery devices). If omitted (the default),
       all Digilent Waveforms devices will be listed.

       Then, if the *serial_number* parameter is given, the list will be filtered to exclude devices whose serial
       numbers do not match.

       If the list that remains has a single device, this device will be used. If not, a |PyDwfError:link| is raised.

    2. The next step is to select a *device configuration* for the selected device.

       For some Digilent Waveforms devices, multiple firmware configurations are available that make
       different tradeoffs for certain hard-coded parameters.

       For example, one configuration may support a lot of buffering memory for the |AnalogIn| instrument,
       while another configuration may support a lot of buffering memory for the |DigitalOut| instrument.

       For many use cases, the default configuration that provides a balanced tradeoff works fine.
       If no *score_func* is provided, this default configuration will be used.

       If a *score_func* is provided, it should be a function (or lambda expression) that
       takes a single parameter *configuration_info*, which is a dictionary with |DwfEnumConfigInfo:link| keys,
       and parameters values for a specific device configuration.

       The function should return *None* if the configuration is entirely unsuitable, or otherwise a score
       that reflects the suitability of that particular configuration for the task at hand.

       The *openDwfDevice* function will go through all available device configurations, construct a dictionary
       of all parameters that describe the configuration, call the *score_func* with that dictionary as a parameter,
       and examine the score value it returns. If multiple suitable device configurations are found (i.e., the
       *score_func* does not return None), it will select the configuration with the highest score.

       This may all sounds pretty complicated, but in practice this parameter is quite easy to define for most
       common use-cases.

       As an example, to select a configuration that maximizes the analog input buffer size, simply use this:

       .. code-block:: python

          from pydwf import DwfEnumConfigInfo
          from pydwf.utilities import openDwfDevice

          def maximize_analog_in_buffer_size(config_parameters):
              return config_parameters[DwfEnumConfigInfo.AnalogInBufferSize]

          with openDwfDevice(dwf, score_func = maximize_analog_in_buffer_size) as device:
              use_device_with_big_analog_in_buffer(device)

    3. As a final step, the selected device is opened using the selected device configuration, and the
       newly instantiated |DwfDevice:link| instance is returned.

    Note:
        This method can take several seconds to complete. This long duration is caused by the use
        of the |DeviceEnum.enumerateDevices:link| method.

    Parameters:
        dwf (DwfLibrary): The |DwfLibrary:link| used to open the device.
        enum_filter (Optional[DwfEnumFilter]):
            An optional filter to limit the device enumeration to certain device types.
            In None, enumerate all devices.
        serial_number (str): The serial number of the device to be opened.
            Digilent Waveforms device serial numbers consist of 12 hexadecimal digits.
        score_func (Optional[Callable[[Dict[DwfEnumConfigInfo, Any]], Optional[int]]]):
            A function to score a configuration of the selected device.
            See the description above for details.

    Returns:
        DwfDevice: The |DwfDevice:link| created as a result of this call.

    Raises:
        PyDwfError: could not select a unique candidate device, or no usable configuration detected.
    """

    if enum_filter is None:
        enum_filter = DwfEnumFilter.All

    # The dwf.deviceEnum.enumerateDevices() function builds an internal table of connected devices
    # that can be queried using other dwf.deviceEnum methods.
    num_devices = dwf.deviceEnum.enumerateDevices(enum_filter)

    # We do the explicit conversion to list to make sure the variable is a list of integers, not a range.
    candidate_devices = list(range(num_devices))

    if serial_number is not None:

        candidate_devices = [device_index for device_index in candidate_devices
                             if dwf.deviceEnum.serialNumber(device_index) == serial_number]

        if len(candidate_devices) == 0:
            raise PyDwfError("No Digilent Waveforms device found in device class '{}' with serial number '{}'.".format(
                enum_filter.name, serial_number))

        if len(candidate_devices) > 1:
            # Multiple devices found with a matching serial number; this should not happen!
            raise PyDwfError("Multiple Digilent Waveforms devices found with serial number '{}'.".format(
                serial_number))

    else:

        if len(candidate_devices) == 0:
            raise PyDwfError("No Digilent Waveforms devices found in device class '{}'.".format(enum_filter.name))

        if len(candidate_devices) > 1:
            raise PyDwfError(
                "Multiple Digilent Waveforms devices found (serial numbers: {});"
                " specify a serial number to select one.".format(
                    ", ".join(dwf.deviceEnum.serialNumber(device_index) for device_index in candidate_devices)))

    # We now have a single candidate device left.

    assert len(candidate_devices) == 1

    device_index = candidate_devices[0]

    if score_func is None:
        # If no 'score_func' was specified, just open the device with the default (first) configuration.
        return dwf.deviceControl.open(device_index)

    # The caller specified a 'score_func'.
    # We will examine all configuration and pick the one that has the highest score.

    num_config = dwf.deviceEnum.enumerateConfigurations(device_index)

    best_configuration_index = None
    best_configuration_score = None

    for configuration_index in range(num_config):

        configuration_info = {
            configuration_parameter:
                dwf.deviceEnum.configInfo(configuration_index, configuration_parameter)
                for configuration_parameter in DwfEnumConfigInfo
        }

        configuration_score = score_func(configuration_info)

        if configuration_score is None:
            continue

        if best_configuration_index is None or configuration_score > best_configuration_score:
            best_configuration_index = configuration_index
            best_configuration_score = configuration_score

    if best_configuration_index is None:
        raise PyDwfError("No acceptable configuration was found.")

    return dwf.deviceControl.open(device_index, best_configuration_index)
