from typing import Dict
from .utils.RestWrapper import APIWrapper
from .utils.AppletResponse import SyncResponse
import uuid


class QueryGenerator:
    """
    Applet for query generator, where a prompt is provided as input to the applet and the output is the query generated by the applet
    the applet requires schema and db_type as input parameters , based on the schema and db_type the applet generates the query for the prompt provided

    """

    def __init__(self, prompt: str = None, dialect: str = None, schema: str = None, db_type: str = None, additional_filters: Dict = None, conversation_id: str = None):
        self.prompt = prompt
        self.dialect = dialect
        self.schema = schema
        self.db_type = db_type
        self.additional_filters = additional_filters
        self.conversation_id = conversation_id

    def run(self) -> SyncResponse:
        """
        Method to run the Query Generator applet

        Parameters
        ----------
        prompt : str
            prompt for which the query is to be generated
        dialect : str
            name of the dialect in which the query is to be generated
        schema : str
            schema based on which the query generated
        db_type : str
            type of DB either "relational" or "Non Relational"
        additional_filters : Dict
            additional filters are dynamic query conditions which are to be applied on the query generated

        Returns
        -------
        SyncResponse object
            An object with the **answer** and **conversation_id**
        """

        if not self.prompt:
            raise Exception("prompt is required")

        if not self.dialect:
            raise Exception("dialect is required")

        if not self.schema:
            raise Exception("schema is required")

        if not self.db_type:
            raise Exception("db_type is required")

        if not self.additional_filters:
            raise Exception("additional_filters is required")

        if not self.conversation_id:
            self.conversation_id = str(uuid.uuid4())

        payload = {
            "applet_code": "query-generator",
            "conversation_id": self.conversation_id,
            "user_params": {
                "prompt": self.prompt, 
                "dialect": self.dialect,  
                "schema": self.schema,  
                "db_type": self.db_type,  
                "additional_filters": self.additional_filters
            }
        }

        api_response = APIWrapper(payload).execute_sync()
        api_response_json = api_response.json()
        content = api_response_json['content']
        response = SyncResponse(
            answer=content, conversation_id=self.conversation_id)
        resp_obj = response.obj()

        return resp_obj
