# http://pyrocko.org - GPLv3
#
# The Pyrocko Developers, 21st Century
# ---|P------/S----------~Lg----------
'''Compressed datafiles for the CRUST2.0 model by Laske, Masters and Reif.

Reference
---------

Please refer to the REM web site if you use this model:

    http://igppweb.ucsd.edu/~gabi/rem.html

or

Bassin, C., Laske, G. and Masters, G., The Current Limits of Resolution for
Surface Wave Tomography in North America, EOS Trans AGU, 81, F897, 2000. A
description of CRUST 5.1 can be found in: Mooney, Laske and Masters, Crust 5.1:
a global crustal model at 5x5 degrees, JGR, 103, 727-747, 1998.
'''

import bz2
import binascii


def split80(s):
    return '\n'.join([s[i:i+80] for i in range(0, len(s), 80)])


def encode(s):
    return split80(binascii.b2a_base64(bz2.compress(s.encode('latin-1'))))


def decode(s):
    return bz2.decompress(binascii.a2b_base64(s)).decode().lstrip()


readme = '''
QlpoOTFBWSZTWcRGC08AABx/gAtSQgBw7//yf//f4L////BgCo2222rdnLj6dGvb1i8dh2mEkQW2CUFh
oIJomaBNBNNHpR6TRkzUyZNGEBpkb1CAanpMgCRqeTVPSekZDQNAAA0AAAAanoEFNpNU9HqZNTTTIAAA
AANAAaBKekiFPKm0RoTR6E00ADQAAAAA0DgGEYTTEMAgGQAwjTJkwjAQ0EiQjIARNGmpppMoafqRhB6g
DRsoANDSmBmDFQE3XgoGCflY2ULLitODW97otTUQDvxlYiyHgbrmqVNFSnIdxZkB2KShCeTdvy/g5Z2M
JXTZmYBVTWlBpE2aEYxTkMb0+62L0kdy3og4Lywmw13ldNwEir2OCKfeuJi6UJpgRtXzoaGHHClBkpWg
fXe66M1u3aO9jqRrxRThDcsk49MTxTKyHBGkICUtt/aMS4bVTAJsh03XhwkLSHAutmIjBPf+UGGAjjL4
8WjCVuXmvwDmlpm1GrNQiZF06GNJxlUhJdxyMoblNzq+bLXB1UMYYJXnCpwoO0WilpqwLIZklSuQDkqd
UghB8k0LplnuLY7x1Gq2OLM4jFEqkCUKH+x0d4BgCPH+sZ+jG+zFhbSettVYKjdO0yDwFSBDyueJONw0
jhjwQrvghBkjac10x5m7/NFZB7y01xXJFUGefHHMuvDeKJWmqLDfzaQmck+dkbEufcoiiNDnQtfg+kHl
E4VQYbQr5MdYsagM6YpJxDW4d2bvB2pz30j3Am1To0lkIvb16uiM4zvIaiSEBcMNs4QYNe01Jxgk6b72
FDGlqmOwPkEIrGZBD9ssMq4/sR1aUja1UXFTRs97HOAu+6HSQU2ZDoSLpcaHyo8O6iaHOtbURESFVE6/
EgbKdUbCmWxS93Y7k7K73OhMp5rZS9j/g0nrrtnOAKJ9naNSIPFjeqa4bGRTxOsLrI8emwEDskuLiQTl
r3ZpMs7Ou+nNf2dmbXdNzxlib73Tv41kx5F1b9XFwIrk9L42YTMjIg11RPC/dfpfJeJdLntpuAG6ZEBe
CBhCt0BFVR2s0KScpyX7jcNzfjHZay3Td6GcJitudSlGWRAig+6TiAXa7m2Ti9+gtO3gxxgHXjalMH02
SKDmcZBfT6g8w91F+P+QCDt1+1NtESuq8U2MyOgp9APeg1Qjd4GDMKqABxgcAUGSBM16jaYzPzvDhSYc
b0hPraCqag4QgeoGy2FeR5nWzld6uIpEk8t0BL45znQKBwgJwdDe7yl1XhRFbEQssl0Y9Dn5wDr1YqMZ
Y6+jy8WAdDkHhjIThTIwwyKJiHgtucTzHI/zw1TNPUYQy8ez9+wCUPpkgYgZ7DIcxeN3IwnWS0IIEXl1
PaxDIT9K3HSvBTVR5xgsHUmJVY4YgoX1ytvXYVVjeLLrbe4rODFjBtttDNHITSmdjt9hDlMhqPc80CSD
4lN8jA5R1CEekcnSJ53ERmZTmKmgWaPYhPnkpSItTWY0pw8rPy+LycaAja69jYm0GP94LrE72ASE0jXJ
e34lUE/oy0hfhBHMKd10lA1dp/QGE4OBFfcqVMbd8ucy7hoLELmedpPPDpHg5gZm8vCWn1pTX0NBmn8R
XkJFWeaIk60Oe40PM6+OFTCdbzDg267NsEQrTUN0nKHA0c6KKGUp4KybCkros2BaRRURpppTgkppEj8P
HBWcbUQ3+SsaG5zrdMmIkAMVEFSr3J8WpS5KUHxv9R07+dp8RyhdR2FSIpRQ07Mwq+qxSbgucV2qhBkL
3QByJuwKCZDSffStomJpnGAMoMpRA37igVyyaQQJuiXShuXdos5Gh1sayRJDGqmugrXS6tD+/XnK4pDK
mGlnAYUS54HQqJxq94gMoCRgxRXusclcAdQdUPdWBXjYNiuTSXNy1enFOb2pflKYTm+aRr80lN7H6k2c
jMyJ4tG7Lnkp3NNopkVuL6VkSdVy08ZyXHQnEufI5TzsXSutK+I4NA9ORehtLHr0gT5C5Z3mJ+K6ya4I
b8urEzugCGC7vZJdsS3Q0latKjVj32gju6p7DRoefDbQ1UR3D0GFYgUOb1cNlZ1qwdcgCw5GAZViHNF1
S5hDNI+cdaIiqKLdIHutKAooIogoCItUXmV4hcSlEeYLJRS5MQpE1yrUmrBuHrFJHjd5BCL+3qNnjkum
HsxSxLNCuqu0X6Yy88wv1ywtH5IhTYQ0cyCqVmWz/DGGoEPJmE7Cbhkt8ngT1ksm2eN9FixtcfZrUyV4
jNppsGhjE59VadYW4LwN76Zc74kLcvJh1NzyuArG/Y0ISpxKWBIEqpcjSJV0Ttnf24c3KPoa8NnJOQxD
Zreb2Bzbdeb6rxea1qNZJTl3xWUFFs6E+7lnm3RUYj2/CWB8gRAeLOwDEPuNo3w07ltJ4UEe0l+BHVZK
i4CtpoWbJZTYs1TnrzYtg+9OXOE0zhG2BguLLhLgptMPGaoUcyghcaLBpTRdIdQVDjJgpFzrNTQrM6TB
zYpWkRHBEbujNQu0S8mLXaBjG3Kvcg7zcAwDViRoFS7wuYfP58Ehb41stAGOJfeYaGY0Nesggk0cjWKu
lNvUbb38NFfWIl8sHF2VlXex5gdjHAkTm7Do36XnoNE134mTnA+/Pt3RHcxKA7LWjOpyXwleLgjAjWjq
juOvpDu0zL5WheTySJXElquJ5MdyhQarmkqJIkA9Yx6muAuHh7pW3jyXcp0iISCUQTNelwohAy4mE+cQ
5oqibeNdys4MRYXu+I9mFJzvMid9SYVR9NUrWuYjoqhyWLGv0MEZSrRjAR7L4dje+9r2chi0dp0mc1Fm
L9N5DpruXbUaNXW3aysycyayGDiGLNTLDXV2LwOBOIa3Rk6KhzlJ2EzDtZh0sLepCJfFScsXrmpTucrp
xRpXiMNw4WqqBzVqicXdUGFlZju7d2W7nB7W0FbVJOPBx5dM1daVbjHeKFdCJreWsaypbFjvIfzhCYvN
P2qWCNC42J4NGLRvYrWjd8Z3kDfexbMxxtcpllrxCXTJc2dCUtIgiUq1VLW8mxd30IVqn7YgWO4UQsTF
BrRRjA3trSseQWFDRgwigFr5pbkwmuNzNWv4RgrZXSJktwMwzJDUzdSLe4ZedhbN8LI/MWDZUb8SD7rr
bKmMg6+XObZNe/FemEHyULktNAF926oDYjTtAX8ifTpcxxRyCS2nPxp9hKwxqiuaCBnraRJg29IXvPm+
H5bHNPo734u4TPasVMEV9ysOJcxZ1AYo8Byrajt9/yNn/xdyRThQkMRGC08=
'''


type2_key = '''
QlpoOTFBWSZTWTnUseEAbBBfgHgwQAf/5D////C/79/wYCd++fAHgBZSVatts2wGCqpJ4QQ4DQ9gepuc
eXc0r3W1wzrAqdgd3YnNrAIbkANZhW31yXnsvnudOzrl1qi2MO7nNztu2g9zzwAAa665mDWmtAyKBgUM
E0GRJ7U1T96JU2p6gAAAAkIIiUpP1T1PUbU0ZAAAxA0GmgADTVRoABo0ADahpoaMTIGgAAISkSU0NQ1P
1Mpmk9EeU9vVGo0mExPU9QbUGQIlBAjRNAGgIap5T1NNpqaY0GIE2oCFIgRUwqaaDE09QA89VNGRoDQA
BgCg3EiiCkFYJ9+vJuLHOU6+Hp5uPpwOn31h764ANQtmBk7zLiMKmGagoqC3s06aKeAJd6wApsjjSuoE
ekqTqHIIO5cUD8m7aiLxsXWohPh2jc3DV8nNLe8KPXOWrFtURsRJdnMc9LFClAdgjAaDKRqq6iJ1oA0U
qwxBiDBMTIlGgMVjKJoTHdxJjZOCFI7GBxLScYAPSRIFBJFVz1EZ2oDMbHUHOSA0CI7rHJQDRuDi0c5o
uE61zm5aZJpT1v5Xd9MVD1108OjPYmDjBg2DBUdVBjYwfjm7WWPns7aMTWtXbsoOtTIDZdU3RKIn5EXf
aUJEHHOQjt8adxVxyzsmuxyFDdckVM3Kpt5W1VrrKu6Tfbp17+Bk8frMgMbI405yqOX2Vjs92s2pByQa
jUyynyqov01nxtyRR7YDaTY+EXmcDG4EGMt2nVQplNbfc87fOI61dTkusozHZC5kptZmKO5JIO+XRmEw
b3z5uNZqpdUI1yY9bbswaal1Om3O18nCc6/as8mPCtrHLpnWQhrfzKqDV/NBXdopHqKTD1UU+SDGCUSE
SCVYWRlDBkOmRQhSFQwjGoQoDGEA8RmfHekCJ1rWnYJxZCEaVKVCPFwBYUkB0EIcwene3QZpuxTIELAY
QhZ9AeiXzOtnPvjzphUHuG87yuke0xagkx45xWnvRo4R0KYNoy2pzcwjpG1wNHDnGyUV4dd+FDigdCFA
kLKCV3GM7O2AXoDwEPFQ4RYDDeUQVMjMLDMDhpoYkpiUoSYqkpIWYSUNwBOaBDqjrMDiPU1e96hkJxlU
j2qLLXMiUK9VCzSCBPFB17N48mD3qMJpBdSPkqzEQTMIYkhUCii2zwFd2qCJBQ9v1GBA6KqfS0Gxgooj
GDBUIaKKCakYaSCUCB4k6IPW+u9myZigRonzqYbsSjHRmHmdblijbTaQFAtFURUVTKVFUlUVIEDeHShi
HBBbA5mvLN70puKQ3BVMZYFB1PGg3BGK7fXCO4UWkA1jAY6lGr0MDGNgsbgMkM7O4EMoSgd+nBEk7x0Q
oYHKxJQQIQIQB4aEU0CG/FTSjGewEDj1HozMM4X0gwh2ISNID2K6YGEhXnk89a6uOzzRrwVDDEE8Bn0T
jL3rEHQQBhwb1sWdwJ67x1FLqMT0IelCF437V0+kmGESwzgAtapohlHk7UNJsHBdmnUISk0BBMhUAS6F
J2cifEhBgC7Tb8i9HYwuKFIWsOUd4YV89h0YeSRjZ69HK9RWutVD29Eszxt99OtJs8NFBHrQMqecitWF
7sPj7H5BuM5fEcIHrfygMRntKm6C9sKnzHSO7B3CVd5z33xpBSFb5YQaNbXnl0BSoXQmtbRtYlLAWsQI
NYc14GI7CzWisM2+rgkKAg6ktx/H4/jWBTGZvhF8CujtB6EhKNqDiFFEECoElxID0EHiXnTeFDXqKuW3
5PjFl7OPXOe4589xqPeaV2KvSMCTMwqBXWVsV+dEKJB1G1M+HzpXwZVvmG5a6/Hdp+1oXS7z4Z8QviE0
hMZYB71dDz31Hrfs9N1WGpPSuoekYKICCCigmZB7Rw7E0wWsxiJiCcVMREMYmMQRK0dfXwlUi/ajrr7K
7aONlzrGN97y7ohTcjZCT2DnMnz5XCJO6daEmVs4nKIlSUpMhWqZZqZ17eOlHhnft6x+TOvvbXcxlL+N
Wxk0++irgyGCMgyd7nu2xfDY58IUQgsKqr4se3h7/M1tL0f4fRXCMzbXNp2bt0YWwhBtDamPctNTLWrM
k7XzsnoiypruN10eFu7G73wgRFAC0IBegyRGrMtaTAIqhZkHiKip1WQ+7ZFeuFWE2XTypjyln4qkKanL
DKhiHDZrHhr4SWh2L9WhM447Yyqm81bljy592ODt07Bl1FercqzKMqffdZ8fNMpkYpNiVSupDlKFj0Yy
6+LzpvCe7Xibfltyh1Wd3OurdUumcPHPu+v3/1/tqYxkH6rRRlVvZZuyhCyD7J4wsodW2tjFqYzZsg97
4Qg5jnOR1NSQqc1s0OEXx5xjxhusr27WJTy57o1vttshHdtsjTM3Ct0JvahvxShLKWOrfVsZY2M61+27
2429uPp29/fru535cK+Fuzhx6t62cWo6PXZO/h0rvmpGKENUqFJVFUKFKkIoCKiTjFRFqYNYrMWaNrax
VgxU6Gi1pNpiYakqLY0Uw7LpoqmikJFgIqSdbFioqlTbbbQ4P7pqfmVMsCYGBNEJIZVYIkCAlhwapYcO
mYgkQkAgXnXNp1tmKfhmpISIq/HgzNSEHypvkUD6gNIiBWMt0FoeJUGU44rtWW1d44RydRJLIPSLABxG
mEwQL49DvRtbEaM4xtOvnQPqPaYHSUUVRVFKfes/STBALj95hP09H+/AOo5KHcDrSIbZCIaUD0NJCEhI
yRk+qhRagudzga7VOTlockxluXh7LVzoVjn08w/UhBtD5PCpC6LkdWVJy369dy22DKlx/Hiq3Jvu4GJk
mFLjY2ZTn0SPPlut8i88tbcL3N8wJ2q2S8UY7unQ3Hjbb48bf0sj9U+D6UNS6or4e1dXQ8eVWmEz5Pma
Y2MabfZzsKfqM8RCUism+iukslmnC/dC7ZdyzJMj34n5n3v/fzZygCLslPl8vlrIdyncGOsMK+SVnw2F
03G3XC7ui3+k6qSSnkl19LotKRh1x0P+pInkIoLCRrjEpPV5rNwZmGRkUd5hyhCEyogu/UPV9O8A6HLF
rsu4Y7JDfpJjrcFxBagttjJCBCVBZT4QkWA5AGzVUBBIkShFFWxiI4iZOtCkUZUtYxgiHXAX0EvBgs58
QHzulYLweZATSR3wIkcQESRo+++iPQTpGLD10j1EgLNmpFjBTIvUj1AZSOJYCYVXdq/TrybH6B8OnWqc
Vz+r+zOvpJ911111lnSqveKWmdAxAlCMFBIYiKk4mDUHrnHftu78eXe7fwxvvrvsvvmmdM2mrezk6ZMH
mt/D4e+G7u7m/X53ve9730VJ8GgBoYDxxqVCSxClDCSY0yav6+Afmi+Sv3B64SlJDX/dyeXVscm4fdD0
TyNDpTvmLs6OB3sB5h7Zpfk68cDQdTuTkDh43IqQ7JdSnGv5HjfkWMRSaWsoYx0wcK1Ia96DRzYJgetx
5NSlaxWyxc5qCuMNO5tBy0BiOTzhtIrSDduXCi4rTp3d0cuI/QfBnHbdos8HI2xfz5tdCeM5NJEIfB94
Z6e0y5FU0WM1RRUQHiHsEmYth/Yv/MwsuTVkNw1V0GQkhCiXqv0Cov1/Ip2nzEpPqOBpOx9eGb3AUJSF
NUyGE+x7Nk2rCaj4A9lSP9sCHKpAKGckQ43OynGoWoUtGTZC8awkJd1bvB1aOp6tUUdc0RT+brJygufg
3SzJTpv9lpcbz8Mx0y4pO8xVkyiiFF0WXLVBKhdK1CmSDturLLakJTLHGoDb4RwKIQG3UH+G6CVNZSZT
RuG0GcCHgSHH5G+UjvgQeQFZntPb9vTPTPPGYGhnTK1q0pSUxs8gm1HeG5SiJwF5gYpL9yBxMFHmiKSC
MYoMYgDB8iqxyClaCCATlH9dCRw7/Choj2q/f5/0O47dlN52lv4lN1DgVxv44mWMwKBifXueWiXh2153
ucHnhYOykMusLClGjOOxcrjjcjDDR33DOnV9ndzp5JzCI7incmWHtFPud3s7w4pcr2rQzDnJ52dect9y
jH59mjux4cnld9zdOuPd2sO7pu5m07vK8KN8tq0XA6jw8YqYaLNHSXToHQNnN1jNpNFhY0UYNCrWqVAO
HPtpC/qD9Q6zsjuk4hCkOvwpWDxggnbSgCBjFED2CTr9qko0fi7wXTq5XezDPtBJAe8SsxpZQrWaAyK+
GlSVyr0KcE5xokC2R6BLSRO6AyD0bWkCFoOKam7Y8KbJShSAS4NCaV0JBDQxBQUCMyNKpVE+r+z736dc
zeDLcuoF0XlkOkNyYVlSjineZRzkaaQymMka5FYxydf2NWWJIN8hchbAOsEF2hLwQlt9NObnca2d8ut4
yutvzDeeOER1ibGPpS6dSVMhV4/Dko5WNiSYxLMjy13BI3yTNGVJ3edeNp8I12975XECEjwEIhygjAOw
WIAANoiQpCoyCvKvEAclFMgRyHHBUgRQul7g5OAg97wICYRiCBJJEikJFlQ7QOgIKQhehAegVDoFE5FO
UpdiO06U2aOlDzngCAENJCSQXYeGZEAcoL2iOaQA4MwFTAAsiBhTpQcCicFLKQPjmPe3NtqNtY201c5t
cVeTQUBMaGmwY3Kzm884g51G+UXjlYVzYYXdzmOInXGsqK+vzQ75teXXYGmPzdO4o35jjSA9QIjC5CLd
9m9BV11mrlpFu1AzKEoysSq5ADAIRCJEIhOsXAlPLK31rDNGmCZxunDzkyq3UsQ3ZeHnl1gUSy3T8hyt
1onHTC7FuFXnk497s6tc4Kxttt5kkkE3yjlTqj52TKM1+NPpm88SBXk3zveeb5u9vM8M7XXM555VJ5S8
6QTfLmcxve308vp3tM8L6uhrFGcfDrdOWBd3UaoEoigaAjFagV2o/OcCy4ZvKIspUOpHG5KqqbdSDlUF
Cu1ihvM3YMjUKhTpSR5JsvlZzM2uWHEVvbuqznNU2UMjZq0HcboY25rrau4gRkZThAF32KAEKjp0IGkD
jFHajzyIcoKzKGwAHaJwwKJoNIHBrSinhvzhPLt5AUIJpOFVevAQ0I8U5xTAIpwVew7DCAsEqEVBG4yJ
ml8QUwELg3pYBFMAEUoqn5HaL0qAeQvKnirH5yPmJQhQE0UkEEQRCoFgoIg9WzETFBIhFDRSpSJENJQB
RS8CJtBS5KGihkpHTQgUtLFTEkEzUxFFKAECSjBCwEEjBQTBEhQlLEU1EUhEpBNFJxND/Ah0D75Jupc6
CqMRJrY0RSBUJRrUTU1VBFEEhUdpJmKgoaIhGgJgkm6nbQ4jaT4pThCCdAPRjucwRIjQCkRbxM5y2G97
iKZo1qNDVn7fPu+XwJ4GhJiqYpGhI8Pu936vt+30V449fE7AJSUkuQCiS/Ekkl2t3a8KlUVKz7qcz849
CfAFpVkkycTIGLjCdbQIQXGsx2aZ4MD0QebIoRkoETCCrvBfmRpbewhSEO3VVVowDAMUxVpF2UMAcEUG
XXOaTZ5YarfFla0a0mjoQKKShyqDILoBFMxVR1NWSUTcOF2NYCngBAigAQSAEgYAXTQkQB7jAiKJgpHi
KDJIpBABcYMKIkQgFYorDStVaxEKUnX8flx+f4d3d8Oz8PhqESnF2ZZ6K3YuNkJNClIASCCAShBKAk0E
AtoEodwLggB7rn2euh9f09B7E2NE9ghhFHSw63v+JNrQJullSY3TtMBQT+YFwMEOklEWVI5QgeZGqJIV
AJEBJFQjyG/18KVuJjhDMF0tKmityLMZQuiBoAimyiomPzrdLLa2cqJqAIw26BA3vZvam5HzRlIKa3eR
tZgkqmUBZUQIqqKAdESqEppBVZChZcCropFMDRAneKyDW2lrNPJTTjqMJqnTdxlOoOmFderP9PDyuPgF
BIMH84QQX2ammgAoes+Um+FRGFFlAiQkWQAkRdxwCdxEgSSySLQgMSAzISAAF99ddddNNJScQDESlKUv
EpS11wAJBEcyAOoXkFOQBOIBNIIAHnWonXVRrq/METoQWwCgdlvNklRJPX/eo1EKGAFVER9qppQ50iJp
NL1oVQwgAWFNqYhyEqIlI9SOP4DnDZyC9qmoOQAOMCgQSirCMoMpIIBCEiHakvEKvgMRQLNIGpvujFHz
r7va79CNN2iLe8FQyFDUoClSoJBQaaaacpSlKTiIylKUvEpaS3kAAP5iQxe6dwPl52+fP9nnzjz59X0C
nU2+cjBIS+eg4oBkCRAQ+lJWVIANA9hLhBECO/dvaFX2Yh7Mc7aINUAW4gRRRkDxlVcp74CKZEcDAJ98
qfgBU7TFHRCGMD1OiRn7DXeG+/vBQTegiKljtBB+xlFMBAKFaYAikQmRkQpXhovr+/69NJS8eGhwCGB8
eJSlKUpaaYAEgiHwJHMKF2mZeQF7EDz59j5848+fb9AnXr+nrhyKkECN7VTg2qGJApvMEBz6n0sHrg4j
lcVRDAvW0pQHUj7QYAzJWIHYMb6A4am0YJy60E6y/W4ck7qUaCkoG/XZKXiQJJCYwaGmUTStIPqk6jTJ
KqJlFP1ghkOiAyv2ILn8YYCQiVMCAH+SjCKPF2wDSUC9sJkAncOiDpp7a66ad5NAEECIARBCFQAxmUCZ
ECAxQTyl58+fPn3efq+v5j56IohKaWqaAA9b+ZfwiECSAZPxCDKYApDsjsNnExFMQuPsN43sjHjp8JHc
Bmr2UxEncifap9qfaQUEMEaEE/ND8fGdUlMnx8aPj44Hv4MT6wlIN9cHIKKH74FoMnBolXuAwihz6h+H
W9htQI4vg2O0HXPBoNbGrFPgk+BzBomXV1Q2UK5wA0LmaFRFAkUDJYwCSqkgAlUJAUIAAip58+fPXprr
rq8AjTTTTTTz7vd8YqPx+78iREflHOOX4B2exR8+enz5x58+09vtA9b4PWxKL11lFNCAwjCoykpqUjfz
U+fByHIoAYgQISnzwP4wwUcceOrnH7ufBIQFOglSUBc88888M88+zyCH+c/X1zzz7Z3gEgjj3TiMlKAj
YjRhdE8/O75+dPPz8vRnB6sNgFSjVIQoQEfUDAVCJAFqwCsoa3S92xcAfUGIAOjbrgbrSUEAgUqoYKUQ
AgICM88+vSgEioZcMcccccvRxAIF5AyQhEB6SEhVXmQHv9Po7+/x8fHuHZ1bQA1loAg+ALGyELhwPmKC
PBr6Wg2bIOAdBKAEAEKgQoCUAByxzrtypIDWKQMcccccefTLJ4Kc3Ohqge2llXw8OCXwpEOynG1lM65n
GVJ6VDgAAADOfiKKOfPnx9Pbly5ccOHPnjljjjeQbgQcUO9EBRVAVCiBVKoVxQCZAswOEcMN2D8MMMLt
+W/DDDDAE3Am6221jrbbbbbbbbrrbrriDcbwUIN+yl73xhXZsr2AgDfRCyMYxfCDxtB23xjFznRe8Eh4
D3ui5znOcQC4B1TqnOc5zgS5AAXVVOYxlQLGUMZVU/od7znOd71Jcy8zADv3DGDbBiHzec5xIcjvBcR6
F631119O1Ds7IimCiipg1YU2EFZCh31110pqV4E5ZeenynaPSHziQPfFch3InI4JJjAUv1bK2wPA+g+0
/h/7/Muc2YsaLV93/AHwL5C9wZ/kRQ+SMEcg+joFx/HTQsKG0CQYuEDZtGKQDINQbCuDUQP9fHEQuEdf
BXejBPWeOAualeWD+Y7hUN2CB/BvP6Iiuckkgp2qfNI1H7oKhUV33VT92whEiNmH9cwh5QiSEiv1BBlE
R76bhc8y7kLyIPbgEGIjBCiEomWS7LAJiMcoZBmpkoIn8930IofA5PxsdwDYuDyRUkG4RfFGAe9D7Q4i
hedkkW4ruDnDpkic81QPIX3h0BsUCwvKnFAhAAE06RA1RsL4st0AIYqB69EE8jBA2DcUOlTmGw1TYYg9
QGOsSFdo5K+gD3B6uh6YDISISSSTcjkAin29IvgLcORELGq7g8QEKKG8ULg959gPcAPFGwlGKvLVXwEi
G1Q6y6o8VDl6to9YGoxoaDsV50XkkDSLcDoBQxAD/vEXvLlhA9rVaKHWEOpS8QOCtVB56KnOaoJAUE+Q
eR/mP9pmpiDX6HwOU+QdonpJaHQ/eOlA2ljs7QQp3hR6BfkfJkkkkkoKv6/p+7P9XX6OaxmXn/GFMO6h
AHGkkiqbGm4DIDIS9BWXrMKV/auDYtWrp2ggfWnAfQVP2Rvj9ZFVNTQV9KB9SIeBA9h8amRy87KaI+1s
l0DtUJ6UD1MYQJGEZomkC6lUB/OSppH5o9GJ8x1DPXSB4P94vRpWPshoZJWkpqd7xJxmSOSuhIZXPguR
bLsYta5zhzRq1Q5wcRyCwP7+Cvp0oETgF0EsDsoAilBWLoAGxGqIY5qxBMsEqglVCgZABzZC7UgOYwHy
bagIf+B6M0eWKNwF5kZ9bIPuDziIbkdQ1gCe8x9W4UPcMB7Xp6EdxyDsQOl6RthQTSBIEIkisf2T4hRH
3kRNUhtThB6VS9AG/Y6FDgYnLJA2FVcBBqvzVoBAYiHiKFQcfECOA9S/WbJlin8XPg/Rvf1tmMY2+SlJ
UkklSvKmQoofaBhuFUTwViFNByUDtGyONxzWDoqO6Dwcxum0WUCliUpYnJMLCB0GvSv939C8HcGxEzQT
gL/tQLihZLIhsDsN40NgH7T5gaD4poYyZUxafXzU01QXw6IJkIFDTDVLkwBdaoGOA/tQopqCOb2DvHk2
rgYmoEtuXeG4CIh1cyhqBFdgP/xdyRThQkDnUseE
'''

type2 = '''
QlpoOTFBWSZTWWRB+CcAADBeAEAQQAJ/4D////BgKr2vBQACiD1k03YyDdh9A6+jUAYklUh4+9to0+98
+CltO3KBz7DSdSBqaSkDSitTLF9tvqIlLTbT1js327nO3uL3Hjt3PbOm97wu9Wt6ed72+3e+76tza73m
B6lF7d244DTJACAIEMU9KaTQaYRECRogoBGACU8pST0pPJk1PRiQ0fqEASaRIk01MgJT1BtTxEAhRFIV
PUHqNppP1T08lP0UaBCko0NTaSbSEn6Iyj9U0Fh55CTo8eR888uT34FPXieOhAqdsLBlBrB8cd9VVZoz
ZPExK7u7u7uSSXd3dySS+c5zm7u7znOc5u7u5AJFgB7MkIs0ZBoXMzCBM7gnI7I9JiO7iOxSDSimxyUM
5uogByWgQoaEOxkInZGhDIU7Kq8kEe95onZH0kMgAOyodqRTYoAGlEpmAchUKaBTJKiEQwYVRkCo+9Kw
RyJVQKqFSSJkIpQjQjSqFUihUwj2WgMgANLx48tUDsidIDLJUChBpAxjsK7CA7IVVVQ7KnZQUNlGhdg+
XTFA54wEyEQ7AZIniR7DpKCFIUCUdkyVGJSmtMxmVpAgkoQyaFDICzA5JtSNuDkDEg0hQAeJFdkHWBKV
ChRIkGCApmQQwgQaBKJkwzFTGooChDIFKVWmlOyAbW4OVUtIClIHYHCQ5HJCldlXJXJpaQDJTBA7K0Uq
aTsmSBkLsvjhjMMQM08sXhm7kTMkrRgQZuGlrGsbUZG7sytYSiEmooIlTGpYxiQCSkhYzACxwpSsbTMN
IiIYgzHEkdgfEbKxDQXDDKonTMy2WgekByLcKTu4Ux2xJnWojIKVKhWUFl1w1DZlKgwtkrJqoQx7QLtl
Q1kxjlilaigUFglG5aoxQnGYuWalSpUbZRBZLMtbQrIKU2hqau1ZMSdPh1jthUNYHEDEHpqmI8snpwNN
0jN4qInIcIckyChcVLZUEtl+t4guoupPGY4Sosb3eHPSTtQUvp3C3phu5YtETsrs0B/GZyRpAoW8sEKH
IaUCkXWVERtlYpL5sx6TRKklXBpFhxBQ1kx1zKKGwTUczKWlXCmMqSIg7WDmSLmbQO0gpiwlGI4lRTOV
SawdrrYGMKxYOUqHLQrUKhNSGKirqRakWnYirN3RMjYNgNm3ArYGh00TTIDFm2gidocHMzCYqIZqnfCk
1iqObiMmJFilLTpNZ2zlaDut59dvOBzMmuCGoakU+HMoCmqwKgsG0IXO8N6eZU/+Q1CBqDtgWKgZZClL
Q5EQgZUBQVkhVZeJySk2fEbEE7LknyjItxyUFkVZFiwzaamqxK5hTKFcTlqm+acQ1JUeWmVSVDjKqW4Y
FYnVhYqigr79fTs9vdoL0MvTRTrNzLuVxKhUvlpu3p48njrzoV2Ip2krWVqSxkWVoxGbautRR21pYljq
zm0soxJ0TlaNQETiVdq7LMMIBLs8w7atTNV1gnBN6DuuKLimOMzGiMvMuLomaty+yamCw1gdOKu6GFQK
VszGmNtmd94rw756zjx7Z7bZ7Wp3YVMpzJUHbjq7YN6Mu3benVBUVg/fLnjbUdtcCVGJ4528ufcQ1d5O
8g4yJQFJM0hk0wnSAoctDUDWSiQvdCYwOmc65hIcRZ0hWoUT3TphrJFgagwqBWL1tmJBQU1BZFqClLcE
uNXi5YY6lZMSitGVlVWBYIutQTszSawREy+HibaE1wQY1stocpRBWbTpDBNVTHrPHLdpW4I4yyV+uJdk
4kN0O63dcxOmFQbCJJpy2dCURMmJ0IoVHableslM29ZuJ0NUDhsTDlO8bnkKLyi6xcSv2yhlbSFJlkkF
L8csyKZZcF8bvDeOY1FbUeWnNpbzBtUNdcYuJPdOIbrRYZbL5syuOY1hXo97qGFOZWKKKKnWfNvLssOO
VkCIsRjgraAshghFJW2yBFWCYj3TmbhKgjqVgmJ1TBNpNRgFEIWCzWRWbIYFxIF3LfaVZkSB3061g0km
B60qZg//x1cVGwQ2mgi2FHOVxdEpKFwgFJg8GGjpTFeIE2l5m5u6QIkRNQ0YiQQlSSGyyHvGH1eTWLg3
Rnl2TK2+9vSMG+tQFj4/aEirG2Drxq5XDKz49jTzAEaxIJqo1QJhRoxBS2IRtK7VNC2M3q9Ay0YqcASA
jVaD/gWSGfW3DCTkM5NsfC1rx4uRGQJ4+7to0iCDw/JpmWHusI9LNGdtWPolXhR55XBPL7a9Qdb6537D
kqz7bpfF+RTzwJvB/NjZ/+I/eMt7tmqW5caSWONtuifFq8QaMef9g7rNRqgd3eJgWc1I/uHP/pxhSN+/
XpN5i5RoUaRTaw1OlYy7VqIDUjh1uXnGlp6wd1ob8kh8Zw82AVNym0DmN7SMHJQJokEE4eX6SINf9Pm8
dKj061vec9rfi5YmpcC8iK/x+t8eMUdd1vvO+LwW3iQ8mka1CQXCdHoHAi9bp484mIgkgGqqWciFHcgt
IiTpuUExIHbTY2CtoBE6XadxvphVgKAsFNL2p7gv1b2kQyPsw3pc6jqyeEEQJhXZxoH6QX5HBx9CfR4e
DMyPjWmmleHm/9lYmiHrpB4HkuCNoQd7XV/m++lO/23paQ5dBcpcqFMpj9QO+PxHZQudHvxRPGacT5d6
9dioTTEsSxJLGZcuDqg7lACyyaNAg19pUJv89OKE/CR+7fS38ua0wU3hzz9/a8HHqCD3fnJSSII4K6Ie
5h0R16zQhdfl+ka3Xrxq3+vbbL1Hc+nPj1mfArjt5+q9ovY5d/Hka3h/FekMeh+PIff3Hz2QC5nSECR/
yXhRkSTSJ+dJcLKYxcefRPXNBzHV04fugxJLQIsQJa92x4v0cLx2htcZd7/XOfnPpUQK7lalEdzbqpxR
M3TkAyIOVnVxf10jXFzsymyIpK5l3Qa4ahDilQSKtNIqqb3S5w8ICJPdxu9PaN3Us1D5Mia0jQQ5I0Gf
6xXzk5LzLDIKavS5qYS8nd+crm4mPhlYNrGsO8KYkrWIqqqsizmLgQhZTQSTigKcMzsRXCf1g2PbmJVJ
QbNB2zcZl5mTKXEmMBQFxFArCtZWVgFJEv0h2a2DMwSsqGIwgDZN3K9o0aB5OFyHZaTZoWhYlcgycjJE
oyFbMKFsxDJDPCsrdzMkEcRwFyZIZwJGMTIoqgyUpXKiCoqPnJy2CgrZTIKSmjMxiaiKXKLfIpQnvGc/
cPBo5W/PFMh1JZ21kXjg/cNDcUUcPHRhaxN0UAiRx26MOhGhBxLWDEFo0RZAHGL9RaIwu0tR0GkiiAQo
z9tjg3aqhg+1jO/56u0oU1ZDIJBDSAZZELK2xZWLJrrIa4nbP2Z34oH67qz889XAPLADiTvnMPIgsiiz
6Iazj2+mFdQzv1nhhOOo92TpUgsU53fhTb54U9W+UNtA4h6Z+U11D2QXvzYePV1MYQtvphiagF7sOJiL
Dt8pMhnVnTjBgIH0amKqJhvKu/dgY99TZFDw1WAkgDHpOJkYDaDfbO/godpacyHlCvl8Y8yTGslczMDO
x+nKY2jWv7Uw1LN66WT89dPpCyl88caGOx4fIq6INqubXtz1jsjXDZxskZoChIdFVdg4NePC02OM8w+r
py5Rh9v9tNSVCvNvvlP193uj+Es7ZaUPE8/OarTlAXIELPk6HcO9DfceOSidMxmPE6w2e2RgN4Z5lbAn
ZbZdXthdLdOV11cMRtyoxnOMTiWWZyRzB2saX3z10mWHiq1iMdsfbn99C87zeNeNKfp8exzPKQ8uSqlV
nl1z6U51+sOIm2KetaEU7Q/b0SNa3l6K4NM8qy7KBB48+gDtOZ9T4nInnz1BzwmxZ/GxnN68VUFepBeK
iAWj6+2kzxSuD4TUDVEih4wpMFVlUCSClV6psmGgpt1CK6UN2hhg34k9K2ECP3w3OEVzJ+kjEM4Sa012
czERpIGUxmvcw5dhaCI+DxMJaPPCcHpjXPd5g2nyh5rhiJdkHHuDkNZbxIsdWGNlWQgmo7+dve9O9qsN
ka4CJ2ImmITwaQPaqYRB5+O/UGgSWVRokjHJS1KjMsc1nRjyeD6+edzLs99ULTSpOFHjKuzyP3F5b8a2
JchJkFhw4/7rCFZI71DjN6p1FC/R3onI3u2R51y2yzxaPbLTTmG28ueV6nVsZNas3WKyrNdzSlxKdW6V
ENUy0SivfWzFbePASciDhCLMSokGrScr8jXtK6jVthhs/ED9+7lSlbDOHpJfM7iNkkbC/i6jXm85D8Ga
1S0qAZAZpkaIDPFvvZBwlPXpYzh16dkkd3295UxJLqamRqUXEhWc7TpGP10ttpjJQDbUWY1TZbJgooWm
0QiyNL1+p0Q8BVaX0wJoIfJpdu5VnNrgz6VYRyoXodr+c7fGFbUYEXluMdfGraGUn1OJymoaRppLEiZG
OugoqrI0KJLCpCzCYpDOokzZGlALkDXOkVo+5hvroxzCAJZyGtpqCOI/Fvx85Prm52gGOIcvMnb7B8r5
KsFg1ILHtuDvrEH721RMnO9EkswJbtrpAO6RDlIiBv3wMciLzjmGpVcZiTDn0++GRBV20rJ5zDKkqZkF
WLbaISz3dNsWDTIiQwCIAYbaAwrRXluabkCr1pEG4ocjnGb2SlJwcy0D320tlpSBti96khR4i8e0kJWr
bPOU5qpiChl9oSEEIljigJzg5QLErCiRFIWLPRLzpR845usVnnDKc6pK6lqUKUleqJqZVk46VjiQo4tv
p1Wk8PB6Da4tUXUkWO6aJcj8ICIyIuVOD45hHfQPDKUcul9KZ8O5BBopKrioI47Q6ICazCtEWdRCDbYv
8fCw5ayP5b46sdfcNYWGlS62u40yi96UM65ytU5TlGKoHaRIImTZ2NAjAys4B/bdX2R7FCLXXOaUziCS
DO098EYRAXE1Pr48pxfJwscnhhRUiCjTL8LZqHTdoxxxF33zOKd5YqxEFGeMWyI0iCTs+z5NB7G9uyGQ
EYUPaGpA0lpBFrRbSNdMCGFxksEFBHbtqIl0+s1Vz5czltssZipWF7sxREZDFtg0vSczZhllHLj3huAl
S/FxyVG2yfDXrXGYT7Ojski1fZBg4aSLaDRNRlMay1WtD1vOPVnRlOIY4nSuLhSpN83HiB3liMFHLUWq
0tGlgSKpekGNrzhDDZJBbZaGmmG2HpBAtClvEJ4bfBVZwsI/cW/o730bHPK+NW6iNtGAiFlHOLDi060d
R5l0DEBa2wFEglArY7tQWhhDLaADaQJPe1TBOlphEjrXqXTPI56m7ea5eXop6DDMNZa3aGEtoMbBM1ny
n1OVIRrbce+Ozvzm8+M31nUZGIaCJpBwSLkrFFIwukWBogspPRX9YfBjSGEUaocnFkuzzH121AizoQKG
+fc16lna0/SEOtWRwPEFDmVEhwbBMmiz+EL3BfEA8MhR2M8DAcEGNg/dX4lSk970kNEbyDtMRUU4nOQk
kfZwgude0uVUSzOcjeWaY5gqC7t/GXsWgeUPyeZ8LOLHPiR13j1fyZrS11YcO7U33i+OdzSa0UjkS8sf
V3pKF+N+1z+Q+998aj2u+ByvrcdHItatJik3nmMTd36GE02ZEaYz2yvV8s80J8GNMXBA+Xr1f4PEVhAd
EUTnQYs1ZAwZH9miUPrbKIGMxfZWmMxxxe5PKLrc+3T008FmWdFG5HCnjUQ4NtAnsrS9HsbvQ79esMUW
ddYttxJUohQcpFwZMcaJlq1orFnwh1cynBa4MyXum6Ur0Ax/QGfRpi8QibFKsYCLVHMxciNWjGVQoS+u
k9Rq8efvmn15KhLwPKG3CA0BcEkEq7uSNCx+GcEDfraJAy38snEXFQrCsEQkYRQpEWkCk17lfJdHntZz
YZJ5oc7lvb5uTTi+xp8E6/d/6/1mZmSSTMzMzJJJmZmZkkkzMzMySSYMbzMkkmZmZmSSTMzMzJN3euuu
uuut3d3rpFVVVVVVFVVWqqqKqqqqqqKqqqkUKFVVUZJmSSS8zMzJJJmZmZkkkzMzMybu711FVVVVVViq
qqqqqxqqqqqqmqqqUAAAClBOyPWGICP7xP+cHp37+OPHj43B71AsSBMUhQMSJRQgxCUBStIjMoRJQG0J
kgUUiG8qHEKhqIlQKAShSgQoUCqQqkTeBypVO0KhkLQiG8qZALSKpSMSoLEgtCK0qDMgBSCUNIqFIpQB
Q1SiBQqtIEQClKtUAUUgUAntKg5KgUgjSA9YUDIQpopEGlBKUKF6wJkIlUghSoFIDSAUiBQtJQId5UyU
SkUoASlBPEKmQIBSIUqFAqNKDShQUotYVOSIUkTSlKoopLaX6es1Ajth9EyRUiDD1YVIgKKCCIIoaYqa
IoKaIiqhmIiaKiSpImCaKKKa6QmRREURREUQxTC8YFEWLFEE8tWLExvu5lVGmCMwyCp56nPdD9J/3Kbx
RH3UYO5jOupTEq1LYHfgsBHGdW/xTUmQxrFI5Qoz728Pqnew1MPnMOyLS84KDAz003yyy26jita1rWUp
SlKVZhhoNdWBDf6EwdrYYljJFhC3+czaRExK4kMQrCKgZkP/SmSFNUpQ0NCFDPHAHIXsBkAmyUlhgiUK
UKZI5FIDEgUBQlIORZYjkK3vgZrAMIckFpGgMgFSAVV44mWiilBFFQBZ1aSZcwMSYwWcayKT8N07BETM
axTPbnDyugTMM1ES0NPcsyIkyJo40gUNTUwkdUERRJpixSVQQS9x7tZUx6DL/3nAQZp3TDWFz0V+j4Pv
8fs8OZr9c+iPWOhI5FKPiQB1gAKRBj5YZecpnxjvkmEFArwFe2y0jFQUjsxGfYnlq/HNE+FYYoueYIw1
OH6fInB6SU4i9szCf7rHw9HyTx5vNH/Xg1eNuTlPLxI0fSVkUMEU3H4w582dMD57aVFJOlKeQbANLMlH
dPLQ4UVCfS87zMCBO6nzxFoQKYOczUQrwrNGRLomxQmWG5bwMOfgWEm2ELE95yqPfmKiVNTgcf99x9hh
VVVVVUSKVVVVVVRIpVVVVVVEilVVUkkkkghAwpEP+x9+/fv379/LsI3tWlZMMyNywbvL4h0SkfjADGEc
kDIFoCkUoR/WHkn0lORSiYQpkhSo0qrZgIUOQmQZAfOUeyoclTkp2VyED1lcheE1FbzQ2JkD7NQUeWCJ
UmsMykKxvX5+Oc+fWY9+M9aXNzEGQWIiCGDMbbclMVCxb7O7ZpdSmH52ayoSf6QNTEA8lFsFmFSBEwLM
BuQImXmUgYXk7yQWp8ug1HMO7ep8B+fEfLffbnxd748eWnzX8QFCOpzJxhyTI3JFdQBTHzh6SajUhMHy
1wXLfXdz+upOxtshGeHXlinbKJmH95uwlYXlMtkCuIazAZ+x10RKp9E3V9eBKxPQFOGO+lGn48KSC6+T
1QfOx4zCgrzOEVh9MvGKco5yGPtShvRIe3wfRoX1c+NczN/WajxtZVUvMwMcyzjUXRLFVlakUzKZwSkQ
RJQ+U9hdgD+JU+NymJP7zgwWGsKpy1i1P35+M1/XMyhvN/jd5r3O+HN5nzmPu9+rp49w/LNZf4SmOo4n
Xy/yhntgGTMZ0M1AV25b+PtTwf4+/x5A5SDtVUzMiAGbBh69e9e/vv2/f7Wta1rWc5znOc/0f4KBBCIi
IiIiPfy889TPz9bNxDDo4uGZI5HZV2QeQrkIIHCBHCEDkibG2lEAU1suSbmOW1GW7lQTc3LcbDJ4lHJ7
ORsukL2V2Qf0l2dmK5h9YLDDkUPJDJDPeW63GllSIPRwhfbD2/mU6+799Mj7AdJBHojWTt1MSjUmrJaR
yWnoyVJSyKTthzKFtKyVBtP+ejyqIxJy/o1WbvjWAWca06ie0Oi536+QJEPn+x4/OS9zEr/Yc9/e/nPf
SwRdqVFPxg7Gk5jDIzlNYJkgq5aXDqEu4ZIC1Dc+cm7S/ayX5vWe7z83iMXnrWir5PpcA7Rb8fPHoJ2l
+D4PgwwwwwwwQQQQQ/9l/sIfPnz58+f3+wjSanlXm8lmCxBZtSyJ9syBeIHlVVK0U0Av8GYAUYEWZlde
/H37bRWx0DszUScpOJ1P0j86MQpVKfGYviXYdZo/Ce/3Oe2ZzObp9u3vzw/v4PyL9oUaAA75+f8orI3M
UfGWDs5hq889Is9u7gsNT3/xcywBZUmlRu71o1PHkPwdO2t3aeIuib+MDy+sWAwz2/SjNOW2pZ2lEBFJ
MCBzzGWSckE5FKHVMCphrpmJYVBGTHdp/thZibI1VAn6yRoQENCk519wPP7/rYvkEgB182LI/l9/nvz5
48fz+1rWta1nOc5znP4PgoEEEEiUEEEEIj0M/J7hBgfMRTMw8yjBT6HGdvn02DCI6SAeIH18v20OWymx
7I/EPt7Ye256XMh27YbeE+VFe/6TqnS/XIWNXj2O/e5yChAAA+Xb4MEkkBmWmh0mn+ANzJsSSHOSWxKv
1rubu5+aHfHRPT3O0N8QCOn9Q/f91/cJEoJEoJEoJEobUGY/yMffv379+/39vE1mp5399+jFPxYJ+y21
lEor6T8YWRTWAWEaU1z9enfWoNjbFTi60KfFX0ja3ywzMihKVziqe0r9qkKHtkmSuX2hDDcKTIv3AF+s
ArzngwmcnHjlcqCyLAxEUf5LJ/0cKIzyyeU6fgr8/pPtwIcSAHSz+jqciQMskfSCyjf636vGs+yH5+An
y1JUCt7f49upOWyECzAPKDhvvUM357+Wk3KH3+flPy+3b3+1rWta1nOc5znz27L1KWKWKWKWPTtnyj1c
PUZJx22FaE04i7lfjd7dzG3FekDufsnrfeTsuQH1n19lUE+89X0m7Z8M/ABDBzcFikwZiI2xnDH8IAGQ
yA+DP4J3U3fKHqGMhvpCekEAH+EIs0rLYs5AhRwdQP4hJsDt2J7BIlBBBBBBMggu6/X9evXr1T48pRlW
lumO18QxBBg2KYhvHgP1gA3Vm9+kzfRAgV+ABmHj2zW7+fofkCmyZgGhok3tmYdbh4MDhdd61uEtXwW7
UnBvvYD77d+7+JacfB8rWta1lKUpSlX8HkUQQIIIIJHj3E9TPMfg/PP7BiAfxKC/X4HB2JCGDdfv8ZwP
oj7YD5UEUs0Ei0/Qk+oQ86udhsgieuL0ThrbCb2+m48oU5hnue9VRb+iBB12nUb7/3Inue2Pf6qhy7Xg
kfVeht+37diBimXJIPInUcTsNdMpDLIVvyBwzBmAAZmjvYltgmyBvPEgvmfKXCR5w56ta149XUc7lduW
vtwonU1B5MMsH6p26mz5u8cHkPKoCnU8tXjsccI0d4eJaQD1m78Ig7CJ1JERXgbUYMNqm7BqkAnLKy3A
iHyYLjjj8QkAJ4Phe+z3+t3abaBjMTG+b92/M4Z0oKM89Nnq+y6NtYHm3GhOr2e6qAvbdeyJkZKUU+Y8
zTRx0FyE3zD0ATMBEszGjVmKAZ1szicSwWGd4te43aRAna2gLfcABQzsPR3Dddw376236vZBRZTwmznR
fbknJ3Nsx6PYPaeJNnBO0h7EISQ8E2dzozgdFvt3IEN88C+CRgeCeLw6kA6hXWHfYdR7653AALZ0B11v
CQR6HUec4KqdJAAOh068d6dEReao9wMiXhcABkkEYgUUYYRFIBJUQgZEQkkA/OEDgrZA4CA//F3JFOFC
QZEH4Jw=
'''

elevation = '''
QlpoOTFBWSZTWc1/CV0AHDbYAGAQQAJ/4GCqjqxSVQqFVFQCqpKVBRKkpFLNaNBQSJUKOgA0OgAaJHTt
nz7RH32ZIyraZVs2zZqsYbQNtbbGplpVbWjW0NZUiFVVrU2DKam1poRSptoRaoarbTVmyXD3u9QZCkiF
CL726d3dbuL7GXr3d1zNxPt3T3jckioVFArbfAdrAtYGbBuu4Lu4OywbrB97lwbXcOOvGKCiARWvvBnu
4PEH3u+8G5gbrD7ve8Du7gW+4HesBtHqUUESkWtcD7u949PudwZ7AfOwMXo57uDx7HZ3BtZ1up6QqlFK
FTL3q73t703veHNnrve88+++3ed9OO7mdfefeLHnfd9alLYGiKLZd6+++95fOty977vPvvtz283VvfcW
ec3vvO59znCRSgBVU9W+exObc7nM9vu7177dc+ze77552+5vXEaylBQSi3O6917vebd33em++9u+76z7
Puu53ZZj1KiiSlCLzO93e73nbu73e973vvudn3d33WLe9ZZdcNptmCQqrbPLXuxkb63s773d973nfduv
e3W973hpoBpJIGqfoMSUkgSmmKIptVGGqfhAlJIEmpM1Sim1UYSIyUSZSLxtZDidYMwZmB9A/ZM/1fpn
+6s6ROIOJlRbdKzwpMJoCPMOQHndizoa55R55SzX2NgPVPzMwAZgBe8juhQ/LeMiZqkqMk9g3eVPrNNd
xLC28ExKaHlCmTjBRFPXJ1rleI5wJ08yJg2+qTl1zg86FW26BiLui9jbs2tsWKT7Xyl6QNejDMgRdWX3
ZjDd7HnMQb2VsT1dR4dSItx89R5S6xsJzWgnNqcsy4zGXWOHEu13ezV3ddbtcY6e1SulAo2IO3YNzb4K
cHVLtPYq41M2XWqz0Wbm3F2zqsNins4J7tc/t20ckRzoEz0l3nEWajsrruUXOY1K8J7LsONBq9hsbWHj
pNxFJ1zo0oDNHDXOnywFHH3dJu2Hct9OKKFWUZ0qkNrYuvXx28vi1slqjKrsGLKlIittAlbXH7Zh1Tmq
xLOwisrlAlhQfRdjWzfiG3uSkea++AZH1kbNb3O1i4ibMC5Zu7sYRrKNhBLG1zC1jcbpTLvsksGWtszQ
Cr6uzmnq19zpbNzkIqQfV1qWtudR2VjkvKFXmUZdFWn2yl2zG7ql0eam3QytX0zKfNTscg029o3RqXQR
HBLOoaWl07hwkVKw47w7u1M5B8q1pZZb7IC2lacoJSzxgV9mw442h3LsxUWWftwjcoLlYSc62YnXU+O8
exX2mruLZpHXsad08LsUqElbdMbTtSLVBTuzNxMUQ3BuJby55Qq+ipVsxUjxLrlQoqSXV9DWMYybQ3K0
1G3quKDhcrcZWDda2ISTeK3lmm3YNclhFCpj0N8v97F7wXjGNcqOw+cHr1bBNBfqQQj0SdMvsLDO6t7h
qpmEzdjjhDnKSLVht8OW81WnGflR6aEYnvFbUQSKsKW6q7zDlXYqU9dlXyyuQx27Rct0mHxoaqFZ3W7o
VrcPS7XVTFZqtc4pAlTpIZ3C0gucvZmjZdEzHVX0Mgr7ZEOu48JDMnWjuiGDjpxGGrOQTdU21WkVtKGN
lPLebC8kXF3tYiROt3RVdHfS6nbsW03VwPpONVKNprMV2RlV2i9OKRIq7dyuM1HcxGd05ztBKx8t+z5W
t3S3ebmLV0u6N3dOq2srg6xW+la6t8r7qa5hoR5y10zxUYztrMMMySC2tV5xPdUOcIVy3col3ta8jUOS
uJdTCbyp1iN7szah4w5ebbFj1PyFe7fbWwpeiGlMv3dlYy84885bc0M2LLPXqVTIqVFLlTOVuXBUOBF4
iNqQuTKyl3EXg17mGj1V1IdzpXWcnFJeEXEOq9LrseIvLjre0BdeM3Wzd5HKtWRkQOzZzg3jSdYcVzch
1hYbbJSmZpG7wy+MtQ8SnThN2sm8lmPVazT90fUsWw5WwrdrdZXRNuPOTOEY8GQas+zE31C9F4lcgYtA
9evAnlLZINDuITlq0VVdejpcZbyymm9qq3sNbDsLfDGhTxJSdrW1BXTZW71FKsGO119dPsesLlb75KSC
j3OyXtnOq4JaWbKySXWzkNp8ITXV2a8g25Y666xqrB2hNJvWfqUsVOOXlVeUOiqUuvqpMdUfzYiO0qpm
bfDM7tZlGnj6BWxfSSBcd7rze2VTUlBhbRqseraR6pUWpW9ZN0pA9m5xPyquy7SSQ2OzEdPKlJeusYln
ee5Xzqq48VcetUDuc7ZHXBM7jcDtG7VxCiJmEPduTbxtZhUd0mEOiowKSs2Ygjx0UL2N1QR66mbr9Oxx
k+hzEi3fE65VUvdQyBxWYJfCsKjS09KdZIkeHa7u6nIgzbey3m3MEcS4PbtQLqytVO6rC9o2OfcN2E0+
bkWbb2nGauRlqwtbF9Km883KZU7Ta3t3eJ00avLlG61OnFy2LAt00V0tm+OpVpmHn0T6irzsac3q6VIp
g19i3QblaKswK1NrOQzQnl8G+d38ukMOrKnbVlNQ9t2XvFwatu1KdnYp2qPqiIst2acWpjdBZ3eQzp/Y
LMpE4brvV6thWtaSpq1bDuB14na0Vzw1oovJtUbVOOSrsVp5/Vjqu2jbpvMw9Lw1fJVknS2ApFpDJz6Y
LIeKOvu0L4sLFW3FDQfYxmwGyqE3d2hWGb3UhvMVU/e3sjxRLYfe4kOppvGRertw9laHVFrZjqlWdtrH
qmrbpM6aqbwSiSy1znIZpY468o1dFYd7MLNtKW0QmLcmWoGs6RVbxjFSabU7sm93RQbF0wimGke3qvD1
dTuzDqxlUMrQpazKpY5e25stSmoFD3Z06S51cd40+6Eo6sJIrYM5aHco68kK1bmN7myq+1bYaqX3Zz3B
WiCXLor03fSvW9+u/KmXfqZdS7grI0NG2afpDivkBsF2MVEXLGpGob6W3LGVklSZ1Vmtdf2BK8xLaGxb
KOOtlLt1W6Qwzsj69drLbBBhLLdBs9rBNGswO+aWvJux5ezVta1db1CoHq7NIaN0GtFDZyDudtlzcv5k
lVwhp55uiSUHqdL7s7cbVAsmDxGIvo0WI6H7iHl6rmC7NIfhscrhCovKK+yvGKvZb5ZWerHar1s+AD3Q
tTM7E1g9UpU/BhdDqjoS0J83pUneAEXWPxix0y5bT9mCJnPS6M4AShW0vsrGQ5sBtqb8Adx6zXfYVTK2
+vma3zsRV8xm0Tk9Iw37wAzTnh0ON7vO9iRrpgFnqDiKiJjZ3tDFRMMoSQfJzqdM2Xk0/F8UK0zXFnRW
13E+2x4zwCfiFvvbGevKb4fU5st5GB951dwVpxgM7nvlHOoJvQwl8rulbEYZzWAh+u9yugnqTDZCE7mq
06DbdkaKXRus01bh6SfIwNPVevts29U6qjaveiEwEPQp6mmHOZNixMJ3NhhW3UdiXOEzW9Xyr2BhkG7T
qLa/CqavWAI9ueNMHjNLpWALgrNjpvlonwT3NV5dYh9sGidPLLpj4KMia9ewARU9bz1S9jpi802zVvZ2
7NakxwLUHXGgaEvOBuitSrqxgD2SeiYjBbCno8C8ceewHoGk4lhNCKUXpkRWMBy2ShQA7YzomeWmNYKt
ro9bDYEuHDwzKfHFTT1ZYpto9DCVHtwi2aVb2nVLAO28e8k2JhaMrjlVqABbfm6jHt4JZ3mgB4eOysD3
fZ8PJb6s0JwQw9i+rH5bMKpvy+HsGA+vAyiKVNxYXwwE73rb02r2rZpXhDVMl8kc3pwl9phF25wX1iuF
1lNVXA6erbsM/O2jM6HhUAOqOu3NtiPCAXyGQWT12K252m6iIy9+3Fmfd6L3YgIjsegNpTfxHw8pXvdU
/WNuoPhaVe28UY+B65Cyzdv79588ZmM10EnqODrr1medzN0VusBuDXVd14ArIxzx0bven3wjR96g0BJl
c89lRgBOYwn0ar6zj8AOyLtHKUzrMOdVU4k4saG2IaVknMZuF5JpULYRndDvvF7AGqOe7xnh8YC+RLu+
RFvdsBGdcCtlHS8F+QbaSh/UzaPW/k6S0Nge/Vuoy85mgDNjyfDU1bAap4XC5Hc++HhmZvHqf3yg3XCU
AKSCrbfHV9sPy5mGUDMw7h6pmeb3gcUzgbaF3CWboDqud6F2Gtz248imYR1EJSZkFdjNeWHqQBC6oqHs
ZqQSiVD0GOE3ABgVNnJwtWgQi/Mz2w3+/pTP7Z5vv5rgG8/PA+72pTFbyZhVYuwK8YVd2/X2MB2mscLd
wNOzplmrYztsMHy3k2rYNWxJ6me8fszat8ZkKDsBWWpZhMS6lh3Tb6M7GeaaGnt4YN3GYKHZsf0AN5Bw
hmv6WYpbju77bAW+wwqBMuKpmy8vqYHRsBscV6qGXfLbZoQui8sxSWqwL4h2GVwmuttXatgxuMOQo87y
zZVwIfVIanLs1zKiQ2cYeTXrZuxw3vcJNKOwU6rtVnGbHU+uw19ux1Bq0alTBuWSDLQKhKgOCpzchpfX
UhxfQwFZMBltwBW7dUGfVDDEdxEPFM0q6kCZCEBqvDprZqw1OqnxYRVcjQZ10+oMLwbW0G4gCTfz0Cxe
MZ92JPQAvTuT8VPn4LsAjnIU0wCxGG3Dj7LXQO9l9jYJsdiWBprVgoS6K7txh21piJaVj0uiw2dqpDcD
B5hb0exm2uo9EU3Luh1U/LX3OwNX0TTvsWB5rhSAMkZb5TRUwzS9V2pedC3oPJygLjsO0PvUwkx9VvAV
1AbOQkH2ip7a+AtViUgXk9VMJpQBOJ4Dd84ASw5woA+iAL49DMaDt50YA2psyBnouWGrLlg4fAIAVVwe
1SxmrtFWMsNmSIbLwUpmaG7EM3K+VADZmAO6SYYDbcV3oDP7PSBGjND0KAXpwTY8KgMa29u2apiGErp5
rmwArh6E5OvSCfal9TgM1YgBAi5LhtpgO28eNT92gCXBNuSHyWkT1VDC5zkmIZGnD7LCKLCcuaDqjmdN
s3RLTyL0w6YCvWbJ3uFMIEJM2zqQ0c9Zm89yjdg+QEe71qwJnVjzMmgDh2XxusNU1i33DL1o7uq+eLxK
rFRfM/TF6FHU617ZqEZL3Bd5oMoiiZAC3ugXeMCca8rlrtxGMJDInaxhJtdk6AtUxUIQgBQCus8HtT1V
cA0ZtCaSZCufwa+HjZhMIihL6nA7dYczTA4KmAMWpYbAjkHgUAOSxz6mA8rblTN6INcE/sA2KjdjDLB8
qTIGjdt8n0rGdT23YYYN5QhGdcnGahzeAiuVanlen3z3j7Y4L7houYisqCqVsDaxNjC2BBJO0fGLwcHu
3J8MCTvB9WRAaF1bltYioYVcr1MFVF4EWfdICUQwx+9XLA18N6WG7UxLCrCTyGvt0qmFxqlrEDoYB+tz
fUG2BVh9oMYupbXOKbXYIT0CJ7tgUdYD6Pn9Q/q95MNY4b39zPX4elmnY5eseQdhGRuUAKuheUzSN26p
pHkYYTZ2ZYYWakLhupIS1kDF0Mdh+l4iZs/P2YoBQUfwP5ALN6P1MA5AwFv4enwwg484hl8H3vJwL+/u
wwk3EsO/2h3rYEQCokOMLGnYDxaFSlhhYD0Ox84GW7AI+298rYC52A9Owgsr6ANLcQw8WZmkha41iR6w
RWkvrQHjDLh61jSXNH1oR4zuxCS+tII0g+sX80AeMP5h87DJgvkh/vsJ/kKkKyT/JMSfiQn/JzBX9wdb
hiQzDsdEUQLIEEdzsOpxxYNRDDDPxhrCQxIEWARSSKQIsJBYSLABSQFCRSEFJIsJFIRYQUCKBFkIoSAK
H+++/7Q99x6Q/KdolPQlhz+iXQdbTC/CFImy4GVXWoGZJZqfukKIqu7BHatFM4mYZxMdtHBnSwUOHjkq
bst367trrnmrnMYeIm7xxAXuDxgFe/MWpWr59n7mq5pb8LyqcRcxCqqgfvN+RU9sY+TW95/T2n1RvbFy
n8K2d12HzzOsRuGtPWD0QweAuF5Yyl2uqsAeLD3gQ4FJx1R3oPPYq3HVhoywsUvalk1Yqlr2aLHdRveB
wffepSWNSma0K0X7ZBXCbvsvAVcWmiM83ox9Pjc9FyYr2WJS9zu0BXFGsoAeIGQCfa4ibrEutqrXp6FN
Vefnlh1FdSFiiw3hE5a7W3RaMsMqXpXbLLxjfgJMraeiaRkz4X2+pnm4t0Vcmx/BwLH5A0Dt/Un7KXr+
7OxTEFNp73jkWufAfhTzvei/LOA31pjI1D1TVfAPDjuAlzmlpCyt+azfzr42NXpVWLwt5YpNLw0197fe
Vdk9bmj4Vk3UJle1b8COcgi8sUmZ15W2+R9wUA1rkdysZ9AgJe+upF8lbvhWeGjUDGAPx+F/pV0B56kx
EZdPDL+ee1JfeQXXHPx6Fzvr1iD+8h+cMcu/efPkPG7vNVDquOhczDklVqY1UZ428c2vz1TV2NeMGuD+
nVBxoSTnemwkZWMefJ3LqjRNXCTrJs2d6BYb4gZwzxWXXssTU12HpulE2otr9bjIt6a6tFXFW0cSuKVv
wzheTVWy/ekExT2gD1PrM9oQS3Lr1rzF+1vgsrfdfeNF1M3s9QWWSk6WIZ9zzWYhB3qm9O30DN5DhsaE
97mVmM3i0L2J99g6mq9NDY3hYpdo3VNOTVign6vbBrnnWRBeQ8yuyLF205z5VxPV15ugcFNdTC45DKGK
llOOvho967zmocAtB+VWuC31vee+D82mQ+ozevai3jtEiclj9h3zFx57uxjecNPR9+/bvN5K5k4cPBXs
phJNO6FOJwT7zmVL0VMZe6OjG6KQzJcqu2rpZF0/SS3mcfGul1W7yIpX61us++++TfXhLVaU55bfe9fl
6CdTushZfdFwqc9idvjcz2altZ3xm7r+2lw2YrzQ9wPyrFaeRYgZ2LYDy6e2Orz1CQrb6AbVJQqqlWSa
Pmc1NT2/Z3JKyulAJcR7tgRYXqU9nyfr8w+vTq8MG7LZFNYM6LfciD3Dg549y2bzf3VVZM9oVMO9Dz33
wzs8hUMOKF2TAhvbPecHsmY7nBefTrVX7OFXWX4UE/OrB9pcB4gp9j11ktdXbupOuXOjqeV9mQW/PPDz
9AT1JHuQ6s6skD9MqThmvKvRJvgLqFQg5TvyPIGap6OnpEQ+MB45ddcBAmSTz2J9r+oq/RW3uttFZSjX
i7naKvoJd1eTNLMudiY1X1amq8KS7ILlbgts6tp27jXhivhdWruIVuba7w1LdDEbqWeAuVWXkkHpcj2i
zwg+eX7R3L28kGfTl4bccpReft94Tyw8D4X6t9lVTmJjvWUc1qV3sNI9vMbj4GVleOT2hO15+e56B4fD
y9vZvTBcK4ctdYEXaLrPey6jqbqpzu5GTnIc1dK2ytupGhvEWvbqMGafw46PKG1UR0FH9deq8qgrio50
IJ8qRUmnwmcpvutMY+xUvHj3YPLBYKG3NPdZkrQdh6zQxtSvAza89sYMh6k4/AUFp33gpBHNfhnmPVyG
A661M0E5QVkul4AenhN6fNha4m/T1NC/cvDU1Bj1M6h42163b77kL5HbMW3WtzqrcrZnp8reTWsnqluS
X6zfe2082gsGIqSqEyzDi9n3QMHOk9CfVS3ohSpL92iuGbVyb1YS/MX1bu+z2VlSGF7oyIyBjRLnroYS
wh4ffjN++ziN9b0Jj224r7UUnIT4k6+9WbuxoSQ/Xu5y1svJW0HzuPi72t3naTPXr1WqzV1N1Ws0+nwx
8NEam9FZwMTqFDVVmzvV16H6OrehV2iZEnmeqvSo5HOHSZcxu5dsd2J7hdbu/WL7KCKrdKK7yQ98vITl
4eYc2tw8jXVMnFuZL21phzOQ998m+5kXJnS3hZvB733MucszHOwW17jnK56h7cNO+UPlY9ZNEcKfovC4
fe4VdE3mz2KKbV5ea96d14rg651dsdde9PDEyfKh4RSHbiyo8tAzRfmt1n0S93lda5e1s4eZ9TV3S8mn
x7I0NzsUpatVNTgeMUrzp518LHbp028khPfCxmA1zl0wH07fJV1mLXOfCbKFbt1OTUbaMzS+5XvGIuZx
SLJ86cnl0Cm1qL6rrLuzgXKo6rGVm5W9aEqCj2hN7tf32PvJCrzsXFZ0YrfRCUn5VJ2bPDF1TG6frZ7h
2BJ7e+m8tdT1KFs+dE8eIbvl6RZmyz6n6s3by0Z5YH47MSyLvVJ7IKjs+ehuyxl1b5XvPvX0DjIx+H2s
VyGeeavCoxJu7Rz1nykytNLa7z8rZlYKGCkkKqIaNNmSu2d3triO7OnuDPvVUyHVvgMmc6uOtXXQJdVg
35B3sbJkT3CbB3Aow4/SsNkl1e4/IdnR2R3boda73yeX2tvyz6xNqeZkoe7vKW/e8NHAmMPV28CNsSD2
J3oyq54OZeBBh9fkHC/O022kerV+72UdFQHc0r8mds1Ob0Z5EdfRL1XXz3Ytkr4TVhbAfe2T2w5p368q
NPNVHFa7Z7A5qc17qRxoVPWroVxrEJ6n6IqJrlUKiQ6ok9fX2DIu35TvXvSNvIty+juemxyqtG/Du4re
36vV4JeDyCVe6biXyYmxpahBfZ3I2usjUVb8zPRgJWF6uo5Vq23E79pKrTxSN6+wqgr2iwqbfLElebFf
PcU7TDLwnuxDlGtV0Ysad3nKHRRZYjqCs7Vu1ZjSzX4K+1kNL3EH3ekiVUM2i4cPayDKJpQ9QVVm9m2m
Ghhv025BrKLypw0SQ7QOrq43ej3z8LFb9ir3c1QNlOb21lv3BMtJrV4LKzo3OubzTo5mHi+N6lx3riRg
nqcZczHqp4TKY2fO+h2Ry7Emd1BDaxnnmc+ZpEbdpzU9879ioI+F6vVvrN54HGot40rWYb6LLb147XI5
yjod4Cd6s+1eRPl4P3bhwEtCnuLaVpUunRiSyMNXeO49mz29nU+R7yprQy4q9BSRz7M+8PNygrv2bhqV
RK97X7qu8RwGJvbEfVRWG+rwDuzaeMjfCoRK3DTnNPfLG25ppVJnV1buDbFW3rZFjV6p5a6x+x5fBvVm
3krjnScbyrGs+Z7dZjdYbPkfXSM71Q1gzfcRXyaN1D5vsvw8Id9SpicuaJHnim6VCctSs8PPhtW+i5N0
fLVVhV7ZZBszhNaG3yE02ViBFbCZKo1U3MpcuQeqZQ9nt5qog/J9yF23zrUmqveAH7sgr9orrh7Axbh/
WNJf6bMcRO1LeljKdhyiOvVmk9J12wxs35iW+uYpMrZ83efrTTnCMM85lZWQP56UuF1i6hngrqVS8j/T
9hjzgmu/eQx6dy8fsLDFkhuJdl3NH7Wh7zhE2nV+WrTa3TVv3pWT3BmEelyR1qFjMM8lme3I3cMR5/Pb
LbpXlye1v3qdmr325AOw6MWb7CffC46uacnJCzF4jwRgNX2JJyos2Dym9ad+725Km0o16QdVeWerYLzh
RqHGVST9FGqqyIWXQ9W7VDpRJPTEiBx9eR1zELYQvBPqmHMau96dkt6Ftw7b7hgyIvHiKkJGH8L9PLwX
Ncq86nWlmguw5pYQ6dfxzWxxJFtVt3TY9wfp5v1Ge7x0uVU3LiLZ73hoOUenhiK9ky0Mcc0Afzo1voMB
5sFez8CpCMy7rK/WoeB4RIpIXijEotQre6envzdBnhY3r4hBxM6mzT9d8UtR0vhdy9Gndrbt0O49guPd
hCqS+rGaWYJc6Vmbq7LTfdvHzs3F2zQVu1yCRHd8IvRsdL8lAxbuvEKzt4D4azxRU1+zmfc2zso3tqgj
q63ks+BzB3gvX49m2dboeEoiYI+9SMNKWnNeeOtDJHcDzwzWffDyugT3yh57I/M+oXRuqZLhyhnq8FW3
sWBEJ0X94+FWHnCur3Xd837fekuFVKsIhIRpHb2tkln3stepad7JbPto5C/Vy67w7Zk7JUuMikQHPdvi
b6eHeEJ8/NYjHYvpmzN9XYha30UMDmKT8U8R2UtWvr3vDRh7nudVz0rqW2rgSqHg/uPq9MGH3pcKS8WI
LXjzPvTFDAqG4q6nmlvBvEeETvnjZ2Zf1epxYyvXEvZ53N1UpF6EXNLsyp4ytfVBIx37bxttKfcjMmO4
vcZs4+QnLSyCX556Gli3K26xuW+HqFeFScScDPvTK2Oq1EP2+2Z7w9BtClqlcZ6pHRBXZj16tyShfgMv
hkQMsV151jg8VpvE95JvWhbIylZmKJu+5Nut3bvD44z2+qimvffB+2De1wZmLxQ2r3E64ey5e3nu+Xj6
5E72OkaWw+vwi4Vw9mV1+wZs8U51W+UC1R5VRSpXK8yVYrqsbAim1oOmmak7XmPreCDtphcvamQrV9XS
9GEakF48A/CByrHH2Ve53N641IQV55nXsxUeqmxAtqdJ0tSbu6VdHs4Lgg9wdOl+AH4i7jftv1ZnmmV+
O3cpGxvVGrolEH96UxcSVLob+9o255132V4w37zU044+tih4ZPWZ05TunDiN6rRsNjJRimEGJ+F1k6Lu
ppeYb8QtkcZxVTF22/kFXruaLXvMcYqp9c9bT2IbpYlXXvD3TuJ1r1bhM89ft5++bcoY55go68tSt+r7
GV7FwpGxvmbbx90ue8YJC8PCeOcXJnlqxp6MFW69al+lWNyFpVu0alvkq7KtWRM2lIS31TK22b21Moob
sd1BDOmBWHTDwrsrsdSnYnfUOUebLGrVww53lvOnHPBk77sDbjo25TxemlyyrnTGbSfIwXPLbvMl3uuZ
vEeF3xPufneoCYJfnKonhSY2huUZyrCP3lynecsqt6PKMiTvA/s8YarwliIQhIZXXeCKad5svnfPd6o4
UEs8PbKwl7QYOc7vwZ0VKxCK4W6FtDzpWmLrla2QcjlXmdxrvCXF71nn6hfbJMzzJWjMMfNdVPnN0qmq
FV2dopz0vyrhx3S0L9aRfWFa9qtUTvJKyub+9rUvs98J6bWzT5HdJ2/YvZrfm5zQsvZ27VreXVGQe2+a
HZcqz5Tyrx7N9Lw2Q1ego7fVwijBKHnnZYZut1ifSSqfbL02rKo+myjXlAsG71BzJLOHSuKozNVXbcDV
eLOnH58SmTispcBMLNWXFzZTBcDUxDfvTLq3KldoQNRtUC1v0opRyy7u6UdsVaDeM+5PG6FNImxR41Hm
4dmYsjDXZb9GLmvi6ryg9ScgVu9j7evd+JX1JC44znl6XTa9W5vltxpY9o2Ro0VXdtu5eZ76ugQ3Dwh8
Ojg31EUvDEml1L1VDDp7H68cwbXTR2NlzO3om/D4e1PUZWhX6t8c8b9fe2U/KeN5S8Qq83iaJ6/eeDaR
itheDjP2Cpvg+oix16G/78Hsf7WffV+Xvy3Gs8nQWak8fUEaBmhWmlvC3w8RZxPV1qY079F4Y/Rhq9zv
elUHmPMLXjFmXsSp2OyYm7XMPiz+zk7mZzxPqto7539HjlVP23W8utV210e+FVfutwhfd71vXRMK7mN1
vbcvu1baaz3qniPcyO9buC22HbnitZrOG+5eHbkOw8cG1FHnc3vXWXaJ7H28CLCM86eDqg+tKkdm2+0T
hU9nC0sy6fmd1bmIT5EBJZNzndUW+t+enOwN48d+OVAiukkutN3pb7oeJlJ6U3l8HzJui/1UfWw+76/v
T8869+jK+G3zR1T9lDc10p2yp4TN+49tdzRkSzbJv1Mb4bUrQOUrcXh5jssFLscxM11NkajLGsP1R9IL
fJC/rezwQ+Dnsrezu2vMLuaZ4b1Z3Qjz8SIxTh6Ig7EDjuVHbXxqR5c5nL9cUHGXlTZvMAHxjvq+ia6Q
vPjzvPBx5cX6u9AN71P3V69NWGFTw8+kV+1jfo2hnw3N9GZcG+9YlFWw5PDnnZ9Kun14S72BPc9k7GBv
Gl7zwFJ1VgoX6BPOPO87kflySuxx+eTGf7cIGP+fxJrCGoAKEikBZIKQFCCyAsIKQihBYAshISSB/QZA
JJEjIQIQYkCQhBIhCEgRIwgSEjIkkENJoSE0DAQmhoSRfv+/b/vmf395932waf8ViZsZja/1WXLN4spP
s22FSLfu2l3ds1scSo63hhhm5qX1+AZr104PCando2CCF2ePKxVmOagxW1l0O0ucsGH2kbPjsqgQXFT1
FWxwEQc5Q8lkaCNXOOMtsDYeDIaG2W4JOzvqE22rgtLOq1V3jnzzLS54q3HyW1V9191jWnSJsg5jvUms
uzfdfQvpQZua3m8lJOs4FsNd3dV50pKSrpZkYpJjqy45M1W8nfbWVXY5oVYNU7b2nZ2b3BO9vTotrvi/
d2bcNKhB6vQaFdWED6+zi+0YGr61cy+kCpOBWc5ZlPC+3mFhj4uIIKmhMPDsel9zKbl1t6BjsW1z1XFF
lns7C6guduvcXPHNDbGC5cx8G9WShd0zse2jOricOQ6NNi0LrdpbxN1ddROZ99I2O+Dv4SYXFeOOu0zX
udlTg7FTgsI6VCMv65r5R4s2ZdqzGSEgbN/NbBTHHnl1beHUJusOU3wXZHk6sR5pu2NBUt27tqI5t9W6
OxlzC3MzZWrsrWhaVVtTBKEGuNZyyqxntusLW60Yb0aDlPKeSS600kqEo5ValvZXcOvdI4zHqrXccqdC
MgR4ZS2VwQI2OWBFcWTc7BWUc1GPZlXu3sOadXI4mKrtu9qQK82643qE0QW8ae9c4gsusKZ02jXbJ9eu
2gcqr6j2Oi1zd1FyL7KkzdCqHVm8JrGPbgNuC2+HMLs13TlGjFNyg0rD3qV5VKbXUnSG2VWDqKzErjFb
y6HpJR2p0RkNbaV4NuQiKajZqqm7y7H2u1LM+osZTbtHq5mPKL3hc3s7L6Pj3EaJnAiW9St5OSKG8aby
+tV1WQ2yoOp69+cVN0VND2xJWTrG1O29TRLVtHuq0rvIuB1gvOrUxpQBklDqV9N2bImn92Bx3t5O6KE5
ndRip0Y3uHoP02E92azr94Sa9VM3VXHThOJQE0VBV44czeyUg5YVKixqDtfDT9pqj1h3googg0+mlAOa
+wPpd3AcNkw07DYx7E3Sf1uaOrZONd2HarbCFcqqTLzsL1oYLKie0IB3y3gdT4XOLvLSaNU7QJWr3Q52
hFaZaS0oXBtMaQSJCkY1AtVLVS2CkUkjGiLFmaWNvQuVRBNJAIEJ0pYNOGQsbeEdR4Lt+O1du9U4dMtr
eYU+6PRgPEljRvasvnVZudGph5ckleaTcmPhRLZo9xjRekGahUzpcKGXuVzMulMTFYKmjPs+SE+YbckD
sg4jKQkEpkGxKbsJVGEQyUiI25qejQTSsH5mMmV9mG4TsxyPUNXWUeHR/fQ1tmtOZS3UuOq+f2bsow/V
zkDXFJKK+FnRoJV6M1q8eXzrquOuvGod3VGCQ/ouqEIyg+zASNBOj6DLiZEoQgbGEjHriwiJCEFCEhJk
bTxwTJAbYMFMe4MgbAUW+6VapipZOQUSw0qaubOTvZtaNyZouqGTs7iOh4xlnqFdbGZFWdRhEt6LJIsw
4be3Wl4tUCe9gVVb2tVUBAKHz+YDD+fSTCEyPHlVqIkPBuyGQg3mkeDw7t43oIppuzUXe/Mq/M+H5zX5
84IMpVmzCRjV7hUrZGFtZKF5JRsLFhowSlojybpYFuxNp7Y3uxoreYGod9fLe5jKlnYpdFhBLkFwmYBX
0DIv4amQymFKTJMHVIZG1NyJBsylxypKjHiTGZJjxg4MePJBIUCEiSiHdNnTFkYKxg9q6trUsbQJobtU
nqNaeGAVupx69BV0F10xzV7itS5eKrzSyr44ttjWlD2TioLU1o9QsaRiVYaUdZTGPRtGBy9chEtWGJu1
VNwgUNNIyYa8aQwbE2GyJExLCGMJBSB5M374n4I0DaJWUsyApnGM073F7tGrk5NGCrrh29WLiEI3b0oi
qVq7+cy77HopWRdNmyC4HQLeDBB1qrV7NzO2W0EBweBvxuRQU2PYmHUtraLItjUCGxRRTNmRFsgZdwzN
VbMWBmkUee5iswE2Q7QI1YerVaCOHVYlXVjk87oHGJRsulgeaKVRvdLbD26ujXM511u2W863FYjKSFu+
XAXwiPGUXRtty+EMg6X1PVonolyLHE0geWNENzIrr0GNIkSgIWQIEBgmMo8pEVN61oXRieMGMTweEbeN
rIjPVpYnuJHu8aluMZbgPZLTp4KcEaEBl/NVyri6rgcrlJGcHOZnUF8wWS6hIRvO1ZNnN0yCdAuZAx3a
jhI3MamQ0CBkuWEsJyRxbSvEOKanMUGxkBaLWKLYTNbBL+iYPzBf0EJf02qOEWmMd/ZFDpfbCK6jr7Zd
2vu4b3Pkr3LzYdobGSoDXZOhlnsdss2VTQeDU7Jd5EjNCuRGs4dy4xaRYV8qct/Zkv4v5xr4wkT5x/Jg
OI01YYYibJFNWkmMPKVWuokeqExDSWkB5E5jzAx3Wen+7WN5vChvsxzb+3FSt3o+E+NbeyaeOjsBssKy
LeZrQuPjpx+ud9MzNZXq9FXWD3dsB99qgPnPAwUEsT+0NDW5ga6bTjGN6Qal08dIj9g0T1QHuzexr3q1
ytW6eHRl1g29hcZli09MGN3T6+m9o2Jdx2tpGSGbHMoOhRNVQ3aTgMN7b1bgv7nypunuJu6SznnQ8Ij2
5zFXBZt3zXB0Ls0OnEdZcypXRK1vWdqreOOlly6CXPeQX5hwO9fVdjUK3PUxetei8T0GUwh+9utdXLcV
jfV5FfQevn0sQVSI1ODfuVrsilcLfx2lublkK5GK+m1jcsHFnEU7zOmq+rOjOhkMNLapnkRC1MNjDfFb
OCvawnLZl999WOmIa7WbJrNQEsWWWKQ1/F427l7U15zJBCycfqBm9gluJa+u3S69djVR481pvk8yKgu1
E2geZidQcd4bFFMG1KyhuSxvC3mzsjD26rRN4SSrOmzmswKHCFuCiQnLfTJreHg1vFXVZwwvJScVq8q+
4GRVUtPua2HdV9lO3BJe1Ty0Kck7sI2LhQDMW0q1cOwVmJ5WOmCcm0Je1qVvRMAVKxSN9OBCPcXL4ztj
HYmRQrd4QQjhDIw3LIf2JW9y8nZJcdrRaoVJZqlTyCq676ubiiC2r2Jzm7mWEKWUG0qgmTnjrLd7Z5VY
bzVNmU8kE6R1utzthy271BrrKZ4VXagzt/XalZz40pdDtGtHKfS5gSc7KrTi3u7rq+hGbyjbFvvjdfS3
JLKZ10Z0vBadukyuJeCRNjRXLDwaUwXat8x1IOwbPBB3fCdsFCk0ppLrgrrAuXX3VHecKoXuWhSdquCl
LYabCs3OdzntO8Y7dmLKvYkOtGqhPcQrcbfA9itLram9trso0Oks9Kbau9m9bnPATS3XJVHnU20ZaSo4
QsyVZ3UezJd91x01nG6LMFm87ImDzwQ4jJtazvBjc9+w6XAgkogy0uHLJ+9AlUfRsEUK2QZkVw0z9PY/
vwn3yemsmur/U0nZJZE9LrhT5+V7d/bsKSpraW6ox2Lgmsffr3/0/wgJ98fYuSw+P++h8rFIsUiL0O+u
dyLFUPkMTHIGce85V4obyQWLcXttDFpwqiCiMLmN3Ol9DjUBViks2vsmdQ85NsRVirB7ME5mGemz2rAY
qbTHyebjODjmAsUUUBO64lfRySllVRiLELnZYZSyhWS/nv5Hwrhk+X9/PQ9pz9R7591rsfevaRGJ5fkL
rTZiJkz0U/tPhnrD+v7zpKnWjDJK4VrQK4VU6hUOzMJym0sLTRfzD4tYfsghxh8zi7BDMSSVAvimDQsn
Q6MEGCgN460P++PhBYqxSMp99Cw4nIKjFBOnw8HTeViRZKY8HcB05EVBYsF7J1pS8kFgoVMwqXLCrPvj
5BVIiCkT776OYfQ+n3x9EixGCiqCU+8/T4PHCqosv9j30X0Z8ni2XBL7XvGcnDi+GGRaxOqKofXoYJVK
8fDGYMSvtx76mKnhkjTxGsouhGssQm8Lmiad1K3Ng4qo4Q+A58U7N4SPAp90C1RUYKqjHcr4QTLsLS1B
WIiIiwUES0tlW1VixRSFy80vmWByCCsBbkS0Hg4OYKA+0ocTuVUYrFASnGudPTpDmCgsDVLxXScgsFB5
e44KWosFRESW3uyFD09YsUNvv5znzQwVhFtL8ZSQ/vyWGKh+Wyqsf24nfUKWdDWbxkXXNXDHldFdti1u
CLtkHTqmzZSFK1lXkS5BWog4BmYprU+98ARYHcOD/iyUQHp3TjpysFIK690O9OgxIooisFpzSks/6/KM
FWLLhSl6aIoIizjDPpZzQybOFVVSJvnZnpqisRY+D0PB5iiKKdwccbOVBiyolEzLtsycqov9MISSII4s
DftKSwNsjRPuo+m2tvRokuVQ/a52aZ31pnsxyr6P+kZ8NhSw73inVjEnwNPvi5Z06QnonJS0qvM2i85x
t/i5ObexHs0s4BGYWFtolqIIIILMaLE7mJzE6HMUDXC3QqMpSwUFQdm7P9wcgLIJ99fvsBDiikVFVjLb
GllKrEUUYD3UvhDLOoiLGMRh5S8HG8xUSJvPmwsoaFRYFMZ092CceOFEWOsf77wHJwQ7iDBlsMzHHl9M
ffWeD3MXuW9ZTczy6wQPQ2ww19Z7sK4NCA46zobqz2dyL59ZEYD0y8xI3G9pHM6VX2dMEuB77+hGX/T4
F01FQVj+6GybqIxRYsiftqlBvVBRVgyVHSplCzelFjAR72NlFpsojMyGcWracSoixGIxnHVKIkoWUVSN
TMLzTLYViqslcy+blhTLIqNciQXuyWtibarJLCPiRgMM62IFU7SeUBFfZho057E4CS7rdc+WXuzVjFBk
z9/O/nGed/uQnlS+z0m7fpYLfnuuimdG31GuhqnbHw3L6LUXM3gtttGnT6C6+7bgBvAWefpPg3lQFTTT
Z41UYrIz86uYWlqMVVRSczFHuO9oqgwZy2WyhaKgiIMje6HHTmR5Bm8CdNDlVFYK8L046Z+WUGIjC/cv
2fQ6sUQVYJcpQ7xyIjH8z6DIYyvJYQiDY1lEQlm074r9/Wsv+3PBheo6SUVu+Vdlo4Cw1qdrMqn+9UoI
3PqHUgzk1nXHhYi66ejjSVlJ5l/OfWTcvev8T/f995/If9PcqqKoh+Y0uZlLSwRGCHbNPGqMEFjeOsVM
vrYgisYKMGzt8ZlONVYrFkTjZuNO5QVE6aaGmxWTsZSloWisFLhmVqlLC0RGKMOzLWhQmUBCEISz7vP0
5DfK0zUs16bxP+4jfeMa9Q68OF2q2Zr7uJed57mRQUIU2XGgYr6u3k3fq07Lsg1wgorTeeyjilbwkaix
MjYY43uImE/XlAw7fGz/EVy7bHgr6oLnIe81m9AshY8B2ygBZBLxEFqJdi2NWU6WlRWCwpQsbTqiKKIg
rIy9l173kWCQ6dwULCliIxZEEV3ovScKe04UVFCK49gcHdyjFRF6lssOKsRFBBOlo2y0VRjFRQ2+czMp
ZZSiIAT90VkW8YMIP9Qb+FNtnSTz2ftxk6L84uioG4x+FicmeGCi717eWw5hZsTKsO6vg6P2MHL3Y+2/
BebXyycODiQa3KaBxN76k1qPi/yv81UjUirMSEIWAgvdns6f49KKpE6x6+wtooohc7LZZxViqRtuZlks
OAqqxh2Dx07uERIqyO+ea3f+++UUFiH2H1b7ookYM9z9efU9KMRGCIwR9q0QKYSKIIFNicHV3d+yKFjL
mJqPIvpsGK+no2kyLnfb6/UPUd7M2n8/VgJW+GAl3XV5qyXQ9EG5MbUce18OOBDGTYPrNW+zj3JeWMdK
+fF36vD2t8naHrtHnJ6Tj31d1jBxxc8nYhkar15RT65OyNmuuIxlD315QyqVF7FbD1ZsXRrMvZGeyozY
qZp6TFlMorF19bSBa0HbewWaurBEPWqy5HYhk2HE9eXd3JS5PFb+vC6Sd0FYNtLuKt3raoi+fXu2lM2O
4xeh23mOs2hS1W6KttKklNxk1khbUW0bKYw1fpteWZnOLoERgu60c7zej3vVGamUoasy7sXvTIPMOcR2
3abcsqOz0WqQUxLzIXVbNJljT8xdvr+m2oNx6o1u/FNw5Jq07U6sxyPSX9yjrrxD5/SlNh2tlFgDktAs
QwzNuoqMQqrGwhl2O77exPZz6GYJlZekmwZXRbc5og1eruqZVpOlJuThMQs5oTkqU8vtul1pjpdbFv2j
tZrYLejtlqCk8N8JjZvL6kj0E3iRy6w+Ztmdcpdczll7qac2oFztGZBxHSueKYtHdvbFAxfWWxlDNbpG
mhN5ZVZeW+VzLeBVA0cU4yNIvu2UNvGL62ds3uFvsqmir3Fzs6anVOAirclmB0G5koZbrhg6pe1mYpHs
j4qK+g6qjtHuOvriGPOXKhOn1AvQdkdtOzB2cq4PCjjWZU2M2zJpncSuyCszdQ6WpeinydTqrDwo5lHn
tSzt8NP3Jbk7tHJBqzwulr2dTBcGdzzauXE6o9CC6avJtaXdgX2rRqvWHWOgrNCBUsWS7pRLThKdWQqd
zdUe6hN2zc45baIYVx3mKt0jTfZbkCO0uVK1tXqHM9KbdF6MzIDedfVKZ3sML3mZY4qb003D3Qk9L7+b
I6qZ6iZt06c7vV19dIzmmuwaXmR92M9O1iA7+Omo+mc3tLll+SwhM4dGmjpVqzuf78vVYHqAVmhVBUVC
qFfJv4+na/VMC0jBk20ffpi10m7k40Tp9B6kVe4dgxIMDm2GkWZ+xfreZPH1vss6xkgJnmVRlaZW9oNp
xON54XpvHjf8H0H85/fdZamNMaGDqlOn5DqxWIDXMLlpYeojBEVIzqpe0nQE43jzZKelRRFRDapbxwiK
K9hhncdDeWMF3JMe8J7joUuooiiMEYzseNQ4MBT/NsEURC2fOfFClRUYis0UfNlh2hCLZHqWh3+/tDcx
730cvf6HPVlil8/rqrDd9/LVVbc8vXiyTOXlFXTUgw4vUQluY60VnQTKFkokMJ93IFzEF0sZd125574f
gOf0w+eB4D/quGDabQNNdOOeDpFVE4xt6W0FWS4NW0vWKsFiiW+aVZY3gQVYlnG9BdOioqqjB6b0Yccc
qoqdkuRqy7OgqCk447syGQpaxEGJEWLKg8avGZiaHCxVCz8/f4/P3M/IwCXcwWkICNOR0H3b3Oj1V857
z32P0CZEEyHFVKfPDvPx895OkvT+T9QrRW3WiK2TdeaCFG4o+qLnrnIOvEdN3EGhCZvAV4ZDdFQQRTcT
Nz/gycbyoqj+TM6HHTTURQRSXMwpcsmWWiMEREUTtthZaKIiqMK45gne4NMERFREZu3WcKWesir2Yy+y
ZlBSyhaRWDFOOHjtOgKb7O6eHQWJZOcfyDsiU1IXvLprZsFt9nrn4NnKl5aX49Sn9zDl8Icr1YmpqB5X
o7jUGHB1NarIR2o3ZUSqtTT6U+nVtaEq1JWH8aKzjSraezFwYPJQ6ot/gKH97CL+H+vEREVSXDQ2GGoo
iKrDcO96+LKdVYgqRsu9WGWgoosQUONncccKwYsBt7EtlqiixVQetPUKZUFgjFQbZTcLLdQUYXHpS1Zw
VVElwji9X3BsnPfMMBtnmCx0y2kRJiZCt5LuHrvvZFTPQIJ6z1X38sGvl4Xq32ecfeLFW7ylS62PalNV
zwIeUF02XsomxmHBOODmW124Lr3ce/Jo/f35/sATrspQtFERFRB3hnB2nCRRBLSykspYsUZEpxWrCloK
LGNCtllOJRBRg9bTuijEEEt6soG2iqsxwFttLaoyKizuHg7ydwioqorE7Vh097jjwkVU/pFR+OzPvYOv
24xUeskyaaPPVOPH59x/tDKdcYo0WnFTuEuulKK+b3vuFiO1QnLSqG1eHaq5zItbBOyhWgag6KtIpVOf
BtPoj2wE4cuLsoopextK3ooosSXJxYjj0p6SoxiXMZnToboqCIs29OJZSiLIyXO5Z08ThYjI2+S+TjgO
YjIbgd72PFClKCiIICtpq+NGJkwfO4dyKHMb1fv7zNyB6Htbua3v9+k/KS6HcVoTvZl79iU9/PJaYeMA
3gu6UzyB8h2LYZVTzr2OjMyBbSjqu4VWmUyNClBAREpkKJRlQtHzoxKm3sNvEn8hZ/g5WCkVD8wZZxOO
4EVRRYdfYUtl5QWCLwbQTp05QSJOl5id06EvHk6HeDmKsYO8w6L3ThRYisUZK5ats2HCqiJG48vHcTlR
igiMAFk5TjjjeBjFGRFD/vvv4TcxyTz7MyV8fh+YY+BVUqhLk/UNvw0m2b0xP79k/rKvjV8DY2LjjW7t
1x1Zm70x5cPNiE0CmQrNV10ncO87TzYhF3e4i60F/mA8PWHx9bzWKYMFRjBkXdN8GoxBUUZSwtgacLFE
BGIsTlO3szFQsLZGCiiipOF4nE7uRIoixZ2F8+pZbVijGLFiRlzg7jjYCyMVBZHuOqFLViiBepZelEjE
WCKXDMKWlsKgjGLGP5+fx+juOlnQl2ghSnFJxHvDPueOZekqjY8K671/dXzpSEZaCn5zS1v0bEOL25tq
jypsEQzzvPnf7syLu/x5/FP2YhCECBHGrhcKcCxGCiqJGmvX2UPBaMRSIKkSJTrZm8MYqorGM7Dl1OoU
5iCMiiMzLnVs90sUYiCCxiCiJTpkPQRRKyZs9ebxUEERGMVRJcb3vZWU2WKqRQQNOtL1AWDFEEBD9/jE
+cAz8fQ/a58fuIz6sFZdirX0IbqPn4Z2wOGSpubldV9nXzEaFC7GsK9QffUeeoJLFLY0TD2gE+3CKNCU
q1TU7ERU7uafm8o1tA0FoMYLEEURkYNN5mgakFYIxFB9tcOktLBURYsUEinJmbNHSaRREFjBjGIJvUe9
mFLWIIKLIMGRYxYwfT0PHoLBBAUYICxBe6tC8SgIiKrBgXsTZsgeiCgxRK9fXC0KoxFILcAwu3GXwdwj
BYjDZ+fvDlvViHCvYHwK/V07XJ7Yqp3uKlu0t+L6UYSr91XGm6chFc89U+hnoyeed89Rwcy5X69ue2dL
W6qnZ1VMtlHV+TV1VObMMvr42su9ttRSmleINmTbDugRFmXNYvWqf2Ui1HUQo6+ncJrx2Qw3VQ1xFx/d
g630p6QhDNV7b1cu3b4ysI0udOG18gXy1dkzafDuvtgGi6avd0yXoknV1I3K2tFvHx6xHOGh2TIabMB2
VFzD6t1ZssiDEWDRxR/tcfe94Y8uxq3D3pdvrCjdVPtwswjOJKwtVWEM5pCeZqGpYVB3co2Usd4x1Vhb
U3NDZjYdOjrOWF3YFndq3tGGuvu3iJvVb3bpKjOpRODEPqW9powmoFL7nidtYEasldDezjyzXcoOERLM
dYY4nJZy2TVu1vSniVWdvdamq0tjVPft2j1K9UbTHTn3AnsvaZHcIYZtHej2S1ylcQbeZW25XcOeBXUe
Vx1ykV8YaepyduKGLMrpkMlA59vBpaSDmU3dDYhUjvR3Tne48FPS+qHry6vsu1qWpnlcEN9ev6k8reyL
d3t2qe7NxdadqubTrJNJlXhoKM2MeEXYsIPPuEillp1gClR7Vld10flam32vjdTM6oWCVlTKVaO0jBd/
NdWXOPRdVNbTg6nYupJVPOltzePZwsYxR+znh4QbrXWxRC7LxnbbVsR/akJcN032UcSP10y7wYC507tG
59Tqcxk59tDOxQPVcTd9FQIUB1JUgn2arrEe12pg552L5yZn2Xh27OSzuZnfNCpxDTWDrcZiyydakJqU
J2q9KGbFxD66TeYeHYxc5iRypJm5bM13Hpu9cyQ0u0VlKUL1jHdYtFLbkgtdhcuhaES0vW6VXaLpF3bw
cuvb7cpdcDy7Im4XMp8g9y80rn83kzhpjJ2uKrnztX5ph/UyLKmqqJajRJaHzNCHnsseBTH7Pk/X4et/
BnsZxWh2yCVYszJklukbFPwm0JPG+68zj1lo8U6sP/e8Re35pQ6oevxuVaOSmXvaCTHi6vfAQDn+I8SA
frpFWKRYqslm+y+GkvSqooxBiKOvZHeceOIoA8bUss6gLFEQNwMe8VLKUKxQWQROwHpwcdyDFVjGJzM5
e3Awemw5UUVFIJ7VWbPGioIMQQRQHW09SXpWCrBVI65k0DZ41EaaaYmH4vmaswa+9nq9fmfsNf7rpmeF
eX/Qe7XqyZ/IN/fMei4WvHasBboLzRlVAoK6RpF1eaVUQV8nYMexzXN7NXE2axNL3AyBiQmjBujkHO+Z
pFNYFyKYkMWMFVYxEfpczDp28iKsFERYkQdp3ccMYLFRBYbgZnc5k7TmKxigkvKHcG9BQRBYgyMTMRxY
yhSkTGA2k2DE5TLo1TTkFAWJLl6zKS8pFUWR1lMAs43oCkUFBBjxbl8z28LFiwdnwcTVBI3qaiEsvJpB
m/ty+tVVfUTIJEP423z5vHJiQ+1HLggdV1U51PSqoEUHjH8fFdGZIkgwZHBdkM/XMGCtSMfcre+f2TeR
9OOMRrQxgxpvcnhtxi+q2RZBYIiFhbZUoVIjCMLTczcaaTvRYoxisF7Q0KcHJERZBYIO233PvY28xRGS
KMqMLczKG8QRFIqxJ2RX1mmxSRiMNvGyaikGG7cHgp1IiqybH+P84HJEsiHKjdeDU2Y+gE/csOEOKNl4
nIxt+zIVIkZHuMct2O6j0ix2bRc3unZq5F4hsloQm7FN1U+hVVsvPmq8sBh+mEKX+D/MM/F+D8UL1BYL
EYLILLhMKWUppOVBYCCCsLmJ07p6akWKuzJZpoHmCIKAs9mZo1wLLOljBBAWIdyzp06HCCiMikQ9oJs0
DUUWKbgF9TqAw7YsiyIrJGJt97vJ002KApBZBhZaUOOUEWCwXj+P349oa57zgP7+MNSntZjHIj9VJDoK
ptOv6/7BqwbBK6nAdR3Wr2RRoca3ZorOzTJJww7kWCJ0dy4+3o1CIn3TuF/JlmvjYwlmogEFiQ0FnYf8
wfPhe6cdBQFIsiib3TltAKhFFYcelTGU9bJWRVgKCxApQ5peosiqEREi6Htk2aqKRRRQES40e4e4QFCK
ChGHdSqVZxYAojBBeO5Z04IcgMnSlosylkKxGAsz+MJ9XyIGCV5qL3DQJ/Jbgv+/ruaU+8p5abN528at
cOsbxk7tE61XPFETReRTVTvbW8sYkisnV2Xl2KKIVaLvY0tJBr++5Ch9Pu/gWKKRYDEpcb606QrJEbS2
AKj4uZSrUKsTGkmQ7tzA4OlIsFAUihmYXty+sN6sgKIgKbsmk9DWSDO4YzkV5eOGSMYCNvsctlLDkEYs
AQR7lDl6cwAohmv5rgM4gElaIrYLnTJUMbJj77BgSFjs+G+s+6onXb2yIiFDm8rZx/dY6zCcvNemS6tl
Kxffy7y/fu1LF78fzaQ/PbubWaefF8x+Ne++/Qw85lOL9AbQhjTqKqccHMkUikZXcpR9C0JWIwFIh1ph
QssAogKLFnsxWO8m8bIHIRB7ZN8bEARCLFiN68dyMiyKIkUjOzMDnihWysFhFBYJB2mk4NIchFY3qpZe
gVAHzn8VjEh+bIZIZ7dP39Hp8Xf6GVxW6WdwIaNf38DJRirVKA0YdU+ybWiRuRsQXz57WhHsN/fFRnbX
DJdsawHUDe4gdzMf5h5DQIhmDEsQE/SwLDzCAGoEMWDVbZaFCTlDbcOSzCdQkQuHG8GZ7kBSTnTcr6w2
eCVgsFhBG9czEsvQlQjG2lllsUkE5mVVt1s1CIuztzNodxUJT8jBN9shwew9fGEH+j6CHtcRRiw+at7q
Xys0iOsKpFVfXGqKVRbWdtUqSjXhToo+zYHOhuAQ1qq1c9sz73KbHsql8sOGRaJ39uYpy72tNcOUs/2j
JFih+mQzruZKXipIetoeUyh0klBOnLvoSbB7suTKFMIUQgx6yyhelQBQQeOmm6CkJ7IZl5wqkovSSWO8
w47QA7i0pZZKkJ39iD9PZ5czE/2G6ifr+wQk9y83zjSK3S1evLjX+cIOYEKtFss3QonFkzo/jd1unmGi
uq9GzLJOKrG3UU1VnboomFREUqKLyt+4WMj4gswJYMSIsEn/fuXzZbCHGhbZKUKyKCyJeplKWErAUghV
5udUsklYMSUdKEEVibBtJMHxShVaJJq9RbKUhKgDE2uPHHiE4m52dymsGAdvLfQYmvXg30b0y01URI13
xThvbOb2MzVzt872tyIGMKp1s7X+si7H49a9PQUXjIsVbt5xPRsyJC957gT1Tz5sDutYr4plSDOuaKV8
pkz8R4dax7nHExl24axfv2vuXZ+9KxvLNe3IVeqGq8/Q2q0mMOtryL9nrEcy4O7VZyCDuPY9RrbxDhEM
qB9j0WRL3mZ0qjpEfNrOoVgo7jtS1u7K5PNt2E9TN6psWB0h13d7SdUxWWKU6wWV2WesnHnTTspVUmbd
SoKGQvXjYMkmwzVlFC84K3uEk4DOMpq893szw57q8N6tzEb0iIbtnweVcTd01iVym6JeyzQvmNwjmMwX
HCdp7Hh7dY6bnKsW5EEbVZQmQEbRNLOex68rizNq+lbS7ug2bTBvMzWZFO5luoYhbnc5J3CsgMA3eRyq
unim7qCm58rTnLhrKx3CramX+1OH96g6/emzPfqDR99Wo1mZeUVskZ1UhJmdvPKd1fLIju1YgMOOuuqg
epzCsfXVsG4rm0XtyE0LRQzu1zBy6qtvhm7MlhbGeXAzNaQrN6+6JY0JZrCODK2h1xRasdI7Dh3TaXYO
2BXYfJsdpfKtUkoZypYdroMkfPE63EO75aeevu6Ys7J2qFPds5kgmHJyhDT0G5Aqsrre1bLDOqYqrHQd
cFYsmy8jPa6HZyq1j7W1lyNbtbFLVPsRvejwdvA1ucRhqq3U87WdM3ljJ5VNRfDrmW421stZUvmhL198
Cw4Zgl50VIZhrrPdw6rqCxw7otG0lOcy5VcO22OaEMmWK6+sHcDDt31ayk+mETIF1dqgqHcXOS8pzBYy
TJD03BRf17iq+53SkUQgPdjR7dwRXyBGUOlHs7ummA0cfVqnCWEXsG7j7SNiD1NngpkrOmK65mws589a
zqZlUs2NTJkw5Q2k5q2qtSpu12rR/LRkbfmvLj4YojLz2pelLBu90TXUf6htqeffZp9jdTM2ZjLkzi/E
lh4F4YkXOO49DiCNB93OcjM9z82JR4vMobXXV85E167rIkKr6qYrMKxRXsTZ+k1RncLZ89q6U3Qn6TqI
qx2VKx96mvX5sn7zsD7F31joM4yoA4WVo9pJNJpteG2ElI73F9kShwAU6hbUAUbGrUqrAAq06h3ckjnC
Pse9hwUv30lQBSRE+oNPSagGOTQJFNjWAGhV9hpi9FjIPndox956W02VVBG26jGUz+DCG5lUH9/Ll71H
cSqg7KomV66qldWudFZvslc/nhKDu5b+N4i83kIp1U04lVJyBYUrR83yoMihooBgLXXmjgqQFHLauTdE
LkOXcqIIDwNPQP3wQNiW0HEKD9wSVIY4J1R9nNMOKJLhXaqLCpCodSgVUAHQziV0UQB349ab/nEpGes7
DPQ+iBL/W/nFWfqVe1a9itmnU9VCxOhuAzRqPn7ve90Rrwns4i6kko29sRdeE/NYNm8i8VLKrQNtGYEV
tfypD7qDAskCx++AFCynWMtkAsKXjTWADrJm05aaAHRtoWXU4CFnBvTuJI6dPU8HcOhOQghx1eBtJKgC
wE9Nm7IGpGN6tKdIFYT+D18CfO/OfzzM/QIOO6cv54+jNsd8u86WP7z3r04caIFN42KnJrMqSapRCiCb
VVDFHTZ2Za6O7mX0F2RvyoKcjuvo08GHhR7+AdPACSElD/BIWDeKUMvEkpKWVZ3EnIEYXlnTocScwBm2
5QpSAWnTgtACiZ1B0sQgyrOvK9QS4GOW5FSd0knFXGcpw9NCT0f6ezkMcmEMTbORYOmz996rnT90dsuS
5o2hzwKiKCGagNXmxO20TWLi+srDVjedLxm1d9WWik6yqOkremzs+0kPim7NthLNIWbw9lve5+U+HwM8
n0SbAatIj4qgFQfZTNKYNurRJVcVVDiSDFFFOHucFgQ89OWsE4U0CdOyGFtUsOgBxNcel2WFLIQsi3vc
BxMIBxFOIuUgYH1D+Yfz9EeZ2+M/TN4zwR45rZD36ZdxVVt4wEte/C5KhiL2XwI7M1NAnEQeyGStwPYu
V7HX0OWTl39Hd26gl8aTm+MiPR9+p/Jfw+hvfwZhMXIlSkySccve9Txb0JDhbbg42rLJJQ0tRSoFR9Xz
ekYgGnsjzCVVJBgntzOQ7okqN7WEIJKCfUq5ESF++cv7WKIDVUFVHH34nuyUKs7+nBNR0LIeGvjsH9TP
33ZwKQ6vKYV7Usy4SVKlihT6oIu0bR8Qdvd1cPXvorygT9yyRU5MQQwXQRqxbh29oaXIF3H6p/Qgfhw/
FD6bJJ0d4U/bZxIFHtRhp6STTitlCyyAalMXlJ4gHh9PHBy6BOvcqIdOJDhndT3vSrPQ5IS4ShzGgsVB
ViTXVOqqglv79b+nfGvHretkfzHHa2fxyoyr/tqKkJ6x03Fyt1V/lvm9RrkOqvacfMZxlS8GzLcLoEPB
mcGeNS2DXz4VxzqrmddGGkkNHEtn7pazpv5ffmcYz56qCVRuOR8RICobfcqcHFELlpGlCAlaUKnARopJ
rWNlnSE2NOaZw0y0hEZeG+fShTYQOO6opUIrSYTigREaSPnl/I8ZUMigCI54P1CcDjgg8Go9S48ySt8I
ih3PFTzO3+dOzFfDzodQN/oc6SrX711lAwaW18bM7e7H0I3mXq2jnAmskJGGD+z1M99/Hp6H55xGGJIx
UpUioWo/qYcQB9gcFp0gaC88co8e6CkkT406dOADhtHKWX0JKdS1zClCsIIN61SnQCskYty9QYWaQlIn
TsuWZmVZnJ9qSGI8fiPOa3urdH+kYvf78pLa0ZBY/ve0nnWdS93M9bMoUwsDFfX1l1Z1vN2rLQlXEq9K
xYUK62bl2dCRC40M1cubU+gsArtowwbxPCX2depz+/kVxF+1JTlhxw+zogVGWKo0EQOpSrs5JHF7kr3C
A5aSsLUgrE4TjuIToUvEslpIVkuBZnm0KTYAC07ZWUCghoHwBDKruyaE4EVqrc+9hej/LIJqB1Om+72/
3o5EOizOEnv3v0YduSTF6/tHZHXmcbMY63vTcRkee3forXHd3WSnLD64MV4qBirKS6syd9eOlaGduTs7
r0iypxZu9d48EWrRzmko5uTszbxKmYNO7eKuvTUijdyzFM7HInsdjWcp7tc3kwmYeF252Lnk1tp0awG7
j1xKAk41A3Ru4YejM19a3K3M7MO5mwcOCXSHou0134uzfJ3Ve91QN+ZEiqseVx08r6hr+4KXcZqmIe7H
iGSZooQwgyV3HFNmu2QWDhT6zKN0dB3qpSuZG4mVqs3Zl5kTpQI59KvCxQlG6x1KrmuIl1pjQ1MyrUAZ
vbttBiGY6Y1bjynJx8g6MJehR2Frypzt0foOsdxFXnDVcCJwti8D5STNF1NcZpSV5u8BQnLLHaveeUbq
G2CxQyIoXfHeuI3iayZNlvbE4YNMZm1ncWlV3lU6pzlxsWctvuNiaol1c+zuGSm2OFVgmLqGW2lZG12m
p9Wl2kFsvLPHYRJ1Vx3dzb42Llt3XcTDcnPmpSvgw74u6Ub11WNjHeQ9WDdS0WhO13HWRXtCAhGdmFEu
XNj0GrG9omDdRza1Unvdm4aNxKk3u9oi7Bl72x6Vw6Tq1tk/IMkF2VOLt79HFpxVoSuCMU7jukRAqG9R
7O60yupO49ObXYT3fLJe5w1iVdOt0yHKrqhrWEG82CI8ouVVa4cJVOuyRR1eWVmKgtPdz7sgV8LFXzL1
SS1vNDOreufPnpBscVlR9Ly9x0tMrrNbm1VRRoZwayCsqzVIjtRrlci5EXycobddKG7VGn86kHSXSk6i
koCUL0FPs20jlX0oHmgeEHLKpFI5ZCpMVWLKyLK7u2ZbJZyLmIpz3VjkOxRurelULvJaFKpUMH4GS3te
YZ4JMgsSb5/O+uXF178K4cMVuuHTqpyucJO8KOx4qlrKowYKH7FsIWvUNi6kDZb2gze3iJ3nVejml0DO
HfnHvGakd3eQOIka2CWDNQBokcdJ2kJ0/LQoWWBCy9UsllAhVHEojtSSgYTmVUEijsd3ouWaAFCj3LO4
DUkE7NgyHmQQ8uNvFpgHwHdyd/I8b/ejF1kX3w+XeHm85xbolUd/nr77rOS2uXBbp+F8Wue0W6n1S6pC
HsQXIquHW2fu0tas1hEjgXj2dhW/vpkHdSjG0Id9bOFVUkFqzjuDlpIVgoMF2tDugcyRgnO0Oh6EnXu7
pJOgtLjbJaEmvRmcVC6QVTInaFFyIkLjsMwx9wtBRoP0jh6bKAkVLPNEgPD7dL3nwYIKXr+zw4zU8hj2
0CTk8/i+zFxfqd47tPOcO3upFVU7/Rikqp96r81CanlrMealRz1rMzF12dNOJbSPS+qj2Suivd2hEhk0
szUWYEk1d8iICtDC1u1jRQFWMHS0pAKwBOnd2ZMG0JUDpvQ40A5IsbWlkpQhecywoWbAO7TTTUgsEhIi
JBGEPyxjSyNHvzz2/L/EuGvfOz7iOzG70X+uX7PSyRqH89xdjSMzFt6FkqckmpnW0aotEkg9JZwknATv
VjVbOXtHPdp4J6p76uqTnPj8isOYQzCYnmM/lokX6WhAU6S4YdwhVMvcHC4qCUTetlOgVJEOeC+wyUsL
IFgmw3CB5CI8XsTgvQhXuDroE+Y+YPJRcNmO5sR5zfB52z88WLgTsDvjd5bPaF9tbL3Lw29/k7/XsLpK
EjOV5gc25cRZOYYc9Xdthntotb1BJHpJJoOtpkHK+d82DvLXPgheudIQCEhmtrpqaiaDHIRLUstLAL2d
Ona6EOD3XMKeZZYSUN71wOfAHRMh4Mkh6Do3lutmgFOy9uPiIRU9cpcHqS421/npxEKu25y8YjWGZHw+
OYZHBdjpdJuIjhvhu/wGvYLmY1hFXl68amui/Ia32QqibK+Yrq40XXzFHAiCw9sNHpRFTLB1fc79/LM9
xHjz6fahc9l7Q4oIzupzY4ilAulKF4k4XZ3Q4KWQEpKcpSgoQiIi0QTnU6iJLmbQi1ETodqymyfwoqcf
R7FOr+buZlofv/Y/pNhup/Y/hJJCVB6dmcikMO5RPZYHUaWCtD2xyN0F/Vs7m2mUyxuH52rtGhpf9vmr
zfzdX5s52Ovet5Q6GH6R53qe4guwPUy+tWF6EbM0qWoy9AKHGJx04NAOicNyh6y5IFnZTChZX2Z0Jxp7
0krt4TpV6T1y5mU0MoXoFTXxiCqPmuVQSjSb/mn80X+XZ52rmcYf3epiMdYu6Wy9OFkVaFRSUso/16CG
J2VyEGOTZye2QTnZd6elBfTLwV1yfD57v48793zEFC2VH8HAdCulKIody3jgwFSOw6ihCnC458LgpWyK
rOA4G9OPQ0TTZTsg06SWHXyWUG6Bv9jDMPMwrEauRuP/b7M/7L7c+xpKXMU/fj/VlB56lsGdbyq3aHI3
H3Mb02tUocqOlrf5q9I3qJBNG6u2e0vT+3Vb1E8fPqOcIN5PgPLGTDwlmmhzgwXsXdIziY09QanQji0O
pVjJ1S2YYoTRUtKEQQZJBs1BAzNSwwWtQigD+HT+/IQNL5IDu90HIkpOH8lRtDyRGJ+/rYh5Yxjyaohu
6gumpybq6S07p492k3zq06KSBzVOq0ca3Ns6GZTwxAnfdWnpqunq8N+5zH7qDT6he8ODVAuXaDOoiqyj
REo32QjaxEHHadqqkbB9AIHZkqpEQsIukJzU5KT4wt/rUKl9mL3v9u/yOqtL3vrXl3o1F+BiUnU0kBtC
Kjm04O4X4V868ICfr382OM8XhkDvLILS/EiNm8ul4dXNK2JLTEFLam7VrM4Ped9kkWxWT2ZndBvdZPb0
fbaBvOWQZQwPJm35+Va+oj3gbXsjcHnKvOwyURmkbwbuYEu6+acq101LKUL3ijM5ora+ScOF6q7c65d5
3UhYo1p6sTJjxZ1cZOrWKuyFlVvUiFLpidm8sR2rlRPKkx73CE3zV5CLrTjt4/T3vah5GdY837s6tWX9
KIeSnfqt6CjdMENp24T76p3TIey03NcJ5bpKW7ubhuqJNauriG60Y6hT14hOMGXcpXULWbK69bjCDT2s
Zd0kq6dVXj375SfbGwDZtvX3zPwb5/B4r17rrbW90rnN3FLQovUdIe9t1RopYo+N2iE1LZlAu3h7D14W
eLKdLS2s1oiVQKVwImUaeh7yO1VXeXVb03ejWGumI7YQnY3orOe4MPO0UFkC3BV1NegglDr2q17sroNT
3OS5VkvdoNZnMxnELKNNCYLkZya6y0L25jvtfb9ekHctVCzYl4Q9fBQTb7KhRrKvg19fMvpe/WH2MWL0
chvG81npd6nYzY9ePJu5qK4UclbhvF2J3ouzS50+5ZHuun180FMsI7XGJ/Oa23eTGe2voswHqqkh19Nx
A3tjdt4i2hzia3mLrcfbMqqvAbwLU+C6+p9Yp72rqhlIsF2zeJ2KG7RCtSoOHYzVXJmK6xKDvps25imU
p3BoyZRRz6HcNPn16VeDMUXd9taeU0YjQiFZkjjlYs4Gcz2dN5OSqdW9E7qt39qVYhhvuqqKKHblLbEd
dzPZKvBFBo4tGLpfJJPtyaaaqcXGrwQxMtfUlfYXNdPc+s3K3r7qeVMi1cct4bqDcWrRIlSiu6I56345
aUp2d3niRsj2ZYPm3dZwzeygkamXdfUBKo0dhI8a3TRmtuXy6H5gR8Jh7CJLNA0RU756Kt4mQGmNHCfx
DDXn8Q0P6LRfqbvpUDTFaHK7JcR4zDz6w86KLokHc/N13WpsO2MHPFIFHtEjNL/e3j9BT7ZO/bmFC0LN
tzeN1qdgaHWXsotlChodTC5dFaqJacndy9HS88zlwV5hnVc7BlKpSUboW/37WG0URvt1oIzHVbPY8UX5
7rH9tfUtlfvWNGZR9+gcEJQtDy3Q3yyyaBKO+pDLfHcXMdc7XMzZ3ctuUUXTYbq7WiAmh9+0FxQWrfxT
CtooorCKN+Sj6UGt1vhtyD5bi1beIEUNRXxW1VTp2VVZfqLGA4djY+OAnu3KUnLsj1rUWPpW+f2U5rGm
WCefvo+2nmbRh4kLDiox1pLc25PD3pwWkJW7aNfGtw484JxnurOO0XlYu4+r34fwfFv2mF11ePkYdxFA
hOZR3jWzT2h7wb4RPF0ttW2WltLKOldU9N0NOOrjChT+qNP27ocnqme+x5Hu0h/rtzmEm7ivhtCrCnFU
oWI9M1Ua+eugzfu9eAuispvse5y5cNEieNcexGsG3172477+6ho0QCSH4gMbEoBPqDnwrG6iq0JCEkhG
DURFVAikKR/bu7mqlkO5YcLjjpWqr7v71p+B7q2PH7su0u0kb8QjNnCnjMf0+P2vRdz0TBPaKGErIT25
rLjlayuhRLynEwygbhPBGkyjUJZzTmPqkyw1iByxjd3fyUV/ML+G/oM+v03Q9LznHBwuLw3zucTVTJCd
VwxoWN5cZEAzk1TKayEaChduLGIaqoUBQjx+HYICIBDp415dprXL/fbmW7/jFv6eF+NegS++nR8qfWFa
eDJg16qVCOb7qF9asMYFS711rNI02tZzKkkExlVdL1dm8p7+GevWCGEMOfI6SBuDUkPkFx1btpaodUrS
wtMy6NuhTrU3YGQzj03Vnp63cZTj+nzWffazc1yfV8KJ/snes/wlhkGK/4hbnEsEZTsSM2nszIW+wraT
hQofpVYtLnxugONPp8nwsGpdyyp9VqlikMZtnNIaO1XqOe/gQ/sI4xnU9iCMHCdxSlULaXpJpiwihIQd
3QocQlbKVVcrGXKqSNxNnPsFysVFn5to4LFfny/33Vdnfnq3exPf282ndw44JeUCclwakx1hPgeaEgPU
J9uip3RWds9czhQkdxnTVsXlWmN/vv5+jfwNMe4DNyEtlmGB/Hd3cu5crzKFsvS3OMKFl06d2IdOOoXm
uFClkhCEyEMgvmsO5qWpPrL1uJc6taHnPGpsTf0Niz178Zx+uE/F4url1jvGWdwXb3PcMFl8/QduP7la
7fM5GV4UlpjwMcqpxaZN7ay+ndTLrqaj9+aN3JPp8/t+XsULZaU7jbcKOheCoeZwynFK7rpVprDu6HGj
4ODLTEARy7NPPOn2d42jVEzdRvYkw3myzmsuevn7vyrD+EmVjf11+/XmBOHTrXa0ZuoUx33G3t1MWKhw
e10dTnvLKAfCiT9W7Z1CA4x/Bt0OOX6KTNwg2GWlqq9Cy1jRVepaFV4IWmOlLzoFE6tWLNci62ptgWZJ
6pJ5O7V/pTGEiqTWW7zA3GMwRjv++nBZqJCD/DnxoOH5lUQg9S49bzWkmhxEEpJhoZfD8v15Ly5fJTM7
W3AdlO5IhsCNzd23vVUGLWLNaI8eoo8mjslbZcrVha0VBhCEwzem5l9fZr0yt5vdy+k5xZtx6dqpbO5J
KusZytKpTlp25WGpaLeGOb1msfQUUnRFulaWzNpV27tHe6B1ebvHMwW4IuoNqwU/lMzBskcancLmaum0
HNVmmluIqBt4O2r09eoLFb0a88x2x5XY9l0jZnqVHXn0JZhovFvSyS822eT4IX+27OcfQbooSKKi13eH
CvaLml+qvR69v25NxqZl8tzCqffSG/poY2dy+VZsvKm6I5tZvdzvG+PK+bfXndnFsYU7x0l2dlEKtLE4
msfLMGLFqo0by8lLYOjR2WNb+1dd6D17exqPejq6duzFy3u5705co+tqbg3JvEXXSpR4mYqpqazUtNaM
Ze5m0ar54evO7F+9cnt1NjfdQlqM7A8zMd0HFrXoYjWZa4ZW6FZ7dLizL2tl718JBUWW86DeLO4N+2IR
vEuvCd3tIemTIVHtixBtUCFt4TmkLIKpEyZKdWtnGKjYvKqY6tZDNVQrdmXbK1sTqWdVtrn2a6lEXcnM
dwqK7vlOBV8h1BCq23lIZR507aoV0TmTHMjN9GZVY4XKW6k83mldbVWdzal79uab6aL6aKyZt4nhOzrj
i2bjcrt68F9ci400EGI9yx1psDgzb5hwbNx4Y9jNOXMreYqCsS5o1mAFTS83iNXPrWcqtHkdtCZppUXj
tRCzW1L5V2joxT+rYqOZMWc2eT4ixzp5u3l2W6CDriLVLmqe30XTQ9m8YJda6tVwlMNRJYJlWdpm1lM2
7d3VSHV3VmtQMLaZpO7c0XxjPbfOhreN9OrJnS1s28UW1lqqNDqrZNVzHQ3Jod9Nhd9falZdhSA7l7S/
z/v8wy/if3vESgywy/T812P8NoSsIp/3hvtEsiX/KUdz8vQ7RYcLWb3LJfuHsq9SS0ZeeE3cSsvit5kY
Lu+M7J3YunzaMWz/c8DxDuPNzciUrdiyETxRstOvDOODi2tWrtJCu8cR9lRQi2EINznIE6SXMVA/5EB4
iYSKB3bgONp+hdqmm/xn7BtKf2y+y2Zu25dzKROzaKVdVfQ08TT4iaGWSmSST2ciHiHLSO6Sxk61OKgg
zjlfp7TRd+z48yZ0n6DiF16DMtgoQLKW7Qk2BB1VEWqLTLIHV6pVeYlyFErBKfBO8TGhlU/Oze7v9+3j
e396+NeKLiB57nio+tZmB3YW8zkWJaKIXqUjEn0res8tabhdvOOabkJDIsiqud1Ta1ZK+wPpYMyrLgTW
fw8OBDhAX2bv2izT6tlqq7Tju5jDu4hIrom9Wq7L2h3PMDqadSE3nFyyDXERO9PyjT9veQErDAo6z9X7
87Lyvf366u958SywN1qbeErRZ/XwSllqr7dunOlvVGcgRF0JqGkmtJN0VT0ESCVcJ8QRsbQKp23bkQZh
730/ZW2dh4m4SIIiLa5IUihCB/gtLVOOm7N5MoWFnUtKWb7V2GrUFSCSxaDu/GV/CxsbFHRQJG77M7BN
sdShX6IHDB2WKFH91VKqQCeP9kox3Sw0fx8luPEJM2/0mOnxvYLOGDtswkCy0HVaL55fE8paREqSL+EK
b8NwgjBm/2ktN32u0OntF0250HU022lxls06wFyE5eZecxZKIErh9F+v5/1Iiet/vIRR0u/3fb48eq5h
58Tvr6AK0aRM3cB5p4eftBkmXXyIQy/l8l0vFKnb9dvembbDw+rnh05PJSPxNs94c+/jhnB/LyhKivHh
6ftJukvHHFNpcMyFl23qMLhihi2XLVV3b156cHZwpicMf5BX54sJuFpVco/JvVLylsGGve/Be5/y/Oqv
eqC1Xz/bNe2Fo3kiRSg7BCTcw5rhW3x2DdnpdqViM2CoL36zaYXq9SzK2N6sXvws89w8IX+bqpSldRVQ
ummk6JwlasqXcZabr1vZZXe63nxvSIiNbF1rTp+1x0tGTBh40XdaLP3hkcYxNnBO4341siT2auI4cKTE
Y2oaKs266YdOcWntXojTwbd/E+3b654F8w+8KNbT3XAk/FPZ8eHupPzzfOPZfucCDw65NBYQWiA0HJP6
RRQ+s7Dd2bp0Opea5b0wihd1bmUh8B+XUhDO7poudaB8/WeHL5ennDfZAgPUyc98LoR8fUjH17T3pusc
1MhxdF0WCocUwFeYWaMxXDzk2u3NVddR4NduFpF9hRpLJwUo+4dujRUFfT4Rigw/hZUFqv7qVtUsXUoU
sjiiIVXYNQlVqEUckWqC+3upIkWeKi878ufpxQ5lSfhzdamajhP9SDnqW3TXtrxLvT15gMqxr0M51fOG
q7dtYR1GClRRLLLNdq72u3vnhw3GeXPDriz+VKJN7AexBpHIKK38moiEiirLYcUsR21xlw4fLuXEdbpa
pRjpWOnk+T5v3/efV8zpPH1vw53rEJgu/cvNXv86cLlAfqWAIj93j9vh79+c2Ts/P1D3ksy+l8LyRG0n
mFW0Ti2/zlNfv0uvK3m+8Iva1qrfysrxDYh20/LKqzitxsa6it1ZXdgiu8sNzU/k63DOu6nO9Ut1SaN3
CR0uGdtLQ+Nsg7bkLnHOrnTbiexyTpmje5awdR5qiy9hqFWWXhtSmhV4gZMK6l0VzeOHXXRKx2mU30W8
tLvDXUt2bvapqToTMl1Wu1fKrcwcZd68j51KbzT3cxTUY03QvaejEkww9Xh7mbr3meUhL5IVps7vglt7
JOl4uZ3tp5W1fY7xFZEoefYF04bWjZJKzKDqEtX24tuiXBZXIsqjUu6yuwVxUZqFYeHVvXPqGNVO4XTq
9y2aUaHIWtl5SH023t5HbFBAbtB2UkNNYk+xZ2C3l7E31S3NN/t9Flrh3qTtckFZvKSzN9QzsCobDG5L
uDO2MdmFS21uxOlekWrZQazBkzR1XcQlYeheG7PYrZTLGupV8X2CpkrglT1qSq5WF1c7ljtKcgV5wpVT
pjO7nm5l81r6lYaLmuXV3M1ri5ujtYBpaNu4NFSlbXF5gNQysiIcmb8UaHPg61DDWjKd9TSQQbvTkXdn
ccdUzmx6HX2xO7VVndTh6u6OPK6w6sPCwyc5tTDd9l7DwmqTu3egkVGAl61vMlGHEDE6fVdc9HKsZL0v
O3UQiaceyskI613bt12XJtDbxZSr5UJKnVTEW5QQDLXW+IpHOLXV8tVPGfpitTFgl4XUXHplEH6AzGOy
VZV70FX11yjdYs1DsyHWoOu4Kgu9SXWHLzfODOeKbMWg0evE/K8wyqejq7cx4lNKd84mba2hlb0mmtfA
7tDe09goibyTDORZtbGL+KKUdRZJWWbPR8gTazm1xVlX3W2668t1f1UsNyOg7MtmarjxbVXml0U3/LcW
9ortD1UVIKjpc1OY+ZDlavx5wvgCoIDg+h8QroCXX59R4GZ1w/cIAribLj7B21FaQ134ViFXsKEM9hhu
uVcU8vbqRskGzdZGpih67E0Iul3C/78mvryDBmihRISCU7utpIdKrlCpSc4iGx2QXS3ji1bwgwY85apN
rSx0jW834P5/uaVUdsOtr3JBIMHl7JxzqDoya5wDervn3VQ695xyTDX5rBibtLAlDTF8TjPatEE6lizY
QnbWiXksbVCsg08VTgl1T3sqR+v1OdPyPL9kk8NoecsqxFEssg90gtIWlUtpQvMpymGNlUNCLKzbNECA
1vVH5iXNWSL9EH6z13VuvtkaNmaNhooU5u9fJv6B9I05n18q8Mntzwd52RjxBvNrfFBa6pd2l0qut7sm
aFybeyUXQ13f9mGi4/vpx3I44b47rabxDTuIE4lVpUi5SqulDdRstdSRRKkkhYRrkNbb7RfzYHEXMEEQ
Iweh4z3pIfqv5Rp2jFyN6PVTxnc5EfLb8039Nexwn1PVXc28k7QRIL01XF0dBL45hNqYTfchb6/Kp8eH
9CD02DIs1izNI8X2wJnBkhM2sgUwliG2EJCEXVTlyypPXsNX1QoFkiy4oWZ0kEh8k/UO2LkjMv6eh7/K
ssRvdGujB/Hd9n7Z5dafuPWdj5Oh2FOiIRPma52YVCHLBLnbVcaNVa6hwIfx3Ia8eHSSflalS5CChDO2
SqvTATxcnlL7rl9597EpKLG93L3DOvKEkVXFMTKsz96/WYz357Wpt5+emzhIXP5ESfy5IHhj0mf3ZcmC
qtm5eIXjPIG46n21zisZZz6LNO2m9E3cvCZSmk4/O4eO73nefL7d/gS98PKFlVySTdNJlpFWWo3MB0gc
w2NNSlrpjC5jwmLcwxmq3pjqpnfljCrQPHzvapN74fu3Bb89Q/32u+FQRm36thXO1Dr3RX3wyPl48i0N
VZe54QW3vtIvKpBLiMOOko0aJK74c8S33+utm5tD22yiKpblyIWJUlP3b2Nc2dxS9wuDJSq04pBkIQxJ
CL++ZTXypMDaVIi3I+qLhtHMTdfoEdr9ly97Fts2f5kSuixMXYd6b3G2JypvLy81QEYuata0+FU7UWic
KfcoeDPA76fL7+vw8XpIYjFkZ9xVFR9288FaEOigTur20MCkIm1u8dK+XBstVFQb6Fi61fL+RcMY8Tve
mM575jz9Xu/cv58K9W8a/jkpaUXWQS8HDGeltz2VeUuwnFjd7pPx0f19JTWXw1c+oSHPtU/bFRvGtFOb
fHFVzV4HBjDCQ9AxJYS0tEgmWJ+dauXGLRBoaRstyVY4PGXEyFhpaWmJMD6ywd8BzVN1hIwsH+h+Dhzh
fxyx6v78Dq2UtSpn4ABLhdVbVOmUtx4q7hqxKDR1nToto0uehM29Q3TJ1Z11RxO0JJQPUNILC2uGENTO
ZqSDMs81X1XY3FyV4lAhHYELBsdmW4XJjUMbJnChSTcWFw1Fh7VYm0ecT87/3OvrG/oqlb6Pn+y5jc9d
XNDqocezFZ6129uIzI3krVxkhkNo6S+Nwfq4yqtDjP0DOfWvz371p5ytzsgybtazweiMtGCWGR2p3u5n
stbKV4dlXjO9VwQv2zykWcViruOZvfFZXZba+3UUTWG8dR+G73K2HvPZIRS6DMmrKoeWLfKuKy5601op
01vke3qrt1Gd3RtX1jbGom9dUx3a5dgldzDVUt6klaiTJW3BppVd9iy6W6ZlbUeUUrrmHUU9+2K1qdKI
qFsXYrOh9jeHaa8oE0uqI7vGrvdj32vGU8Zx802Fem8ZynIlpx0yKe5la7hi2peWNtTqGy41GOe2i+0d
MLoaa3A+1ahu725Qpsvp1SzvZrgpazMxlxj4YWiElWEjE1rMQVEFCRVvYpIttYe7VnVu5gOShZoLH+Ti
x0Lo9INrGsQ6eXOtqJ7fdBcqwtrZBSKN5HzqMYVE9JeZgfWuKlbvDVU0TBW8Mqkrv6JA9rpprHqfZS7J
liar6WyQdBzuGtasG3bFDcvWMLzOauhs1Xo6NuZ2da7uV72m7dbZXKdchFM22dHDbvM1cZv1QKVnXFnF
MjLewK3yV5z5ybvfLqWYFuWdkj3Mxd3ZWEIZpQaZ7VsSgUVw2KvS8KN8MHbBqTOWqfXCrxRnsvgV1hcD
TWpyHKTUQ4YF2q3LUpdeZbXbSxYktlc0KSF7vc8SVKoNnKs58C6fcne70vXpVO4XXXzTtIGq1A1Llr7o
Vt5eytrDvRdnXs7GzLwrOWmqrIedcZ9pMvGJMovtNXOkFSipu2rGt18ps67NRoK9cqsLEvcvUdGIk7NN
tiVvVWLIFCMw2orShfBjTfJzrdjWuPyC35Ogh2NqzNhySLljhs07e5YNJ7lbly7W3I6dOGVONOVyRlJ0
7T3OqZRqhSeVRrAji68r7eT0U4UpJxVo5dTigzsVeXDjGx7qtf8GHg3WTeS+9M/HJVfu9s4bZsfp1dRT
WpJJ0HodT9ME5u49Ok/tX+30qod/e88OmqKoaZcNmyNu2xPb2vDtMTbr5Zi/TcXy/BQ/MTAxehS1StKT
daare2qhjHVJI3MaL1tLnaZq6lhSlB5lWwmNhD79yj+A0hJLFGuQ2gimaBCL15YEN2V/dFsm7+/hKQ19
wYw6HDagx1lCkQ+tG9jrarHvOVCd3QtOg1nKPDdJlvZ1Fl4q4q93vuv3fv1zL5w9fLSmsRhHqKTlLzdz
sMWccoSKQ7uXPsZSajToWHVK2iGV9v9p+RsEB/2Mhm12LZdCTwCEefxRi/kHw9fgFlQZdHFiJHneoaar
xs2k9S3Ru3nHOWRpH43ohhGvuifVRXLuMpLd92eovm79w5n9nR4N4yYSJZBmewHsNLSq7TUt3UUd+rEi
3Kcx4xyHOMOXdiC6Ar++H8ZSSWr8UWR+Yfhr5YZkIrV0v9LTnv5o4wn7LsP2tk1e6cGid3Vyon5KY0pw
1zZe/RLarnwpPOVdDEjo0S97PPx9dBRH9N9zPW0q2XkQrKFLiNEYzzfZ6X2m7Ot4pKWnUpbXt2PsjUDv
1bzE341rVOuY3M7Meen5yU/nduozOemXn38Nrj/DfPK3F35yWndhYFURZome9CaIrKVU1hgLIdTkys3O
Yve+o5OuRzz8uIuR02LRrCLBYs/GrQ1WWjClX3OqFkG5JWA6VtNhZRhG1GN28PHClt56+chTtP77G4ez
BeyzsQQp/oWjMjRCK+34+HjA9bQZQvauHMvrxKthqomfw44OG6sXF1rp4N1XAplBvtPy+UXgZq9hrX5m
qq0goKJ4bXmiimnpojvTT29hgcv1vQrWWW7duU2EzkLaqPQY9+q/lwTSIjj3fkOf3nr91HNl9+eOM/LU
zasmoC151xNnZze3ShUxgszEDSFJldhGmg+KLydiDsRaIbIrT78ZvTOFMxYqZ+J925mH2hYOLlSqUJO0
5sIEko5yzeIxkIajUdLSjr1PBmU+21i6hi5p9vHq6/lvF/XZfDkQ6o659++lB+aV/3K8oyJPdS7HUq7q
Wr2EcDyWYbNHGEzV0EA9WGnco7uJDaxqLj9ZyP5JHIC8PrY7uJLEqFY8f6Zwjo8FVvM01dDqrJFIseLX
w10EwOPVBTB5k4QmE+1H3hr84Pj/vaDl3n019XieeY5Z3TLEPxPZtTJibuY+oMbD369zL/XlScuq119O
+4P5JfPgjCkeLjMgVnkvGvNSz31X44dDoug9SHjCDJCT6xjyY27JuExaWhQqbq2MNQyLCBJppsQ9mbUZ
8ZoK5Lvi1pIEhN8H3D8dkp1Bv9V2HZHLeq9vJVtF1uqJ7/O0e6Yni41kg4TdIoEy8PG3AZZ+47lU9u7o
EZhmYLbFxt1NSUpOIfNYRIthv1GiqthRnzl2nOhXbdp2I2Rard0OJIEw3iaFjCsGDzc0Nxn+F3JFOFCQ
zX8JXQ==
'''
