import time
import numpy as np
from multiprocessing import Pool

from scipy import sparse

from jwst import datamodels
from .disperse import dispersed_pixel

import logging

log = logging.getLogger(__name__)
log.setLevel(logging.DEBUG)


class Observation:
    """This class defines an actual observation. It is tied to a single grism image."""

    def __init__(self, direct_images, segmap_model, grism_wcs, filter, ID=0,
                 sed_file=None, extrapolate_sed=False,
                 boundaries=[], renormalize=True, max_cpu=1):

        """
        Initialize all data and metadata for a given observation. Creates lists of
        direct image pixel values for selected objects.

        Parameters
        ----------
        direct_images : List of strings
            List of file name(s) containing direct imaging data
        segmap_model : `jwst.datamodels.ImageModel`
            Segmentation map model
        grism_wcs : gwcs object
            WCS object from grism image
        filter : str
            Filter name
        ID : int
            ID of source to process. If zero, all sources processed.
        sed_file : str
            Name of Spectral Energy Distribution (SED) file containing datasets matching
            the ID in the segmentation file and each consisting of a [[lambda],[flux]] array.
        extrapolate_sed : bool
            Flag indicating whether to extrapolate wavelength range of SED
        boundaries : tuple
            Start/Stop coordinates of the FOV within the larger seed image.
        renormalize : bool
            Flag indicating whether to renormalize SED's
        max_cpu : int
            Max number of cpu's to use when multiprocessing
        """

        # Load all the info for this grism mode
        self.seg_wcs = segmap_model.meta.wcs
        self.grism_wcs = grism_wcs
        self.ID = ID
        self.IDs = []
        self.dir_image_names = direct_images
        self.seg = segmap_model.data
        self.filter = filter
        self.sed_file = sed_file   # should always be NONE for baseline pipeline (use flat SED)
        self.cache = False
        self.renormalize = renormalize
        self.max_cpu = max_cpu

        # Set the limits of the dispersed image to be simulated
        if len(boundaries) == 0:
            log.debug("No boundaries passed.")
            self.xstart = 0
            self.xend = self.xstart + self.dims[0] - 1
            self.ystart = 0
            self.yend = self.ystart + self.dims[1] - 1
        else:
            self.xstart, self.xend, self.ystart, self.yend = boundaries
        self.dims = (self.yend - self.ystart + 1, self.xend - self.xstart + 1)
        log.debug(f"Using simulated image size of {self.dims[1]} {self.dims[0]}")

        # Allow for SED extrapolation
        self.extrapolate_sed = extrapolate_sed
        if self.extrapolate_sed:
            log.warning("SED Extrapolation turned on.")

        # Create pixel lists for sources labeled in segmentation map
        self.create_pixel_list()

    def create_pixel_list(self):
        # Create a list of pixels to be dispersed, grouped per object ID.

        if self.ID == 0:
            # When ID=0, all sources in the segmentation map are processed.
            # This creates a huge list of all x,y pixel indices that have non-zero values
            # in the seg map, sorted by those indices belonging to a particular source ID.
            self.xs = []
            self.ys = []
            all_IDs = np.array(list(set(np.ravel(self.seg))))
            all_IDs = all_IDs[all_IDs > 0]
            self.IDs = all_IDs
            log.info(f"Loading {len(all_IDs)} sources from segmentation map")
            for ID in all_IDs:
                ys, xs = np.nonzero(self.seg == ID)
                if len(xs) > 0 and len(ys) > 0:
                    self.xs.append(xs)
                    self.ys.append(ys)

        else:
            # Process only the given source ID
            log.info(f"Loading source {self.ID} from segmentation map")
            ys, xs = np.nonzero(self.seg == self.ID)
            if len(xs) > 0 and len(ys) > 0:
                self.xs = [xs]
                self.ys = [ys]
                self.IDs = [self.ID]

        # Populate lists of direct image flux values for the sources.
        self.fluxes = {}
        for dir_image_name in self.dir_image_names:

            log.info(f"Using direct image {dir_image_name}")
            dimage = datamodels.open(dir_image_name).data

            if self.sed_file is None:
                # Default pipeline will use sed_file=None, so we need to compute
                # photometry values that used to come from HST-style header keywords.
                # Set pivlam, in units of microns, based on filter name.
                pivlam = float(self.filter[1:4]) / 100.

                # Use pixel fluxes from the direct image.
                self.fluxes[pivlam] = []
                for i in range(len(self.IDs)):
                    # This loads lists of pixel flux values for each source
                    # from the direct image
                    self.fluxes[pivlam].append(dimage[self.ys[i], self.xs[i]])

            else:
                # Use an SED file. Need to normalize the object stamps.
                for ID in self.IDs:
                    vg = self.seg == ID
                    dnew = dimage
                    if self.renormalize:
                        sum_seg = np.sum(dimage[vg])  # But normalize by the whole flux
                        if sum_seg != 0:
                            dimage[vg] /= sum_seg
                    else:
                        log.debug("not renormalizing sources to unity")

                self.fluxes["sed"] = []
                for i in range(len(self.IDs)):
                    self.fluxes["sed"].append(dnew[self.ys[i], self.xs[i]])

    def disperse_all(self, order, wmin, wmax, sens_waves, sens_resp, cache=False):
        """
        Compute dispersed pixel values for all sources identified in
        the segmentation map.

        Parameters
        ----------
        order : int
            Spectral order number to process
        wmin : float
            Minimum wavelength for dispersed spectra
        wmax : float
            Maximum wavelength for dispersed spectra
        sens_waves : float array
            Wavelength array from photom reference file
        sens_resp : float array
            Response (flux calibration) array from photom reference file
        """
        if cache:
            log.debug("Object caching ON")
            self.cache = True
            self.cached_object = {}

        # Initialize the simulated dispersed image
        self.simulated_image = np.zeros(self.dims, float)

        # Loop over all source ID's from segmentation map
        for i in range(len(self.IDs)):
            if self.cache:
                self.cached_object[i] = {}
                self.cached_object[i]['x'] = []
                self.cached_object[i]['y'] = []
                self.cached_object[i]['f'] = []
                self.cached_object[i]['w'] = []
                self.cached_object[i]['minx'] = []
                self.cached_object[i]['maxx'] = []
                self.cached_object[i]['miny'] = []
                self.cached_object[i]['maxy'] = []

            # Disperse object "i"
            self.disperse_chunk(i, order, wmin, wmax, sens_waves, sens_resp)

    def disperse_chunk(self, c, order, wmin, wmax, sens_waves, sens_resp):
        """
        Method that computes dispersion for a single source.
        To be called after create_pixel_list().

        Parameters
        ----------
        c : int
            Chunk (source) number to process
        order : int
            Spectral order number to process
        wmin : float
            Minimum wavelength for dispersed spectra
        wmax : float
            Maximum wavelength for dispersed spectra
        sens_waves : float array
            Wavelength array from photom reference file
        sens_resp : float array
            Response (flux calibration) array from photom reference file
        """

        sid = int(self.IDs[c])
        self.order = order
        self.wmin = wmin
        self.wmax = wmax
        self.sens_waves = sens_waves
        self.sens_resp = sens_resp
        log.info(f"Dispersing source {sid}, order {self.order}")
        pars = []  # initialize params for this object

        # Loop over all pixels in list for object "c"
        log.debug(f"source contains {len(self.xs[c])} pixels")
        for i in range(len(self.xs[c])):

            # Here "i" and "ID" are just indexes into the pixel list for the object
            # being processed, as opposed to the ID number of the object itself
            ID = i

            # xc, yc are the coordinates of the central pixel of the group
            # of pixels surrounding the direct image pixel index
            width = 1.0
            height = 1.0
            xc = self.xs[c][i] + 0.5 * width
            yc = self.ys[c][i] + 0.5 * height

            # "lams" is the array of wavelengths previously stored in flux list
            # and correspond to the central wavelengths of the filters used in
            # the input direct image(s). For the simple case of 1 combined direct image,
            # this contains a single value (e.g. 4.44 for F444W).

            # "fluxes" is the array of pixel values from the direct image(s).
            # For the simple case of 1 combined direct image, this contains a
            # a single value (just like "lams").
            fluxes, lams = map(np.array, zip(*[
                (self.fluxes[lm][c][i], lm) for lm in sorted(self.fluxes.keys())
                if self.fluxes[lm][c][i] != 0
            ]))

            pars_i = (xc, yc, width, height, lams, fluxes, self.order,
                      self.wmin, self.wmax, self.sens_waves, self.sens_resp,
                      self.seg_wcs, self.grism_wcs, ID, self.dims[::-1], 2,
                      self.extrapolate_sed, self.xstart, self.ystart)

            pars.append(pars_i)
            # now have full pars list for all pixels for this object

        time1 = time.time()
        if self.max_cpu > 1:
            mypool = Pool(self.max_cpu)  # Create the pool
            all_res = mypool.imap_unordered(dispersed_pixel, pars)  # Fill the pool
            mypool.close()  # Drain the pool
        else:
            all_res = []
            for i in range(len(pars)):
                all_res.append(dispersed_pixel(*pars[i]))

        # Initialize blank image for this source
        this_object = np.zeros(self.dims, float)

        nres = 0
        for pp in all_res:
            if pp is None:
                continue

            nres += 1
            x, y, _, w, f, *_ = pp

            # skip results that don't have pixels in the field
            if len(x) < 1:
                continue

            minx = int(min(x))
            maxx = int(max(x))
            miny = int(min(y))
            maxy = int(max(y))
            a = sparse.coo_matrix((f, (y - miny, x - minx)),
                                  shape=(maxy - miny + 1, maxx - minx + 1)).toarray()

            # Accumulate results into simulated images
            self.simulated_image[miny:maxy + 1, minx:maxx + 1] += a
            this_object[miny:maxy + 1, minx:maxx + 1] += a

            if self.cache:
                self.cached_object[c]['x'].append(x)
                self.cached_object[c]['y'].append(y)
                self.cached_object[c]['f'].append(f)
                self.cached_object[c]['w'].append(w)
                self.cached_object[c]['minx'].append(minx)
                self.cached_object[c]['maxx'].append(maxx)
                self.cached_object[c]['miny'].append(miny)
                self.cached_object[c]['maxy'].append(maxy)

        time2 = time.time()
        log.debug(f"Elapsed time {time2-time1} sec")

        return this_object

    def disperse_all_from_cache(self, trans=None):
        if not self.cache:
            return

        self.simulated_image = np.zeros(self.dims, float)

        for i in range(len(self.IDs)):
            this_object = self.disperse_chunk_from_cache(i, trans=trans)

        return this_object

    def disperse_chunk_from_cache(self, c, trans=None):
        """Method that handles the dispersion. To be called after create_pixel_list()"""

        if not self.cache:
            return

        time1 = time.time()

        # Initialize blank image for this object
        this_object = np.zeros(self.dims, float)

        if trans is not None:
            log.debug("Applying a transmission function...")

        for i in range(len(self.cached_object[c]['x'])):
            x = self.cached_object[c]['x'][i]
            y = self.cached_object[c]['y'][i]
            f = self.cached_object[c]['f'][i] * 1.
            w = self.cached_object[c]['w'][i]

            if trans is not None:
                f *= trans(w)

            minx = self.cached_object[c]['minx'][i]
            maxx = self.cached_object[c]['maxx'][i]
            miny = self.cached_object[c]['miny'][i]
            maxy = self.cached_object[c]['maxy'][i]

            a = sparse.coo_matrix((f, (y - miny, x - minx)),
                                  shape=(maxy - miny + 1, maxx - minx + 1)).toarray()

            # Accumulate the results into the simulated images
            self.simulated_image[miny:maxy + 1, minx:maxx + 1] += a
            this_object[miny:maxy + 1, minx:maxx + 1] += a

        time2 = time.time()
        log.debug(f"Elapsed time {time2-time1} sec")

        return this_object
