# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['chilean_rut']

package_data = \
{'': ['*']}

entry_points = \
{'console_scripts': ['flake8-markdown = flake8_markdown:main']}

setup_kwargs = {
    'name': 'chilean-rut',
    'version': '0.1.2',
    'description': 'Module to validate, format, clean and get verification digit for Chileans RUT numbers',
    'long_description': '# **Chilean RUT** | [![GitHub license](https://img.shields.io/badge/license-MIT-blue.svg)]()\n\nPython module to validate, format, clean and get verification digit for Chileans RUT/RUN numbers\n\n## **Introduction**\n\n### What is a RUN/RUT?\n\nIt\'s a unique identification number given to every natural person (RUN) or juridical person (RUT).\n* RUN (Rol Único Nacional) \n* RUT (Rol Único Tributario)\n\nFor individuals, the RUN number is the same as the RUT and also in Chile the word "RUT" is commonly used to refer to the RUN number, they are not synonymous but they are used as if they were.\n\nRUT / RUN numbers have seven or eight digits, plus a verification digit, and are generally written in this format: xx.xxx.xxx-z. Where Z can be a digit or the letter K, and it’s determined by "Module 11 algorithm" ensuring that typing mistakes when entering a RUT or RUN number will result in an invalid number.\n\n## **Installation**\n```\n$ pip install chilean_rut\n```\n\n## **Usage**\n\n### **is_valid**\n\nChecks if a Chilean RUT number is valid or not.\n\n| Argument | Type   | Default | Description          |\n| -------- | ------ | ------- | -------------------- |\n| rut      | string | None    | A Chilean RUT Number |\n\n```\nimport chilean_rut\n\n# Valid RUT numbers (correct verification digit)\nchilean_rut.is_valid(\'17.317.684-8\') # True\nchilean_rut.is_valid(\'17317684-8)    # True\nchilean_rut.is_valid(\'173176848)     # True\n# Wrong RUT numbers (bad verification digit)\nchilean_rut.is_valid(\'17.317.684-2)  # False\nchilean_rut.is_valid(\'17317684-2)    # False\nchilean_rut.is_valid(\'173176842)     # False\n```\n\n### **get_verification_digit**\n\nCalculates the verification number or letter.\n\n| Argument | Type   | Default | Description                                     |\n| -------- | ------ | ------- | ----------------------------------------------- |\n| rut      | string | None    | A Chilean RUT number without verification digit |\n\n```\nimport chilean_rut\n\nchilean_rut.get_verification_digit(\'22174688\')     # 0\nchilean_rut.get_verification_digit(\'22191269)      # 1\nchilean_rut.get_verification_digit(\'16615805)      # 2\nchilean_rut.get_verification_digit(\'14505346)      # 3\nchilean_rut.get_verification_digit(\'6088258)       # 4\nchilean_rut.get_verification_digit(\'5391862)       # k\nchilean_rut.get_verification_digit(\'12312-K\')      # ValueError\nchilean_rut.get_verification_digit(\'12.312-K\')     # ValueError\nchilean_rut.get_verification_digit(\'17317684-8\')   # ValueError\nchilean_rut.get_verification_digit(\'12.450.547-k\') # ValueError\n```\n\n### **format_rut**\n\nFormats Chilean RUT number adding dots as thousands separator and a dash before verification digit.\n\n| Argument     | Type   | Default | Description                       |\n| ------------ | ------ | ------- | --------------------------------- |\n| rut          | string | None    | A Chilean RUT Number              |\n| validate_rut | bool   | True    | Validate RUT number before format |\n\n```\nimport chilean_rut\n\n# Valid RUT numbers (correct verification digit)\nchilean_rut.format_rut(\'17317684-8\', True)   # 17.317.684-8\nchilean_rut.format_rut(\'12.450.547-k, True)  # 12.450.547-k\nchilean_rut.format_rut(\'61410767\', True)      # 6.141.076-7\n# Wrong RUT numbers (bad verification digit)\nchilean_rut.format_rut(\'17317684-1\', True)   # ValueError\nchilean_rut.format_rut(\'12.450.547-2\', True)  # ValueError\nchilean_rut.format_rut(\'61410763\', True)      # ValueError\n\n# Valid RUT numbers (correct verification digit)\nchilean_rut.format_rut(\'17317684-8\', False)   # 17.317.684-8\nchilean_rut.format_rut(\'12.450.547-k, False)  # 12.450.547-k\nchilean_rut.format_rut(\'61410767\', False)      # 6.141.076-7\n# Wrong RUT numbers (bad verification digit)\nchilean_rut.format_rut(\'17317684-1\', False)   # 17.317.684-1\nchilean_rut.format_rut(\'12.450.547-2\', False)  # 12.450.547-2\nchilean_rut.format_rut(\'61410763\', False)      # 6.141.076-3\n\n# Invalid RUT format\nchilean_rut.format_rut(\'123.111.111-2\', True)   # ValueError\nchilean_rut.format_rut(\'123.111.111-2\', False) # ValueError\n```\n\n### **clean_rut**\n\nCleans Chilean RUT number removing dots (thousands separador) and dash before verification digit.\n\n| Argument     | Type   | Default | Description                       |\n| ------------ | ------ | ------- | --------------------------------- |\n| rut          | string | None    | A Chilean RUT Number              |\n| validate_rut | bool   | True    | Validates RUT number before clean |\n\n```\nimport chilean_rut\n\n# Valid RUT numbers (correct verification digit)\nchilean_rut.clean_rut(\'17317684-8\', True)   # 173176848\nchilean_rut.clean_rut(\'12.450.547-k, True)  # 12450547k\nchilean_rut.clean_rut(\'61410767, True)      # 61410767\n# Wrong RUT numbers (bad verification digit)\nchilean_rut.clean_rut(\'17317684-1\', True)   # ValueError\nchilean_rut.clean_rut(\'12.450.547-2\', True)  # ValueError\nchilean_rut.clean_rut(\'61410763\', True)      # ValueError\n\n# Valid RUT numbers (correct verification digit)\nchilean_rut.clean_rut(\'17317684-8\', False)   # 173176848\nchilean_rut.clean_rut(\'12.450.547-k, False)  # 12450547k\nchilean_rut.clean_rut(\'61410767, False)      # 61410767\n# Wrong RUT numbers (bad verification digit)\nchilean_rut.clean_rut(\'17317684-1\', False)   # 173176841\nchilean_rut.clean_rut(\'12.450.547-2\', False)  # 124505472\nchilean_rut.clean_rut(\'61410763\', False)      # 61410763\n\n# Invalid RUT format\nchilean_rut.format_rut(\'123.111.111-2,True)   # ValueError\nchilean_rut.format_rut(\'123.111.111-2, False) # ValueError\n```\n\n\n## **Test**\nRunning tests:\n```\n$ pytest\n```\n\nChecking the package installs correctly with different Python versions and interpreters.\n\nTested with python3.6, python3.7, python3.8, python3.9 and python3.10 versions:\n```\n$ tox\n```\n\n## **Contributing**\nContributions are welcome - submit an issue/pull request.',
    'author': 'Rafael Suarez',
    'author_email': 'rafael.asg@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/rafaelsuarezg/chilean_rut',
    'packages': packages,
    'package_data': package_data,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
