"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LowCostECS = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const lib = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfn_tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
class LowCostECS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc =
            props.vpc ??
                new ec2.Vpc(scope, 'Vpc', {
                    natGateways: 0,
                    subnetConfiguration: [
                        {
                            name: 'PublicSubnet',
                            subnetType: ec2.SubnetType.PUBLIC,
                        },
                    ],
                });
        this.cluster = new ecs.Cluster(scope, 'Cluster', {
            vpc: this.vpc,
            containerInsights: props.containerInsights,
        });
        this.hostAutoScalingGroup = this.cluster.addCapacity('HostInstanceCapacity', {
            machineImage: ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.STANDARD, {
                cachedInContext: true,
            }),
            instanceType: new ec2.InstanceType(props.hostInstanceType ?? 't2.micro'),
            spotPrice: props.hostInstanceSpotPrice,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            associatePublicIpAddress: true,
            minCapacity: 1,
            maxCapacity: 1,
        });
        if (props.securityGroup) {
            this.hostAutoScalingGroup.node.tryRemoveChild('InstanceSecurityGroup');
            this.hostAutoScalingGroup.addSecurityGroup(props.securityGroup);
        }
        else {
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(443));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(443));
        }
        /**
         * Add managed policy to allow ssh through ssm manager
         */
        this.hostAutoScalingGroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        /**
         * Add policy to associate elastic ip on startup
         */
        this.hostAutoScalingGroup.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['ec2:DescribeAddresses', 'ec2:AssociateAddress'],
            resources: ['*'],
        }));
        const hostInstanceIp = new ec2.CfnEIP(this, 'HostInstanceIp');
        const tagUniqueId = lib.Names.uniqueId(hostInstanceIp);
        hostInstanceIp.tags.setTag('Name', tagUniqueId);
        const awsCliTag = props.awsCliDockerTag ?? 'latest';
        this.hostAutoScalingGroup.addUserData('INSTANCE_ID=$(curl --silent http://169.254.169.254/latest/meta-data/instance-id)', `ALLOCATION_ID=$(docker run --net=host amazon/aws-cli:${awsCliTag} ec2 describe-addresses --region ${this.hostAutoScalingGroup.env.region} --filter Name=tag:Name,Values=${tagUniqueId} --query 'Addresses[].AllocationId' --output text | head)`, `docker run --net=host amazon/aws-cli:${awsCliTag} ec2 associate-address --region ${this.hostAutoScalingGroup.env.region} --instance-id "$INSTANCE_ID" --allocation-id "$ALLOCATION_ID" --allow-reassociation`);
        this.certFileSystem = new aws_efs_1.FileSystem(this, 'FileSystem', {
            vpc: this.vpc,
            encrypted: true,
            securityGroup: new ec2.SecurityGroup(this, 'FileSystemSecurityGroup', {
                vpc: this.vpc,
                allowAllOutbound: false,
            }),
            removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
        });
        this.certFileSystem.connections.allowDefaultPortTo(this.hostAutoScalingGroup);
        this.certFileSystem.connections.allowDefaultPortFrom(this.hostAutoScalingGroup);
        /**
         * ARecord to Elastic ip
         */
        const hostedZone = route53.HostedZone.fromLookup(this, 'HostedZone', {
            domainName: props.hostedZoneDomain,
        });
        const records = props.recordDomainNames ?? [hostedZone.zoneName];
        records.forEach((record) => new route53.ARecord(this, `ARecord${record}`, {
            zone: hostedZone,
            recordName: record,
            target: route53.RecordTarget.fromIpAddresses(hostInstanceIp.ref),
        }));
        /**
         * Certbot Task Definition
         * Mounts generated certificate to EFS
         */
        const logGroup = props.logGroup ??
            new aws_logs_1.LogGroup(this, 'LogGroup', {
                retention: aws_logs_1.RetentionDays.TWO_YEARS,
                removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
            });
        const certbotTaskDefinition = new ecs.Ec2TaskDefinition(this, 'CertbotTaskDefinition');
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ListHostedZones', 'route53:GetChange'],
            resources: ['*'],
        }));
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ChangeResourceRecordSets'],
            resources: [hostedZone.hostedZoneArn],
        }));
        const certbotTag = props.certbotDockerTag ?? 'v1.29.0';
        const certbotContainer = certbotTaskDefinition.addContainer('CertbotContainer', {
            image: ecs.ContainerImage.fromRegistry(`certbot/dns-route53:${certbotTag}`),
            containerName: 'certbot',
            memoryReservationMiB: 64,
            command: [
                'certonly',
                '--verbose',
                '--preferred-challenges=dns-01',
                '--dns-route53',
                '--dns-route53-propagation-seconds=300',
                '--non-interactive',
                '--agree-tos',
                '--expand',
                '-m',
                props.email,
                '--cert-name',
                records[0],
                ...records.flatMap((domain) => ['-d', domain]),
            ],
            logging: ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: certbotTag,
            }),
        });
        this.certFileSystem.grant(certbotTaskDefinition.taskRole, 'elasticfilesystem:ClientWrite');
        certbotTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        certbotContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: false,
        });
        /**
         * Schedule Certbot certificate create/renew on Step Functions
         * Sends email notification on certbot failure
         */
        this.topic = new aws_sns_1.Topic(this, 'Topic');
        new aws_sns_1.Subscription(this, 'EmailSubscription', {
            topic: this.topic,
            protocol: aws_sns_1.SubscriptionProtocol.EMAIL,
            endpoint: props.email,
        });
        const certbotRunTask = new sfn_tasks.EcsRunTask(this, 'CreateCertificate', {
            cluster: this.cluster,
            taskDefinition: certbotTaskDefinition,
            launchTarget: new sfn_tasks.EcsEc2LaunchTarget(),
            integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        });
        certbotRunTask.addCatch(new sfn_tasks.SnsPublish(this, 'SendEmailOnFailure', {
            topic: this.topic,
            message: sfn.TaskInput.fromJsonPathAt('$'),
        }).next(new sfn.Fail(this, 'Fail')));
        certbotRunTask.addRetry({
            interval: lib.Duration.seconds(20),
        });
        const certbotStateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: certbotRunTask,
        });
        new aws_events_1.Rule(this, 'CertbotScheduleRule', {
            schedule: aws_events_1.Schedule.rate(lib.Duration.days(props.certbotScheduleInterval ?? 60)),
            targets: [new aws_events_targets_1.SfnStateMachine(certbotStateMachine)],
        });
        /**
         * Server ECS task
         */
        this.serverTaskDefinition = props.serverTaskDefinition
            ? this.createTaskDefinition(props.serverTaskDefinition)
            : this.createSampleTaskDefinition(records, logGroup);
        if (!this.serverTaskDefinition.defaultContainer) {
            throw new Error('defaultContainer is required for serverTaskDefinition. Add at least one essential container.');
        }
        this.certFileSystem.grant(this.serverTaskDefinition.taskRole, 'elasticfilesystem:ClientMount');
        this.serverTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        this.serverTaskDefinition.defaultContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: true,
        });
        /**
         * AWS cli container to execute certbot sfn before the default container startup.
         */
        this.serverTaskDefinition.defaultContainer.addContainerDependencies({
            container: this.serverTaskDefinition.addContainer('AWSCliContainer', {
                image: ecs.ContainerImage.fromRegistry(`amazon/aws-cli:${awsCliTag}`),
                containerName: 'aws-cli',
                memoryReservationMiB: 64,
                entryPoint: ['/bin/bash', '-c'],
                command: [
                    `set -eux
          aws configure set region ${certbotStateMachine.env.region} && \\
          aws configure set output text && \\
          EXECUTION_ARN=$(aws stepfunctions start-execution --state-machine-arn ${certbotStateMachine.stateMachineArn} --query executionArn) && \\
          until [ $(aws stepfunctions describe-execution --execution-arn "$EXECUTION_ARN" --query status) != RUNNING ];
          do
            echo "Waiting for $EXECUTION_ARN"
            sleep 10
          done`,
                ],
                essential: false,
                logging: ecs.LogDriver.awsLogs({
                    logGroup: logGroup,
                    streamPrefix: awsCliTag,
                }),
            }),
            condition: ecs.ContainerDependencyCondition.COMPLETE,
        });
        certbotStateMachine.grantExecution(this.serverTaskDefinition.taskRole, 'states:DescribeExecution');
        certbotStateMachine.grantStartExecution(this.serverTaskDefinition.taskRole);
        this.service = new ecs.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition: this.serverTaskDefinition,
            desiredCount: 1,
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            circuitBreaker: {
                rollback: true,
            },
            enableExecuteCommand: true,
        });
        new lib.CfnOutput(this, 'PublicIpAddress', { value: hostInstanceIp.ref });
        new lib.CfnOutput(this, 'certbotStateMachineName', { value: certbotStateMachine.stateMachineName });
        new lib.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
        new lib.CfnOutput(this, 'ServiceName', { value: this.service.serviceName });
    }
    createTaskDefinition(taskDefinitionOptions) {
        const serverTaskDefinition = new ecs.Ec2TaskDefinition(this, 'ServerTaskDefinition', taskDefinitionOptions.taskDefinition);
        taskDefinitionOptions.containers.forEach((containerDefinition, index) => {
            serverTaskDefinition.addContainer(containerDefinition.containerName ?? `container${index}`, containerDefinition);
        });
        taskDefinitionOptions.volumes?.forEach((volume) => serverTaskDefinition.addVolume(volume));
        return serverTaskDefinition;
    }
    createSampleTaskDefinition(records, logGroup) {
        const nginxTaskDefinition = new ecs.Ec2TaskDefinition(this, 'NginxTaskDefinition');
        const nginxContainer = nginxTaskDefinition.addContainer('NginxContainer', {
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../examples/containers/nginx')),
            containerName: 'nginx',
            memoryReservationMiB: 64,
            essential: true,
            environment: {
                SERVER_NAME: records.join(' '),
                CERT_NAME: records[0],
            },
            logging: ecs.LogDrivers.awsLogs({
                logGroup: logGroup,
                streamPrefix: 'nginx-proxy',
            }),
        });
        nginxContainer.addPortMappings({
            hostPort: 80,
            containerPort: 80,
            protocol: ecs.Protocol.TCP,
        }, {
            hostPort: 443,
            containerPort: 443,
            protocol: ecs.Protocol.TCP,
        });
        return nginxTaskDefinition;
    }
}
exports.LowCostECS = LowCostECS;
_a = JSII_RTTI_SYMBOL_1;
LowCostECS[_a] = { fqn: "low-cost-ecs.LowCostECS", version: "0.0.22" };
//# sourceMappingURL=data:application/json;base64,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