"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudwatch = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudwatch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudwatch extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudwatch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudwatch';
        this.accessLevelList = {
            "Write": [
                "DeleteAlarms",
                "DeleteAnomalyDetector",
                "DeleteDashboards",
                "DeleteInsightRules",
                "DisableAlarmActions",
                "DisableInsightRules",
                "EnableAlarmActions",
                "EnableInsightRules",
                "PutAnomalyDetector",
                "PutCompositeAlarm",
                "PutDashboard",
                "PutInsightRule",
                "PutMetricAlarm",
                "PutMetricData",
                "SetAlarmState"
            ],
            "Read": [
                "DescribeAlarmHistory",
                "DescribeAlarms",
                "DescribeAlarmsForMetric",
                "DescribeAnomalyDetectors",
                "DescribeInsightRules",
                "GetDashboard",
                "GetInsightRuleReport",
                "GetMetricData",
                "GetMetricStatistics",
                "GetMetricWidgetImage"
            ],
            "List": [
                "ListDashboards",
                "ListMetrics",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to delete a collection of alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteAlarms.html
     */
    toDeleteAlarms() {
        this.to('cloudwatch:DeleteAlarms');
        return this;
    }
    /**
     * Grants permission to delete the specified anomaly detection model from your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteAnomalyDetector.html
     */
    toDeleteAnomalyDetector() {
        this.to('cloudwatch:DeleteAnomalyDetector');
        return this;
    }
    /**
     * Grants permission to delete all CloudWatch dashboards that you specify
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteDashboards.html
     */
    toDeleteDashboards() {
        this.to('cloudwatch:DeleteDashboards');
        return this;
    }
    /**
     * Grants permission to delete a collection of insight rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DeleteInsightRules.html
     */
    toDeleteInsightRules() {
        this.to('cloudwatch:DeleteInsightRules');
        return this;
    }
    /**
     * Grants permission to retrieve the history for the specified alarm
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmHistory.html
     */
    toDescribeAlarmHistory() {
        this.to('cloudwatch:DescribeAlarmHistory');
        return this;
    }
    /**
     * Grants permission to describe all alarms, currently owned by the user's account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html
     */
    toDescribeAlarms() {
        this.to('cloudwatch:DescribeAlarms');
        return this;
    }
    /**
     * Grants permission to describe all alarms configured on the specified metric, currently owned by the user's account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmsForMetric.html
     */
    toDescribeAlarmsForMetric() {
        this.to('cloudwatch:DescribeAlarmsForMetric');
        return this;
    }
    /**
     * Grants permission to lists the anomaly detection models that you have created in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAnomalyDetectors.html
     */
    toDescribeAnomalyDetectors() {
        this.to('cloudwatch:DescribeAnomalyDetectors');
        return this;
    }
    /**
     * Grants permission to describe all insight rules, currently owned by the user's account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeInsightRules.html
     */
    toDescribeInsightRules() {
        this.to('cloudwatch:DescribeInsightRules');
        return this;
    }
    /**
     * Grants permission to disable actions for a collection of alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DisableAlarmActions.html
     */
    toDisableAlarmActions() {
        this.to('cloudwatch:DisableAlarmActions');
        return this;
    }
    /**
     * Grants permission to disable a collection of insight rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DisableInsightRules.html
     */
    toDisableInsightRules() {
        this.to('cloudwatch:DisableInsightRules');
        return this;
    }
    /**
     * Grants permission to enable actions for a collection of alarms
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_EnableAlarmActions.html
     */
    toEnableAlarmActions() {
        this.to('cloudwatch:EnableAlarmActions');
        return this;
    }
    /**
     * Grants permission to enable a collection of insight rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_EnableInsightRules.html
     */
    toEnableInsightRules() {
        this.to('cloudwatch:EnableInsightRules');
        return this;
    }
    /**
     * Grants permission to display the details of the CloudWatch dashboard you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetDashboard.html
     */
    toGetDashboard() {
        this.to('cloudwatch:GetDashboard');
        return this;
    }
    /**
     * Grants permission to return the top-N report of unique contributors over a time range for a given insight rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetInsightRuleReport.html
     */
    toGetInsightRuleReport() {
        this.to('cloudwatch:GetInsightRuleReport');
        return this;
    }
    /**
     * Grants permission to retrieve batch amounts of CloudWatch metric data and perform metric math on retrieved data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html
     */
    toGetMetricData() {
        this.to('cloudwatch:GetMetricData');
        return this;
    }
    /**
     * Grants permission to retrieve statistics for the specified metric
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html
     */
    toGetMetricStatistics() {
        this.to('cloudwatch:GetMetricStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve snapshots of metric widgets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricWidgetImage.html
     */
    toGetMetricWidgetImage() {
        this.to('cloudwatch:GetMetricWidgetImage');
        return this;
    }
    /**
     * Grants permission to return a list of all CloudWatch dashboards in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        this.to('cloudwatch:ListDashboards');
        return this;
    }
    /**
     * Grants permission to retrieve a list of valid metrics stored for the AWS account owner
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html
     */
    toListMetrics() {
        this.to('cloudwatch:ListMetrics');
        return this;
    }
    /**
     * Grants permission to list tags for an Amazon CloudWatch resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('cloudwatch:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to create or update an anomaly detection model for a CloudWatch metric.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutAnomalyDetector.html
     */
    toPutAnomalyDetector() {
        this.to('cloudwatch:PutAnomalyDetector');
        return this;
    }
    /**
     * Grants permission to create or update a composite alarm
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutCompositeAlarm.html
     */
    toPutCompositeAlarm() {
        this.to('cloudwatch:PutCompositeAlarm');
        return this;
    }
    /**
     * Grants permission to create a CloudWatch dashboard, or update an existing dashboard if it already exists
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutDashboard.html
     */
    toPutDashboard() {
        this.to('cloudwatch:PutDashboard');
        return this;
    }
    /**
     * Grants permission to create a new insight rule or replace an existing insight rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutInsightRule.html
     */
    toPutInsightRule() {
        this.to('cloudwatch:PutInsightRule');
        return this;
    }
    /**
     * Grants permission to create or update an alarm and associates it with the specified Amazon CloudWatch metric
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutMetricAlarm.html
     */
    toPutMetricAlarm() {
        this.to('cloudwatch:PutMetricAlarm');
        return this;
    }
    /**
     * Grants permission to publish metric data points to Amazon CloudWatch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_PutMetricData.html
     */
    toPutMetricData() {
        this.to('cloudwatch:PutMetricData');
        return this;
    }
    /**
     * Grants permission to temporarily set the state of an alarm for testing purposes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_SetAlarmState.html
     */
    toSetAlarmState() {
        this.to('cloudwatch:SetAlarmState');
        return this;
    }
    /**
     * Grants permission to add tags to an Amazon CloudWatch resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('cloudwatch:TagResource');
        return this;
    }
    /**
     * Grants permission to remove a tag from an Amazon CloudWatch resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('cloudwatch:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type alarm to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param alarmName - Identifier for the alarmName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlarm(alarmName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudwatch:${Region}:${Account}:alarm:${AlarmName}';
        arn = arn.replace('${AlarmName}', alarmName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param dashboardName - Identifier for the dashboardName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDashboard(dashboardName, account, partition) {
        var arn = 'arn:${Partition}:cloudwatch::${Account}:dashboard/${DashboardName}';
        arn = arn.replace('${DashboardName}', dashboardName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type insight-rule to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/auth-and-access-control-cw.html
     *
     * @param insightRuleName - Identifier for the insightRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInsightRule(insightRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudwatch:${Region}:${Account}:insight-rule/${InsightRuleName}';
        arn = arn.replace('${InsightRuleName}', insightRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of optional namespace values
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/iam-cw-condition-keys-namespace.html
     *
     * Applies to actions:
     * - .toPutMetricData()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespace(value, operator) {
        return this.if(`cloudwatch:namespace`, value, operator || 'StringLike');
    }
}
exports.Cloudwatch = Cloudwatch;
//# sourceMappingURL=data:application/json;base64,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