"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoIdp = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cognito-idp](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoIdp extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cognito-idp](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cognito-idp';
        this.accessLevelList = {
            "Write": [
                "AddCustomAttributes",
                "AdminAddUserToGroup",
                "AdminConfirmSignUp",
                "AdminCreateUser",
                "AdminDeleteUser",
                "AdminDeleteUserAttributes",
                "AdminDisableProviderForUser",
                "AdminDisableUser",
                "AdminEnableUser",
                "AdminForgetDevice",
                "AdminInitiateAuth",
                "AdminLinkProviderForUser",
                "AdminRemoveUserFromGroup",
                "AdminResetUserPassword",
                "AdminRespondToAuthChallenge",
                "AdminSetUserMFAPreference",
                "AdminSetUserPassword",
                "AdminSetUserSettings",
                "AdminUpdateAuthEventFeedback",
                "AdminUpdateDeviceStatus",
                "AdminUpdateUserAttributes",
                "AdminUserGlobalSignOut",
                "AssociateSoftwareToken",
                "ChangePassword",
                "ConfirmDevice",
                "ConfirmForgotPassword",
                "ConfirmSignUp",
                "CreateGroup",
                "CreateIdentityProvider",
                "CreateResourceServer",
                "CreateUserImportJob",
                "CreateUserPool",
                "CreateUserPoolClient",
                "CreateUserPoolDomain",
                "DeleteGroup",
                "DeleteIdentityProvider",
                "DeleteResourceServer",
                "DeleteUser",
                "DeleteUserAttributes",
                "DeleteUserPool",
                "DeleteUserPoolClient",
                "DeleteUserPoolDomain",
                "ForgetDevice",
                "ForgotPassword",
                "GlobalSignOut",
                "InitiateAuth",
                "ResendConfirmationCode",
                "RespondToAuthChallenge",
                "SetRiskConfiguration",
                "SetUICustomization",
                "SetUserMFAPreference",
                "SetUserPoolMfaConfig",
                "SetUserSettings",
                "SignUp",
                "StartUserImportJob",
                "StopUserImportJob",
                "UpdateAuthEventFeedback",
                "UpdateDeviceStatus",
                "UpdateGroup",
                "UpdateIdentityProvider",
                "UpdateResourceServer",
                "UpdateUserAttributes",
                "UpdateUserPool",
                "UpdateUserPoolClient",
                "UpdateUserPoolDomain",
                "VerifySoftwareToken",
                "VerifyUserAttribute"
            ],
            "Read": [
                "AdminGetDevice",
                "AdminGetUser",
                "AdminListUserAuthEvents",
                "DescribeIdentityProvider",
                "DescribeResourceServer",
                "DescribeRiskConfiguration",
                "DescribeUserImportJob",
                "DescribeUserPool",
                "DescribeUserPoolClient",
                "DescribeUserPoolDomain",
                "GetCSVHeader",
                "GetDevice",
                "GetGroup",
                "GetIdentityProviderByIdentifier",
                "GetSigningCertificate",
                "GetUICustomization",
                "GetUser",
                "GetUserAttributeVerificationCode",
                "GetUserPoolMfaConfig"
            ],
            "List": [
                "AdminListDevices",
                "AdminListGroupsForUser",
                "ListDevices",
                "ListGroups",
                "ListIdentityProviders",
                "ListResourceServers",
                "ListTagsForResource",
                "ListUserImportJobs",
                "ListUserPoolClients",
                "ListUserPools",
                "ListUsers",
                "ListUsersInGroup"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Adds additional user attributes to the user pool schema.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AddCustomAttributes.html
     */
    toAddCustomAttributes() {
        this.to('cognito-idp:AddCustomAttributes');
        return this;
    }
    /**
     * Adds the specified user to the specified group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminAddUserToGroup.html
     */
    toAdminAddUserToGroup() {
        this.to('cognito-idp:AdminAddUserToGroup');
        return this;
    }
    /**
     * Confirms user registration as an admin without using a confirmation code. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminConfirmSignUp.html
     */
    toAdminConfirmSignUp() {
        this.to('cognito-idp:AdminConfirmSignUp');
        return this;
    }
    /**
     * Creates a new user in the specified user pool and sends a welcome message via email or phone (SMS).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminCreateUser.html
     */
    toAdminCreateUser() {
        this.to('cognito-idp:AdminCreateUser');
        return this;
    }
    /**
     * Deletes a user as an administrator. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDeleteUser.html
     */
    toAdminDeleteUser() {
        this.to('cognito-idp:AdminDeleteUser');
        return this;
    }
    /**
     * Deletes the user attributes in a user pool as an administrator. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDeleteUserAttributes.html
     */
    toAdminDeleteUserAttributes() {
        this.to('cognito-idp:AdminDeleteUserAttributes');
        return this;
    }
    /**
     * Disables the user from signing in with the specified external (SAML or social) identity provider.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDisableProviderForUser.html
     */
    toAdminDisableProviderForUser() {
        this.to('cognito-idp:AdminDisableProviderForUser');
        return this;
    }
    /**
     * Disables the specified user as an administrator. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDisableUser.html
     */
    toAdminDisableUser() {
        this.to('cognito-idp:AdminDisableUser');
        return this;
    }
    /**
     * Enables the specified user as an administrator. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminEnableUser.html
     */
    toAdminEnableUser() {
        this.to('cognito-idp:AdminEnableUser');
        return this;
    }
    /**
     * Forgets the device, as an administrator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminForgetDevice.html
     */
    toAdminForgetDevice() {
        this.to('cognito-idp:AdminForgetDevice');
        return this;
    }
    /**
     * Gets the device, as an administrator.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminGetDevice.html
     */
    toAdminGetDevice() {
        this.to('cognito-idp:AdminGetDevice');
        return this;
    }
    /**
     * Gets the specified user by user name in a user pool as an administrator. Works on any user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminGetUser.html
     */
    toAdminGetUser() {
        this.to('cognito-idp:AdminGetUser');
        return this;
    }
    /**
     * Authenticates a user in a user pool as an administrator. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html
     */
    toAdminInitiateAuth() {
        this.to('cognito-idp:AdminInitiateAuth');
        return this;
    }
    /**
     * Links an existing user account in a user pool (DestinationUser) to an identity from an external identity provider (SourceUser) based on a specified attribute name and value from the external identity provider.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminLinkProviderForUser.html
     */
    toAdminLinkProviderForUser() {
        this.to('cognito-idp:AdminLinkProviderForUser');
        return this;
    }
    /**
     * Lists devices, as an administrator.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListDevices.html
     */
    toAdminListDevices() {
        this.to('cognito-idp:AdminListDevices');
        return this;
    }
    /**
     * Lists the groups that the user belongs to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html
     */
    toAdminListGroupsForUser() {
        this.to('cognito-idp:AdminListGroupsForUser');
        return this;
    }
    /**
     * Lists the authentication events for the user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListUserAuthEvents.html
     */
    toAdminListUserAuthEvents() {
        this.to('cognito-idp:AdminListUserAuthEvents');
        return this;
    }
    /**
     * Removes the specified user from the specified group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminRemoveUserFromGroup.html
     */
    toAdminRemoveUserFromGroup() {
        this.to('cognito-idp:AdminRemoveUserFromGroup');
        return this;
    }
    /**
     * Resets the specified user's password in a user pool as an administrator. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminResetUserPassword.html
     */
    toAdminResetUserPassword() {
        this.to('cognito-idp:AdminResetUserPassword');
        return this;
    }
    /**
     * Responds to an authentication challenge, as an administrator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminRespondToAuthChallenge.html
     */
    toAdminRespondToAuthChallenge() {
        this.to('cognito-idp:AdminRespondToAuthChallenge');
        return this;
    }
    /**
     * Sets MFA preference for the user in the userpool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserMFAPreference.html
     */
    toAdminSetUserMFAPreference() {
        this.to('cognito-idp:AdminSetUserMFAPreference');
        return this;
    }
    /**
     * Sets the specified user's password in a user pool as an administrator. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserPassword.html
     */
    toAdminSetUserPassword() {
        this.to('cognito-idp:AdminSetUserPassword');
        return this;
    }
    /**
     * Sets all the user settings for a specified user name. Works on any user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserSettings.html
     */
    toAdminSetUserSettings() {
        this.to('cognito-idp:AdminSetUserSettings');
        return this;
    }
    /**
     * Updates the feedback for the user authentication event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateAuthEventFeedback.html
     */
    toAdminUpdateAuthEventFeedback() {
        this.to('cognito-idp:AdminUpdateAuthEventFeedback');
        return this;
    }
    /**
     * Updates the device status as an administrator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateDeviceStatus.html
     */
    toAdminUpdateDeviceStatus() {
        this.to('cognito-idp:AdminUpdateDeviceStatus');
        return this;
    }
    /**
     * Updates the specified user's attributes, including developer attributes, as an administrator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html
     */
    toAdminUpdateUserAttributes() {
        this.to('cognito-idp:AdminUpdateUserAttributes');
        return this;
    }
    /**
     * Signs out users from all devices, as an administrator.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUserGlobalSignOut.html
     */
    toAdminUserGlobalSignOut() {
        this.to('cognito-idp:AdminUserGlobalSignOut');
        return this;
    }
    /**
     * Returns a unique generated shared secret key code for the user account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AssociateSoftwareToken.html
     */
    toAssociateSoftwareToken() {
        this.to('cognito-idp:AssociateSoftwareToken');
        return this;
    }
    /**
     * Changes the password for a specified user in a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ChangePassword.html
     */
    toChangePassword() {
        this.to('cognito-idp:ChangePassword');
        return this;
    }
    /**
     * Confirms tracking of the device. This API call is the call that begins device tracking.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmDevice.html
     */
    toConfirmDevice() {
        this.to('cognito-idp:ConfirmDevice');
        return this;
    }
    /**
     * Allows a user to enter a confirmation code to reset a forgotten password.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmForgotPassword.html
     */
    toConfirmForgotPassword() {
        this.to('cognito-idp:ConfirmForgotPassword');
        return this;
    }
    /**
     * Confirms registration of a user and handles the existing alias from a previous user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmSignUp.html
     */
    toConfirmSignUp() {
        this.to('cognito-idp:ConfirmSignUp');
        return this;
    }
    /**
     * Creates a new group in the specified user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        this.to('cognito-idp:CreateGroup');
        return this;
    }
    /**
     * Creates an identity provider for a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateIdentityProvider.html
     */
    toCreateIdentityProvider() {
        this.to('cognito-idp:CreateIdentityProvider');
        return this;
    }
    /**
     * Creates a new OAuth2.0 resource server and defines custom scopes in it.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateResourceServer.html
     */
    toCreateResourceServer() {
        this.to('cognito-idp:CreateResourceServer');
        return this;
    }
    /**
     * Creates the user import job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserImportJob.html
     */
    toCreateUserImportJob() {
        this.to('cognito-idp:CreateUserImportJob');
        return this;
    }
    /**
     * Creates a new Amazon Cognito user pool and sets the password policy for the pool.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html
     */
    toCreateUserPool() {
        this.to('cognito-idp:CreateUserPool');
        return this;
    }
    /**
     * Creates the user pool client.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPoolClient.html
     */
    toCreateUserPoolClient() {
        this.to('cognito-idp:CreateUserPoolClient');
        return this;
    }
    /**
     * Creates a new domain for a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPoolDomain.html
     */
    toCreateUserPoolDomain() {
        this.to('cognito-idp:CreateUserPoolDomain');
        return this;
    }
    /**
     * Deletes a group. Currently only groups with no members can be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.to('cognito-idp:DeleteGroup');
        return this;
    }
    /**
     * Deletes an identity provider for a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteIdentityProvider.html
     */
    toDeleteIdentityProvider() {
        this.to('cognito-idp:DeleteIdentityProvider');
        return this;
    }
    /**
     * Deletes a resource server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteResourceServer.html
     */
    toDeleteResourceServer() {
        this.to('cognito-idp:DeleteResourceServer');
        return this;
    }
    /**
     * Allows a user to delete one's self.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('cognito-idp:DeleteUser');
        return this;
    }
    /**
     * Deletes the attributes for a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserAttributes.html
     */
    toDeleteUserAttributes() {
        this.to('cognito-idp:DeleteUserAttributes');
        return this;
    }
    /**
     * Deletes the specified Amazon Cognito user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPool.html
     */
    toDeleteUserPool() {
        this.to('cognito-idp:DeleteUserPool');
        return this;
    }
    /**
     * Allows the developer to delete the user pool client.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPoolClient.html
     */
    toDeleteUserPoolClient() {
        this.to('cognito-idp:DeleteUserPoolClient');
        return this;
    }
    /**
     * Deletes a domain for a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPoolDomain.html
     */
    toDeleteUserPoolDomain() {
        this.to('cognito-idp:DeleteUserPoolDomain');
        return this;
    }
    /**
     * Gets information about a specific identity provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeIdentityProvider.html
     */
    toDescribeIdentityProvider() {
        this.to('cognito-idp:DescribeIdentityProvider');
        return this;
    }
    /**
     * Describes a resource server.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeResourceServer.html
     */
    toDescribeResourceServer() {
        this.to('cognito-idp:DescribeResourceServer');
        return this;
    }
    /**
     * Describes the risk configuration setting for the userpool / userpool client
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeRiskConfiguration.html
     */
    toDescribeRiskConfiguration() {
        this.to('cognito-idp:DescribeRiskConfiguration');
        return this;
    }
    /**
     * Describes the user import job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserImportJob.html
     */
    toDescribeUserImportJob() {
        this.to('cognito-idp:DescribeUserImportJob');
        return this;
    }
    /**
     * Returns the configuration information and metadata of the specified user pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPool.html
     */
    toDescribeUserPool() {
        this.to('cognito-idp:DescribeUserPool');
        return this;
    }
    /**
     * Client method for returning the configuration information and metadata of the specified user pool client.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPoolClient.html
     */
    toDescribeUserPoolClient() {
        this.to('cognito-idp:DescribeUserPoolClient');
        return this;
    }
    /**
     * Gets information about a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPoolDomain.html
     */
    toDescribeUserPoolDomain() {
        this.to('cognito-idp:DescribeUserPoolDomain');
        return this;
    }
    /**
     * Forgets the specified device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ForgetDevice.html
     */
    toForgetDevice() {
        this.to('cognito-idp:ForgetDevice');
        return this;
    }
    /**
     * Calling this API causes a message to be sent to the end user with a confirmation code that is required to change the user's password.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ForgotPassword.html
     */
    toForgotPassword() {
        this.to('cognito-idp:ForgotPassword');
        return this;
    }
    /**
     * Gets the header information for the .csv file to be used as input for the user import job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetCSVHeader.html
     */
    toGetCSVHeader() {
        this.to('cognito-idp:GetCSVHeader');
        return this;
    }
    /**
     * Gets the device.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetDevice.html
     */
    toGetDevice() {
        this.to('cognito-idp:GetDevice');
        return this;
    }
    /**
     * Gets a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetGroup.html
     */
    toGetGroup() {
        this.to('cognito-idp:GetGroup');
        return this;
    }
    /**
     * Gets the specified identity provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetIdentityProviderByIdentifier.html
     */
    toGetIdentityProviderByIdentifier() {
        this.to('cognito-idp:GetIdentityProviderByIdentifier');
        return this;
    }
    /**
     * Returns the signing certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetSigningCertificate.html
     */
    toGetSigningCertificate() {
        this.to('cognito-idp:GetSigningCertificate');
        return this;
    }
    /**
     * Gets the UI Customization information for a particular app client's app UI, if there is something set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUICustomization.html
     */
    toGetUICustomization() {
        this.to('cognito-idp:GetUICustomization');
        return this;
    }
    /**
     * Gets the user attributes and metadata for a user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        this.to('cognito-idp:GetUser');
        return this;
    }
    /**
     * Gets the user attribute verification code for the specified attribute name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUserAttributeVerificationCode.html
     */
    toGetUserAttributeVerificationCode() {
        this.to('cognito-idp:GetUserAttributeVerificationCode');
        return this;
    }
    /**
     * Gets the MFA configuration for the userpool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUserPoolMfaConfig.html
     */
    toGetUserPoolMfaConfig() {
        this.to('cognito-idp:GetUserPoolMfaConfig');
        return this;
    }
    /**
     * Signs out users from all devices.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GlobalSignOut.html
     */
    toGlobalSignOut() {
        this.to('cognito-idp:GlobalSignOut');
        return this;
    }
    /**
     * Initiates the authentication flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_InitiateAuth.html
     */
    toInitiateAuth() {
        this.to('cognito-idp:InitiateAuth');
        return this;
    }
    /**
     * Lists the devices.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListDevices.html
     */
    toListDevices() {
        this.to('cognito-idp:ListDevices');
        return this;
    }
    /**
     * Lists the groups associated with a user pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        this.to('cognito-idp:ListGroups');
        return this;
    }
    /**
     * Lists information about all identity providers for a user pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListIdentityProviders.html
     */
    toListIdentityProviders() {
        this.to('cognito-idp:ListIdentityProviders');
        return this;
    }
    /**
     * Lists the resource servers for a user pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListResourceServers.html
     */
    toListResourceServers() {
        this.to('cognito-idp:ListResourceServers');
        return this;
    }
    /**
     * Lists the tags that are assigned to an Amazon Cognito user pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('cognito-idp:ListTagsForResource');
        return this;
    }
    /**
     * Lists the user import jobs..
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserImportJobs.html
     */
    toListUserImportJobs() {
        this.to('cognito-idp:ListUserImportJobs');
        return this;
    }
    /**
     * Lists the clients that have been created for the specified user pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserPoolClients.html
     */
    toListUserPoolClients() {
        this.to('cognito-idp:ListUserPoolClients');
        return this;
    }
    /**
     * Lists the user pools associated with an AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserPools.html
     */
    toListUserPools() {
        this.to('cognito-idp:ListUserPools');
        return this;
    }
    /**
     * Lists the users in the Amazon Cognito user pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.to('cognito-idp:ListUsers');
        return this;
    }
    /**
     * Lists the users in the specified group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUsersInGroup.html
     */
    toListUsersInGroup() {
        this.to('cognito-idp:ListUsersInGroup');
        return this;
    }
    /**
     * Resends the confirmation (for confirmation of registration) to a specific user in the user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ResendConfirmationCode.html
     */
    toResendConfirmationCode() {
        this.to('cognito-idp:ResendConfirmationCode');
        return this;
    }
    /**
     * Responds to the authentication challenge.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_RespondToAuthChallenge.html
     */
    toRespondToAuthChallenge() {
        this.to('cognito-idp:RespondToAuthChallenge');
        return this;
    }
    /**
     * sets the risk configuration setting for the userpool / userpool client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetRiskConfiguration.html
     */
    toSetRiskConfiguration() {
        this.to('cognito-idp:SetRiskConfiguration');
        return this;
    }
    /**
     * Sets the UI customization information for a user pool's built-in app UI.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUICustomization.html
     */
    toSetUICustomization() {
        this.to('cognito-idp:SetUICustomization');
        return this;
    }
    /**
     * Sets MFA preference for the user in the userpool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserMFAPreference.html
     */
    toSetUserMFAPreference() {
        this.to('cognito-idp:SetUserMFAPreference');
        return this;
    }
    /**
     * Sets the MFA configuration for the userpool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserPoolMfaConfig.html
     */
    toSetUserPoolMfaConfig() {
        this.to('cognito-idp:SetUserPoolMfaConfig');
        return this;
    }
    /**
     * Sets the user settings like multi-factor authentication (MFA).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserSettings.html
     */
    toSetUserSettings() {
        this.to('cognito-idp:SetUserSettings');
        return this;
    }
    /**
     * Registers the user in the specified user pool and creates a user name, password, and user attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SignUp.html
     */
    toSignUp() {
        this.to('cognito-idp:SignUp');
        return this;
    }
    /**
     * Starts the user import.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StartUserImportJob.html
     */
    toStartUserImportJob() {
        this.to('cognito-idp:StartUserImportJob');
        return this;
    }
    /**
     * Stops the user import job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StopUserImportJob.html
     */
    toStopUserImportJob() {
        this.to('cognito-idp:StopUserImportJob');
        return this;
    }
    /**
     * Assigns a set of tags to an Amazon Cognito user pool.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('cognito-idp:TagResource');
        return this;
    }
    /**
     * Removes the specified tags from an Amazon Cognito user pool.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('cognito-idp:UntagResource');
        return this;
    }
    /**
     * Updates the feedback for the user authentication event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateAuthEventFeedback.html
     */
    toUpdateAuthEventFeedback() {
        this.to('cognito-idp:UpdateAuthEventFeedback');
        return this;
    }
    /**
     * Updates the device status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateDeviceStatus.html
     */
    toUpdateDeviceStatus() {
        this.to('cognito-idp:UpdateDeviceStatus');
        return this;
    }
    /**
     * Updates the specified group with the specified attributes.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        this.to('cognito-idp:UpdateGroup');
        return this;
    }
    /**
     * Updates identity provider information for a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateIdentityProvider.html
     */
    toUpdateIdentityProvider() {
        this.to('cognito-idp:UpdateIdentityProvider');
        return this;
    }
    /**
     * Updates the name and scopes of resource server.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateResourceServer.html
     */
    toUpdateResourceServer() {
        this.to('cognito-idp:UpdateResourceServer');
        return this;
    }
    /**
     * Allows a user to update a specific attribute (one at a time).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserAttributes.html
     */
    toUpdateUserAttributes() {
        this.to('cognito-idp:UpdateUserAttributes');
        return this;
    }
    /**
     * Updates the specified user pool with the specified attributes.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html
     */
    toUpdateUserPool() {
        this.to('cognito-idp:UpdateUserPool');
        return this;
    }
    /**
     * Allows the developer to update the specified user pool client and password policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPoolClient.html
     */
    toUpdateUserPoolClient() {
        this.to('cognito-idp:UpdateUserPoolClient');
        return this;
    }
    /**
     * Updates the Secure Sockets Layer (SSL) certificate for the custom domain for your user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPoolDomain.html
     */
    toUpdateUserPoolDomain() {
        this.to('cognito-idp:UpdateUserPoolDomain');
        return this;
    }
    /**
     * Registers a user's entered TOTP code and mark the user's software token MFA status as verified if successful.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_VerifySoftwareToken.html
     */
    toVerifySoftwareToken() {
        this.to('cognito-idp:VerifySoftwareToken');
        return this;
    }
    /**
     * Verifies a user attribute using a one time verification code.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_VerifyUserAttribute.html
     */
    toVerifyUserAttribute() {
        this.to('cognito-idp:VerifyUserAttribute');
        return this;
    }
    /**
     * Adds a resource of type userpool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/resource-permissions.html#amazon-cognito-amazon-resource-names
     *
     * @param userPoolId - Identifier for the userPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUserpool(userPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-idp:${Region}:${Account}:userpool/${UserPoolId}';
        arn = arn.replace('${UserPoolId}', userPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoIdp = CognitoIdp;
//# sourceMappingURL=data:application/json;base64,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