"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Events = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Events extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'events';
        this.accessLevelList = {
            "Write": [
                "ActivateEventSource",
                "CancelReplay",
                "CreateArchive",
                "CreateEventBus",
                "CreatePartnerEventSource",
                "DeactivateEventSource",
                "DeleteArchive",
                "DeleteEventBus",
                "DeletePartnerEventSource",
                "DeleteRule",
                "DisableRule",
                "EnableRule",
                "PutEvents",
                "PutPartnerEvents",
                "PutPermission",
                "PutTargets",
                "RemovePermission",
                "RemoveTargets",
                "StartReplay",
                "UpdateArchive"
            ],
            "Read": [
                "DescribeArchive",
                "DescribeEventBus",
                "DescribeEventSource",
                "DescribePartnerEventSource",
                "DescribeReplay",
                "DescribeRule",
                "TestEventPattern"
            ],
            "List": [
                "ListArchives",
                "ListEventBuses",
                "ListEventSources",
                "ListPartnerEventSourceAccounts",
                "ListPartnerEventSources",
                "ListReplays",
                "ListRuleNamesByTarget",
                "ListRules",
                "ListTagsForResource",
                "ListTargetsByRule"
            ],
            "Tagging": [
                "PutRule",
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to activate partner event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html
     */
    toActivateEventSource() {
        this.to('events:ActivateEventSource');
        return this;
    }
    /**
     * Grants permission to cancel a replay
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CancelReplay.html
     */
    toCancelReplay() {
        this.to('events:CancelReplay');
        return this;
    }
    /**
     * Grants permission to create a new archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateArchive.html
     */
    toCreateArchive() {
        this.to('events:CreateArchive');
        return this;
    }
    /**
     * Grants permission to create event buses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html
     */
    toCreateEventBus() {
        this.to('events:CreateEventBus');
        return this;
    }
    /**
     * Grants permission to create partner event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreatePartnerEventSource.html
     */
    toCreatePartnerEventSource() {
        this.to('events:CreatePartnerEventSource');
        return this;
    }
    /**
     * Grants permission to deactivate event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeactivateEventSource.html
     */
    toDeactivateEventSource() {
        this.to('events:DeactivateEventSource');
        return this;
    }
    /**
     * Grants permission to delete an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteArchive.html
     */
    toDeleteArchive() {
        this.to('events:DeleteArchive');
        return this;
    }
    /**
     * Grants permission to delete event buses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    toDeleteEventBus() {
        this.to('events:DeleteEventBus');
        return this;
    }
    /**
     * Grants permission to delete partner event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    toDeletePartnerEventSource() {
        this.to('events:DeletePartnerEventSource');
        return this;
    }
    /**
     * Grants permission to delete rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteRule.html
     */
    toDeleteRule() {
        this.to('events:DeleteRule');
        return this;
    }
    /**
     * Grants permission to retrieve details about an archive
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeArchive.html
     */
    toDescribeArchive() {
        this.to('events:DescribeArchive');
        return this;
    }
    /**
     * Grants permission to retrieve details about event buses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html
     */
    toDescribeEventBus() {
        this.to('events:DescribeEventBus');
        return this;
    }
    /**
     * Grants permission to retrieve details about event sources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html
     */
    toDescribeEventSource() {
        this.to('events:DescribeEventSource');
        return this;
    }
    /**
     * Grants permission to retrieve details about partner event sources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribePartnerEventSource.html
     */
    toDescribePartnerEventSource() {
        this.to('events:DescribePartnerEventSource');
        return this;
    }
    /**
     * Grants permission to retrieve the details of a replay
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeReplay.html
     */
    toDescribeReplay() {
        this.to('events:DescribeReplay');
        return this;
    }
    /**
     * Grants permission to retrieve details about rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeRule.html
     */
    toDescribeRule() {
        this.to('events:DescribeRule');
        return this;
    }
    /**
     * Grants permission to disable rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html
     */
    toDisableRule() {
        this.to('events:DisableRule');
        return this;
    }
    /**
     * Grants permissions to enable rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_EnableRule.html
     */
    toEnableRule() {
        this.to('events:EnableRule');
        return this;
    }
    /**
     * Grants permission to retrieve a list of archives
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListArchives.html
     */
    toListArchives() {
        this.to('events:ListArchives');
        return this;
    }
    /**
     * Grants permission to to retrieve a list of the event buses in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventBuses.html
     */
    toListEventBuses() {
        this.to('events:ListEventBuses');
        return this;
    }
    /**
     * Grants permission to to retrieve a list of event sources shared with this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventSources.html
     */
    toListEventSources() {
        this.to('events:ListEventSources');
        return this;
    }
    /**
     * Grants permission to retrieve a list of AWS account IDs associated with an event source
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSourceAccounts.html
     */
    toListPartnerEventSourceAccounts() {
        this.to('events:ListPartnerEventSourceAccounts');
        return this;
    }
    /**
     * Grants permission to retrieve a list partner event sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSources.html
     */
    toListPartnerEventSources() {
        this.to('events:ListPartnerEventSources');
        return this;
    }
    /**
     * Grants permission to retrieve a list of replays
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListReplays.html
     */
    toListReplays() {
        this.to('events:ListReplays');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the names of the rules associated with a target
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRuleNamesByTarget.html
     */
    toListRuleNamesByTarget() {
        this.to('events:ListRuleNamesByTarget');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the Amazon EventBridge rules in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRules.html
     */
    toListRules() {
        this.to('events:ListRules');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags associated with an Amazon EventBridge resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('events:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to retrieve a list of targets defined for a rule
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html
     */
    toListTargetsByRule() {
        this.to('events:ListTargetsByRule');
        return this;
    }
    /**
     * Grants permission to send custom events to Amazon EventBridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html
     */
    toPutEvents() {
        this.to('events:PutEvents');
        return this;
    }
    /**
     * Grants permission to sends custom events to Amazon EventBridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPartnerEvents.html
     */
    toPutPartnerEvents() {
        this.to('events:PutPartnerEvents');
        return this;
    }
    /**
     * Grants permission to use the PutPermission action to grants permission to another AWS account to put events to your default event bus
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html
     */
    toPutPermission() {
        this.to('events:PutPermission');
        return this;
    }
    /**
     * Grants permission to create or updates rules
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDetailUserIdentityPrincipalId()
     * - .ifDetailType()
     * - .ifSource()
     * - .ifDetailService()
     * - .ifDetailEventTypeCode()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutRule.html
     */
    toPutRule() {
        this.to('events:PutRule');
        return this;
    }
    /**
     * Grants permission to add targets to a rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetArn()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutTargets.html
     */
    toPutTargets() {
        this.to('events:PutTargets');
        return this;
    }
    /**
     * Grants permission to revoke the permission of another AWS account to put events to your default event bus
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemovePermission.html
     */
    toRemovePermission() {
        this.to('events:RemovePermission');
        return this;
    }
    /**
     * Grants permission to removes targets from a rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html
     */
    toRemoveTargets() {
        this.to('events:RemoveTargets');
        return this;
    }
    /**
     * Grants permission to start a replay of an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_StartReplay.html
     */
    toStartReplay() {
        this.to('events:StartReplay');
        return this;
    }
    /**
     * Grants permission to add a tag to an Amazon EventBridge resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('events:TagResource');
        return this;
    }
    /**
     * Grants permissions to test whether an event pattern matches the provided event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TestEventPattern.html
     */
    toTestEventPattern() {
        this.to('events:TestEventPattern');
        return this;
    }
    /**
     * Grants permission to remove a tag from an Amazon EventBridge resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('events:UntagResource');
        return this;
    }
    /**
     * Grants permission to update an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UpdateArchive.html
     */
    toUpdateArchive() {
        this.to('events:UpdateArchive');
        return this;
    }
    /**
     * Adds a resource of type event-source to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventSourceName - Identifier for the eventSourceName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventSource(eventSourceName, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}::event-source/${EventSourceName}';
        arn = arn.replace('${EventSourceName}', eventSourceName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-bus to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventBusName - Identifier for the eventBusName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventBus(eventBusName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:event-bus/${EventBusName}';
        arn = arn.replace('${EventBusName}', eventBusName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:rule/${RuleName}';
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type archive to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param archiveName - Identifier for the archiveName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onArchive(archiveName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:archive/${ArchiveName}';
        arn = arn.replace('${ArchiveName}', archiveName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type replay to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param replayName - Identifier for the replayName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReplay(replayName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:replay/${ReplayName}';
        arn = arn.replace('${ReplayName}', replayName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Used internally by AWS services. If a rule is created by an AWS service on your behalf, the value is the principal name of the service that created the rule.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifManagedBy(value, operator) {
        return this.if(`events:ManagedBy`, value, operator || 'StringLike');
    }
    /**
     * The ARN of a target that can be put to a rule.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limiting-access-to-targets
     *
     * Applies to actions:
     * - .toPutTargets()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTargetArn(value, operator) {
        return this.if(`events:TargetArn`, value, operator || 'ArnLike');
    }
    /**
     * Matches the literal string of the detail-type filed of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-pattern-detail-type
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailType(value, operator) {
        return this.if(`events:detail-type`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.eventTypeCode field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-type-code
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailEventTypeCode(value, operator) {
        return this.if(`events:detail.eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.service field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-service
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailService(value, operator) {
        return this.if(`events:detail.service`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.useridentity.principalid field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#consume-specific-events
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailUserIdentityPrincipalId(value, operator) {
        return this.if(`events:detail.userIdentity.principalId`, value, operator || 'StringLike');
    }
    /**
     * The AWS service or AWS partner event source that generated the event. Matches the literal string of the source field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-limit-access-control
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSource(value, operator) {
        return this.if(`events:source`, value, operator || 'StringLike');
    }
}
exports.Events = Events;
//# sourceMappingURL=data:application/json;base64,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