"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codestar = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codestar](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestar.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codestar extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codestar](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodestar.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codestar';
        this.accessLevelList = {
            "Permissions management": [
                "AssociateTeamMember",
                "CreateProject",
                "DeleteProject",
                "DisassociateTeamMember",
                "UpdateTeamMember"
            ],
            "Write": [
                "CreateUserProfile",
                "DeleteExtendedAccess",
                "DeleteUserProfile",
                "PutExtendedAccess",
                "UpdateProject",
                "UpdateUserProfile"
            ],
            "Read": [
                "DescribeProject",
                "DescribeUserProfile",
                "GetExtendedAccess"
            ],
            "List": [
                "ListProjects",
                "ListResources",
                "ListTagsForProject",
                "ListTeamMembers",
                "ListUserProfiles"
            ],
            "Tagging": [
                "TagProject",
                "UntagProject"
            ]
        };
    }
    /**
     * Adds a user to the team for an AWS CodeStar project.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_AssociateTeamMember.html
     */
    toAssociateTeamMember() {
        this.to('codestar:AssociateTeamMember');
        return this;
    }
    /**
     * Creates a project with minimal structure, customer policies, and no resources.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        this.to('codestar:CreateProject');
        return this;
    }
    /**
     * Creates a profile for a user that includes user preferences, display name, and email.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_CreateUserProfile.html
     */
    toCreateUserProfile() {
        this.to('codestar:CreateUserProfile');
        return this;
    }
    /**
     * Grants access to extended delete APIs.
     *
     * Access Level: Write
     */
    toDeleteExtendedAccess() {
        this.to('codestar:DeleteExtendedAccess');
        return this;
    }
    /**
     * Deletes a project, including project resources. Does not delete users associated with the project, but does delete the IAM roles that allowed access to the project.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        this.to('codestar:DeleteProject');
        return this;
    }
    /**
     * Deletes a user profile in AWS CodeStar, including all personal preference data associated with that profile, such as display name and email address. It does not delete the history of that user, for example the history of commits made by that user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DeleteUserProfile.html
     */
    toDeleteUserProfile() {
        this.to('codestar:DeleteUserProfile');
        return this;
    }
    /**
     * Describes a project and its resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DescribeProject.html
     */
    toDescribeProject() {
        this.to('codestar:DescribeProject');
        return this;
    }
    /**
     * Describes a user in AWS CodeStar and the user attributes across all projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DescribeUserProfile.html
     */
    toDescribeUserProfile() {
        this.to('codestar:DescribeUserProfile');
        return this;
    }
    /**
     * Removes a user from a project. Removing a user from a project also removes the IAM policies from that user that allowed access to the project and its resources.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_DisassociateTeamMember.html
     */
    toDisassociateTeamMember() {
        this.to('codestar:DisassociateTeamMember');
        return this;
    }
    /**
     * Grants access to extended read APIs.
     *
     * Access Level: Read
     */
    toGetExtendedAccess() {
        this.to('codestar:GetExtendedAccess');
        return this;
    }
    /**
     * Lists all projects in CodeStar associated with your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListProjects.html
     */
    toListProjects() {
        this.to('codestar:ListProjects');
        return this;
    }
    /**
     * Lists all resources associated with a project in CodeStar.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListResources.html
     */
    toListResources() {
        this.to('codestar:ListResources');
        return this;
    }
    /**
     * Lists the tags associated with a project in CodeStar.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListTagsForProject.html
     */
    toListTagsForProject() {
        this.to('codestar:ListTagsForProject');
        return this;
    }
    /**
     * Lists all team members associated with a project.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListTeamMembers.html
     */
    toListTeamMembers() {
        this.to('codestar:ListTeamMembers');
        return this;
    }
    /**
     * Lists user profiles in AWS CodeStar.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_ListUserProfiles.html
     */
    toListUserProfiles() {
        this.to('codestar:ListUserProfiles');
        return this;
    }
    /**
     * Grants access to extended write APIs.
     *
     * Access Level: Write
     */
    toPutExtendedAccess() {
        this.to('codestar:PutExtendedAccess');
        return this;
    }
    /**
     * Adds tags to a project in CodeStar.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_TagProject.html
     */
    toTagProject() {
        this.to('codestar:TagProject');
        return this;
    }
    /**
     * Removes tags from a project in CodeStar.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UntagProject.html
     */
    toUntagProject() {
        this.to('codestar:UntagProject');
        return this;
    }
    /**
     * Updates a project in CodeStar.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject() {
        this.to('codestar:UpdateProject');
        return this;
    }
    /**
     * Updates team member attributes within a CodeStar project.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateTeamMember.html
     */
    toUpdateTeamMember() {
        this.to('codestar:UpdateTeamMember');
        return this;
    }
    /**
     * Updates a profile for a user that includes user preferences, display name, and email.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar/latest/APIReference/API_UpdateUserProfile.html
     */
    toUpdateUserProfile() {
        this.to('codestar:UpdateUserProfile');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codestar/latest/userguide/working-with-projects.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectId, account, region, partition) {
        var arn = 'arn:${Partition}:codestar:${Region}:${Account}:project/${ProjectId}';
        arn = arn.replace('${ProjectId}', projectId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * @param userNameWithPath - Identifier for the userNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onUser(userNameWithPath, account, region, partition) {
        var arn = 'arn:${Partition}:iam:${Region}:${Account}:user/${UserNameWithPath}';
        arn = arn.replace('${UserNameWithPath}', userNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Applies to resource types:
     * - user
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`iam:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Codestar = Codestar;
//# sourceMappingURL=data:application/json;base64,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