"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Config = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [config](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsconfig.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Config extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [config](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsconfig.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'config';
        this.accessLevelList = {
            "Read": [
                "BatchGetAggregateResourceConfig",
                "BatchGetResourceConfig",
                "DeliverConfigSnapshot",
                "DescribeConformancePackCompliance",
                "DescribeConformancePackStatus",
                "DescribeConformancePacks",
                "DescribeOrganizationConfigRuleStatuses",
                "DescribeOrganizationConfigRules",
                "DescribeOrganizationConformancePackStatuses",
                "DescribeOrganizationConformancePacks",
                "GetAggregateComplianceDetailsByConfigRule",
                "GetAggregateConfigRuleComplianceSummary",
                "GetAggregateDiscoveredResourceCounts",
                "GetAggregateResourceConfig",
                "GetComplianceDetailsByConfigRule",
                "GetComplianceDetailsByResource",
                "GetComplianceSummaryByConfigRule",
                "GetComplianceSummaryByResourceType",
                "GetConformancePackComplianceDetails",
                "GetConformancePackComplianceSummary",
                "GetDiscoveredResourceCounts",
                "GetOrganizationConfigRuleDetailedStatus",
                "GetOrganizationConformancePackDetailedStatus",
                "GetResourceConfigHistory",
                "SelectAggregateResourceConfig",
                "SelectResourceConfig"
            ],
            "Write": [
                "DeleteAggregationAuthorization",
                "DeleteConfigRule",
                "DeleteConfigurationAggregator",
                "DeleteConfigurationRecorder",
                "DeleteConformancePack",
                "DeleteDeliveryChannel",
                "DeleteEvaluationResults",
                "DeleteOrganizationConfigRule",
                "DeleteOrganizationConformancePack",
                "DeletePendingAggregationRequest",
                "DeleteRemediationConfiguration",
                "DeleteRemediationExceptions",
                "DeleteResourceConfig",
                "DeleteRetentionConfiguration",
                "PutAggregationAuthorization",
                "PutConfigRule",
                "PutConfigurationAggregator",
                "PutConfigurationRecorder",
                "PutConformancePack",
                "PutDeliveryChannel",
                "PutEvaluations",
                "PutOrganizationConfigRule",
                "PutOrganizationConformancePack",
                "PutRemediationConfigurations",
                "PutRemediationExceptions",
                "PutResourceConfig",
                "PutRetentionConfiguration",
                "StartConfigRulesEvaluation",
                "StartConfigurationRecorder",
                "StartRemediationExecution",
                "StopConfigurationRecorder"
            ],
            "List": [
                "DescribeAggregateComplianceByConfigRules",
                "DescribeAggregationAuthorizations",
                "DescribeComplianceByConfigRule",
                "DescribeComplianceByResource",
                "DescribeConfigRuleEvaluationStatus",
                "DescribeConfigRules",
                "DescribeConfigurationAggregatorSourcesStatus",
                "DescribeConfigurationAggregators",
                "DescribeConfigurationRecorderStatus",
                "DescribeConfigurationRecorders",
                "DescribeDeliveryChannelStatus",
                "DescribeDeliveryChannels",
                "DescribePendingAggregationRequests",
                "DescribeRemediationConfigurations",
                "DescribeRemediationExceptions",
                "DescribeRemediationExecutionStatus",
                "DescribeRetentionConfigurations",
                "ListAggregateDiscoveredResources",
                "ListDiscoveredResources",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Returns the current configuration items for resources that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetAggregateResourceConfig.html
     */
    toBatchGetAggregateResourceConfig() {
        this.to('config:BatchGetAggregateResourceConfig');
        return this;
    }
    /**
     * Returns the current configuration for one or more requested resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetResourceConfig.html
     */
    toBatchGetResourceConfig() {
        this.to('config:BatchGetResourceConfig');
        return this;
    }
    /**
     * Deletes the authorization granted to the specified configuration aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteAggregationAuthorization.html
     */
    toDeleteAggregationAuthorization() {
        this.to('config:DeleteAggregationAuthorization');
        return this;
    }
    /**
     * Deletes the specified AWS Config rule and all of its evaluation results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigRule.html
     */
    toDeleteConfigRule() {
        this.to('config:DeleteConfigRule');
        return this;
    }
    /**
     * Deletes the specified configuration aggregator and the aggregated data associated with the aggregator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationAggregator.html
     */
    toDeleteConfigurationAggregator() {
        this.to('config:DeleteConfigurationAggregator');
        return this;
    }
    /**
     * Deletes the configuration recorder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationRecorder.html
     */
    toDeleteConfigurationRecorder() {
        this.to('config:DeleteConfigurationRecorder');
        return this;
    }
    /**
     * Deletes the specified conformance pack and all the AWS Config rules and all evaluation results within that conformance pack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConformancePack.html
     */
    toDeleteConformancePack() {
        this.to('config:DeleteConformancePack');
        return this;
    }
    /**
     * Deletes the delivery channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteDeliveryChannel.html
     */
    toDeleteDeliveryChannel() {
        this.to('config:DeleteDeliveryChannel');
        return this;
    }
    /**
     * Deletes the evaluation results for the specified Config rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteEvaluationResults.html
     */
    toDeleteEvaluationResults() {
        this.to('config:DeleteEvaluationResults');
        return this;
    }
    /**
     * Deletes the specified organization config rule and all of its evaluation results from all member accounts in that organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConfigRule.html
     */
    toDeleteOrganizationConfigRule() {
        this.to('config:DeleteOrganizationConfigRule');
        return this;
    }
    /**
     * Deletes the specified organization conformance pack and all of its evaluation results from all member accounts in that organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConformancePack.html
     */
    toDeleteOrganizationConformancePack() {
        this.to('config:DeleteOrganizationConformancePack');
        return this;
    }
    /**
     * Deletes pending authorization requests for a specified aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeletePendingAggregationRequest.html
     */
    toDeletePendingAggregationRequest() {
        this.to('config:DeletePendingAggregationRequest');
        return this;
    }
    /**
     * Deletes the remediation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationConfiguration.html
     */
    toDeleteRemediationConfiguration() {
        this.to('config:DeleteRemediationConfiguration');
        return this;
    }
    /**
     * Deletes one or more remediation exceptions for specific resource keys for a specific AWS Config Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationExceptions.html
     */
    toDeleteRemediationExceptions() {
        this.to('config:DeleteRemediationExceptions');
        return this;
    }
    /**
     * Records the configuration state for a custom resource that has been deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteResourceConfig.html
     */
    toDeleteResourceConfig() {
        this.to('config:DeleteResourceConfig');
        return this;
    }
    /**
     * Deletes the retention configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRetentionConfiguration.html
     */
    toDeleteRetentionConfiguration() {
        this.to('config:DeleteRetentionConfiguration');
        return this;
    }
    /**
     * Schedules delivery of a configuration snapshot to the Amazon S3 bucket in the specified delivery channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeliverConfigSnapshot.html
     */
    toDeliverConfigSnapshot() {
        this.to('config:DeliverConfigSnapshot');
        return this;
    }
    /**
     * Returns a list of compliant and noncompliant rules with the number of resources for compliant and noncompliant rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregateComplianceByConfigRules.html
     */
    toDescribeAggregateComplianceByConfigRules() {
        this.to('config:DescribeAggregateComplianceByConfigRules');
        return this;
    }
    /**
     * Returns a list of authorizations granted to various aggregator accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregationAuthorizations.html
     */
    toDescribeAggregationAuthorizations() {
        this.to('config:DescribeAggregationAuthorizations');
        return this;
    }
    /**
     * Indicates whether the specified AWS Config rules are compliant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByConfigRule.html
     */
    toDescribeComplianceByConfigRule() {
        this.to('config:DescribeComplianceByConfigRule');
        return this;
    }
    /**
     * Indicates whether the specified AWS resources are compliant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByResource.html
     */
    toDescribeComplianceByResource() {
        this.to('config:DescribeComplianceByResource');
        return this;
    }
    /**
     * Returns status information for each of your AWS managed Config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRuleEvaluationStatus.html
     */
    toDescribeConfigRuleEvaluationStatus() {
        this.to('config:DescribeConfigRuleEvaluationStatus');
        return this;
    }
    /**
     * Returns details about your AWS Config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRules.html
     */
    toDescribeConfigRules() {
        this.to('config:DescribeConfigRules');
        return this;
    }
    /**
     * Returns status information for sources within an aggregator
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregatorSourcesStatus.html
     */
    toDescribeConfigurationAggregatorSourcesStatus() {
        this.to('config:DescribeConfigurationAggregatorSourcesStatus');
        return this;
    }
    /**
     * Returns the details of one or more configuration aggregators
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregators.html
     */
    toDescribeConfigurationAggregators() {
        this.to('config:DescribeConfigurationAggregators');
        return this;
    }
    /**
     * Returns the current status of the specified configuration recorder
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorderStatus.html
     */
    toDescribeConfigurationRecorderStatus() {
        this.to('config:DescribeConfigurationRecorderStatus');
        return this;
    }
    /**
     * Returns the name of one or more specified configuration recorders
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorders.html
     */
    toDescribeConfigurationRecorders() {
        this.to('config:DescribeConfigurationRecorders');
        return this;
    }
    /**
     * Returns compliance information for each rule in that conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackCompliance.html
     */
    toDescribeConformancePackCompliance() {
        this.to('config:DescribeConformancePackCompliance');
        return this;
    }
    /**
     * Provides one or more conformance packs deployment status.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackStatus.html
     */
    toDescribeConformancePackStatus() {
        this.to('config:DescribeConformancePackStatus');
        return this;
    }
    /**
     * Returns a list of one or more conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePacks.html
     */
    toDescribeConformancePacks() {
        this.to('config:DescribeConformancePacks');
        return this;
    }
    /**
     * Returns the current status of the specified delivery channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannelStatus.html
     */
    toDescribeDeliveryChannelStatus() {
        this.to('config:DescribeDeliveryChannelStatus');
        return this;
    }
    /**
     * Returns details about the specified delivery channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannels.html
     */
    toDescribeDeliveryChannels() {
        this.to('config:DescribeDeliveryChannels');
        return this;
    }
    /**
     * Provides organization config rule deployment status for an organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRuleStatuses.html
     */
    toDescribeOrganizationConfigRuleStatuses() {
        this.to('config:DescribeOrganizationConfigRuleStatuses');
        return this;
    }
    /**
     * Returns a list of organization config rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRules.html
     */
    toDescribeOrganizationConfigRules() {
        this.to('config:DescribeOrganizationConfigRules');
        return this;
    }
    /**
     * Provides organization conformance pack deployment status for an organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePackStatuses.html
     */
    toDescribeOrganizationConformancePackStatuses() {
        this.to('config:DescribeOrganizationConformancePackStatuses');
        return this;
    }
    /**
     * Returns a list of organization conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePacks.html
     */
    toDescribeOrganizationConformancePacks() {
        this.to('config:DescribeOrganizationConformancePacks');
        return this;
    }
    /**
     * Returns a list of all pending aggregation requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribePendingAggregationRequests.html
     */
    toDescribePendingAggregationRequests() {
        this.to('config:DescribePendingAggregationRequests');
        return this;
    }
    /**
     * Returns the details of one or more remediation configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationConfigurations.html
     */
    toDescribeRemediationConfigurations() {
        this.to('config:DescribeRemediationConfigurations');
        return this;
    }
    /**
     * Returns the details of one or more remediation exceptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExceptions.html
     */
    toDescribeRemediationExceptions() {
        this.to('config:DescribeRemediationExceptions');
        return this;
    }
    /**
     * Provides a detailed view of a Remediation Execution for a set of resources including state, timestamps and any error messages for steps that have failed
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExecutionStatus.html
     */
    toDescribeRemediationExecutionStatus() {
        this.to('config:DescribeRemediationExecutionStatus');
        return this;
    }
    /**
     * Returns the details of one or more retention configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRetentionConfigurations.html
     */
    toDescribeRetentionConfigurations() {
        this.to('config:DescribeRetentionConfigurations');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS Config rule for a specific resource in a rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateComplianceDetailsByConfigRule.html
     */
    toGetAggregateComplianceDetailsByConfigRule() {
        this.to('config:GetAggregateComplianceDetailsByConfigRule');
        return this;
    }
    /**
     * Returns the number of compliant and noncompliant rules for one or more accounts and regions in an aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateConfigRuleComplianceSummary.html
     */
    toGetAggregateConfigRuleComplianceSummary() {
        this.to('config:GetAggregateConfigRuleComplianceSummary');
        return this;
    }
    /**
     * Returns the resource counts across accounts and regions that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateDiscoveredResourceCounts.html
     */
    toGetAggregateDiscoveredResourceCounts() {
        this.to('config:GetAggregateDiscoveredResourceCounts');
        return this;
    }
    /**
     * Returns configuration item that is aggregated for your specific resource in a specific source account and region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateResourceConfig.html
     */
    toGetAggregateResourceConfig() {
        this.to('config:GetAggregateResourceConfig');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS Config rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByConfigRule.html
     */
    toGetComplianceDetailsByConfigRule() {
        this.to('config:GetComplianceDetailsByConfigRule');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByResource.html
     */
    toGetComplianceDetailsByResource() {
        this.to('config:GetComplianceDetailsByResource');
        return this;
    }
    /**
     * Returns the number of AWS Config rules that are compliant and noncompliant, up to a maximum of 25 for each
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByConfigRule.html
     */
    toGetComplianceSummaryByConfigRule() {
        this.to('config:GetComplianceSummaryByConfigRule');
        return this;
    }
    /**
     * Returns the number of resources that are compliant and the number that are noncompliant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByResourceType.html
     */
    toGetComplianceSummaryByResourceType() {
        this.to('config:GetComplianceSummaryByResourceType');
        return this;
    }
    /**
     * Returns compliance details of a conformance pack for all AWS resources that are monitered by conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceDetails.html
     */
    toGetConformancePackComplianceDetails() {
        this.to('config:GetConformancePackComplianceDetails');
        return this;
    }
    /**
     * Provides compliance summary for one or more conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceSummary.html
     */
    toGetConformancePackComplianceSummary() {
        this.to('config:GetConformancePackComplianceSummary');
        return this;
    }
    /**
     * Returns the resource types, the number of each resource type, and the total number of resources that AWS Config is recording in this region for your AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetDiscoveredResourceCounts.html
     */
    toGetDiscoveredResourceCounts() {
        this.to('config:GetDiscoveredResourceCounts');
        return this;
    }
    /**
     * Returns detailed status for each member account within an organization for a given organization config rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConfigRuleDetailedStatus.html
     */
    toGetOrganizationConfigRuleDetailedStatus() {
        this.to('config:GetOrganizationConfigRuleDetailedStatus');
        return this;
    }
    /**
     * Returns detailed status for each member account within an organization for a given organization conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConformancePackDetailedStatus.html
     */
    toGetOrganizationConformancePackDetailedStatus() {
        this.to('config:GetOrganizationConformancePackDetailedStatus');
        return this;
    }
    /**
     * Returns a list of configuration items for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetResourceConfigHistory.html
     */
    toGetResourceConfigHistory() {
        this.to('config:GetResourceConfigHistory');
        return this;
    }
    /**
     * Accepts a resource type and returns a list of resource identifiers that are aggregated for a specific resource type across accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListAggregateDiscoveredResources.html
     */
    toListAggregateDiscoveredResources() {
        this.to('config:ListAggregateDiscoveredResources');
        return this;
    }
    /**
     * Accepts a resource type and returns a list of resource identifiers for the resources of that type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListDiscoveredResources.html
     */
    toListDiscoveredResources() {
        this.to('config:ListDiscoveredResources');
        return this;
    }
    /**
     * List the tags for AWS Config resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('config:ListTagsForResource');
        return this;
    }
    /**
     * Authorizes the aggregator account and region to collect data from the source account and region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutAggregationAuthorization.html
     */
    toPutAggregationAuthorization() {
        this.to('config:PutAggregationAuthorization');
        return this;
    }
    /**
     * Adds or updates an AWS Config rule for evaluating whether your AWS resources comply with your desired configurations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigRule.html
     */
    toPutConfigRule() {
        this.to('config:PutConfigRule');
        return this;
    }
    /**
     * Creates and updates the configuration aggregator with the selected source accounts and regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationAggregator.html
     */
    toPutConfigurationAggregator() {
        this.to('config:PutConfigurationAggregator');
        return this;
    }
    /**
     * Creates a new configuration recorder to record the selected resource configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationRecorder.html
     */
    toPutConfigurationRecorder() {
        this.to('config:PutConfigurationRecorder');
        return this;
    }
    /**
     * Creates or updates a conformance pack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConformancePack.html
     */
    toPutConformancePack() {
        this.to('config:PutConformancePack');
        return this;
    }
    /**
     * Creates a delivery channel object to deliver configuration information to an Amazon S3 bucket and Amazon SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutDeliveryChannel.html
     */
    toPutDeliveryChannel() {
        this.to('config:PutDeliveryChannel');
        return this;
    }
    /**
     * Used by an AWS Lambda function to deliver evaluation results to AWS Config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutEvaluations.html
     */
    toPutEvaluations() {
        this.to('config:PutEvaluations');
        return this;
    }
    /**
     * Adds or updates organization config rule for your entire organization evaluating whether your AWS resources comply with your desired configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConfigRule.html
     */
    toPutOrganizationConfigRule() {
        this.to('config:PutOrganizationConfigRule');
        return this;
    }
    /**
     * Adds or updates organization conformance pack for your entire organization evaluating whether your AWS resources comply with your desired configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConformancePack.html
     */
    toPutOrganizationConformancePack() {
        this.to('config:PutOrganizationConformancePack');
        return this;
    }
    /**
     * Adds or updates the remediation configuration with a specific AWS Config rule with the selected target or action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationConfigurations.html
     */
    toPutRemediationConfigurations() {
        this.to('config:PutRemediationConfigurations');
        return this;
    }
    /**
     * Adds or updates remediation exceptions for specific resources for a specific AWS Config rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationExceptions.html
     */
    toPutRemediationExceptions() {
        this.to('config:PutRemediationExceptions');
        return this;
    }
    /**
     * Records the configuration state for the resource provided in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutResourceConfig.html
     */
    toPutResourceConfig() {
        this.to('config:PutResourceConfig');
        return this;
    }
    /**
     * Creates and updates the retention configuration with details about retention period (number of days) that AWS Config stores your historical information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRetentionConfiguration.html
     */
    toPutRetentionConfiguration() {
        this.to('config:PutRetentionConfiguration');
        return this;
    }
    /**
     * Accepts a structured query language (SQL) SELECT command and an aggregator to query configuration state of AWS resources across multiple accounts and regions, performs the corresponding search, and returns resource configurations matching the properties.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectAggregateResourceConfig.html
     */
    toSelectAggregateResourceConfig() {
        this.to('config:SelectAggregateResourceConfig');
        return this;
    }
    /**
     * Accepts a structured query language (SQL) SELECT command, performs the corresponding search, and returns resource configurations matching the properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectResourceConfig.html
     */
    toSelectResourceConfig() {
        this.to('config:SelectResourceConfig');
        return this;
    }
    /**
     * Evaluates your resources against the specified Config rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigRulesEvaluation.html
     */
    toStartConfigRulesEvaluation() {
        this.to('config:StartConfigRulesEvaluation');
        return this;
    }
    /**
     * Starts recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigurationRecorder.html
     */
    toStartConfigurationRecorder() {
        this.to('config:StartConfigurationRecorder');
        return this;
    }
    /**
     * Runs an on-demand remediation for the specified AWS Config rules against the last known remediation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartRemediationExecution.html
     */
    toStartRemediationExecution() {
        this.to('config:StartRemediationExecution');
        return this;
    }
    /**
     * Stops recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StopConfigurationRecorder.html
     */
    toStopConfigurationRecorder() {
        this.to('config:StopConfigurationRecorder');
        return this;
    }
    /**
     * Associates the specified tags to a resource with the specified resourceArn
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('config:TagResource');
        return this;
    }
    /**
     * Deletes specified tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('config:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type AggregationAuthorization to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_AggregationAuthorization.html
     *
     * @param aggregatorAccount - Identifier for the aggregatorAccount.
     * @param aggregatorRegion - Identifier for the aggregatorRegion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAggregationAuthorization(aggregatorAccount, aggregatorRegion, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:aggregation-authorization/${AggregatorAccount}/${AggregatorRegion}';
        arn = arn.replace('${AggregatorAccount}', aggregatorAccount);
        arn = arn.replace('${AggregatorRegion}', aggregatorRegion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConfigurationAggregator to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigurationAggregator.html
     *
     * @param aggregatorId - Identifier for the aggregatorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationAggregator(aggregatorId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:config-aggregator/${AggregatorId}';
        arn = arn.replace('${AggregatorId}', aggregatorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigRule.html
     *
     * @param configRuleId - Identifier for the configRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigRule(configRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:config-rule/${ConfigRuleId}';
        arn = arn.replace('${ConfigRuleId}', configRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConformancePack.html
     *
     * @param conformancePackName - Identifier for the conformancePackName.
     * @param conformancePackId - Identifier for the conformancePackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConformancePack(conformancePackName, conformancePackId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:conformance-pack/${ConformancePackName}/${ConformancePackId}';
        arn = arn.replace('${ConformancePackName}', conformancePackName);
        arn = arn.replace('${ConformancePackId}', conformancePackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type OrganizationConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConfigRule.html
     *
     * @param organizationConfigRuleId - Identifier for the organizationConfigRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganizationConfigRule(organizationConfigRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:organization-config-rule/${OrganizationConfigRuleId}';
        arn = arn.replace('${OrganizationConfigRuleId}', organizationConfigRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type OrganizationConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConformancePack.html
     *
     * @param organizationConformancePackId - Identifier for the organizationConformancePackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganizationConformancePack(organizationConformancePackId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:organization-conformance-pack/${OrganizationConformancePackId}';
        arn = arn.replace('${OrganizationConformancePackId}', organizationConformancePackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RemediationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_RemediationConfiguration.html
     *
     * @param remediationConfigurationId - Identifier for the remediationConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRemediationConfiguration(remediationConfigurationId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:remediation-configuration/${RemediationConfigurationId}';
        arn = arn.replace('${RemediationConfigurationId}', remediationConfigurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Config = Config;
//# sourceMappingURL=data:application/json;base64,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