"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deepcomposer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [deepcomposer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepcomposer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deepcomposer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [deepcomposer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepcomposer.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'deepcomposer';
        this.accessLevelList = {
            "Write": [
                "AssociateCoupon",
                "CreateAudio",
                "CreateComposition",
                "CreateModel",
                "DeleteComposition",
                "DeleteModel",
                "UpdateComposition",
                "UpdateModel"
            ],
            "Read": [
                "GetComposition",
                "GetModel",
                "GetSampleModel"
            ],
            "List": [
                "ListCompositions",
                "ListModels",
                "ListSampleModels",
                "ListTagsForResource",
                "ListTrainingTopics"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Associates a DeepComposer coupon (or DSN) with the account associated with the sender of the request.
     *
     * Access Level: Write
     */
    toAssociateCoupon() {
        this.to('deepcomposer:AssociateCoupon');
        return this;
    }
    /**
     * Creates an audio file by converting the midi composition into a wav or mp3 file.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toCreateAudio() {
        this.to('deepcomposer:CreateAudio');
        return this;
    }
    /**
     * Creates a multi-track midi composition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toCreateComposition() {
        this.to('deepcomposer:CreateComposition');
        return this;
    }
    /**
     * Starts creating/training a generative-model that is able to perform inference against the user-provided piano-melody to create a multi-track midi composition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    toCreateModel() {
        this.to('deepcomposer:CreateModel');
        return this;
    }
    /**
     * Deletes the composition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toDeleteComposition() {
        this.to('deepcomposer:DeleteComposition');
        return this;
    }
    /**
     * Deletes the model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    toDeleteModel() {
        this.to('deepcomposer:DeleteModel');
        return this;
    }
    /**
     * Returns information about the composition.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toGetComposition() {
        this.to('deepcomposer:GetComposition');
        return this;
    }
    /**
     * Returns information about the model.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    toGetModel() {
        this.to('deepcomposer:GetModel');
        return this;
    }
    /**
     * Returns information about the sample/pre-trained DeepComposer model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toGetSampleModel() {
        this.to('deepcomposer:GetSampleModel');
        return this;
    }
    /**
     * Returns a list of all the compositions owned by the sender of the request.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toListCompositions() {
        this.to('deepcomposer:ListCompositions');
        return this;
    }
    /**
     * Returns a list of all the models owned by the sender of the request.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    toListModels() {
        this.to('deepcomposer:ListModels');
        return this;
    }
    /**
     * Returns a list of all the sample/pre-trained models provided by the DeepComposer service.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toListSampleModels() {
        this.to('deepcomposer:ListSampleModels');
        return this;
    }
    /**
     * Grants permission to lists tag for a resource.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    toListTagsForResource() {
        this.to('deepcomposer:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of all the training options or topic for creating/training a model.
     *
     * Access Level: List
     */
    toListTrainingTopics() {
        this.to('deepcomposer:ListTrainingTopics');
        return this;
    }
    /**
     * Grants permission to tag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     */
    toTagResource() {
        this.to('deepcomposer:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     */
    toUntagResource() {
        this.to('deepcomposer:UntagResource');
        return this;
    }
    /**
     * Modifies the mutable properties associated with a composition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    toUpdateComposition() {
        this.to('deepcomposer:UpdateComposition');
        return this;
    }
    /**
     * Modifies the mutable properties associated with a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    toUpdateModel() {
        this.to('deepcomposer:UpdateModel');
        return this;
    }
    /**
     * Adds a resource of type model to the statement
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     *
     * @param modelId - Identifier for the modelId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(modelId, account, region, partition) {
        var arn = 'arn:${Partition}:deepcomposer:${Region}:${Account}:model/${ModelId}';
        arn = arn.replace('${ModelId}', modelId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type composition to the statement
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     *
     * @param compositionId - Identifier for the compositionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComposition(compositionId, account, region, partition) {
        var arn = 'arn:${Partition}:deepcomposer:${Region}:${Account}:composition/${CompositionId}';
        arn = arn.replace('${CompositionId}', compositionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type audio to the statement
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     *
     * @param audioId - Identifier for the audioId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAudio(audioId, account, region, partition) {
        var arn = 'arn:${Partition}:deepcomposer:${Region}:${Account}:audio/${AudioId}';
        arn = arn.replace('${AudioId}', audioId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deepcomposer = Deepcomposer;
//# sourceMappingURL=data:application/json;base64,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