"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Devicefarm = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [devicefarm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdevicefarm.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Devicefarm extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [devicefarm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdevicefarm.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'devicefarm';
        this.accessLevelList = {
            "Write": [
                "CreateDevicePool",
                "CreateInstanceProfile",
                "CreateNetworkProfile",
                "CreateRemoteAccessSession",
                "CreateTestGridProject",
                "CreateTestGridUrl",
                "CreateUpload",
                "CreateVPCEConfiguration",
                "DeleteDevicePool",
                "DeleteInstanceProfile",
                "DeleteNetworkProfile",
                "DeleteProject",
                "DeleteRemoteAccessSession",
                "DeleteRun",
                "DeleteTestGridProject",
                "DeleteUpload",
                "DeleteVPCEConfiguration",
                "InstallToRemoteAccessSession",
                "PurchaseOffering",
                "RenewOffering",
                "ScheduleRun",
                "StopJob",
                "StopRemoteAccessSession",
                "StopRun",
                "UpdateDeviceInstance",
                "UpdateDevicePool",
                "UpdateInstanceProfile",
                "UpdateNetworkProfile",
                "UpdateProject",
                "UpdateTestGridProject",
                "UpdateUpload",
                "UpdateVPCEConfiguration"
            ],
            "Tagging": [
                "CreateProject",
                "TagResource",
                "UntagResource"
            ],
            "Read": [
                "GetAccountSettings",
                "GetDevice",
                "GetDeviceInstance",
                "GetDevicePool",
                "GetDevicePoolCompatibility",
                "GetInstanceProfile",
                "GetJob",
                "GetNetworkProfile",
                "GetOfferingStatus",
                "GetProject",
                "GetRemoteAccessSession",
                "GetRun",
                "GetSuite",
                "GetTest",
                "GetTestGridProject",
                "GetTestGridSession",
                "GetUpload",
                "GetVPCEConfiguration"
            ],
            "List": [
                "ListArtifacts",
                "ListDeviceInstances",
                "ListDevicePools",
                "ListDevices",
                "ListInstanceProfiles",
                "ListJobs",
                "ListNetworkProfiles",
                "ListOfferingPromotions",
                "ListOfferingTransactions",
                "ListOfferings",
                "ListProjects",
                "ListRemoteAccessSessions",
                "ListRuns",
                "ListSamples",
                "ListSuites",
                "ListTagsForResource",
                "ListTestGridProjects",
                "ListTestGridSessionActions",
                "ListTestGridSessionArtifacts",
                "ListTestGridSessions",
                "ListTests",
                "ListUniqueProblems",
                "ListUploads",
                "ListVPCEConfigurations"
            ]
        };
    }
    /**
     * Grants permission to create a device pool within a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateDevicePool.html
     */
    toCreateDevicePool() {
        this.to('devicefarm:CreateDevicePool');
        return this;
    }
    /**
     * Grants permission to create a device instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateInstanceProfile.html
     */
    toCreateInstanceProfile() {
        this.to('devicefarm:CreateInstanceProfile');
        return this;
    }
    /**
     * Grants permission to create a network profile within a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateNetworkProfile.html
     */
    toCreateNetworkProfile() {
        this.to('devicefarm:CreateNetworkProfile');
        return this;
    }
    /**
     * Grants permission to create a project for mobile testing
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        this.to('devicefarm:CreateProject');
        return this;
    }
    /**
     * Grants permission to start a remote access session to a device instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateRemoteAccessSession.html
     */
    toCreateRemoteAccessSession() {
        this.to('devicefarm:CreateRemoteAccessSession');
        return this;
    }
    /**
     * Grants permission to create a project for desktop testing
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateTestGridProject.html
     */
    toCreateTestGridProject() {
        this.to('devicefarm:CreateTestGridProject');
        return this;
    }
    /**
     * Grants permission to generate a new pre-signed url used to access our test grid service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateTestGridUrl.html
     */
    toCreateTestGridUrl() {
        this.to('devicefarm:CreateTestGridUrl');
        return this;
    }
    /**
     * Grants permission to upload a new file or app within a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateUpload.html
     */
    toCreateUpload() {
        this.to('devicefarm:CreateUpload');
        return this;
    }
    /**
     * Grants permission to create an Amazon Virtual Private Cloud (VPC) endpoint configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_CreateVPCEConfiguration.html
     */
    toCreateVPCEConfiguration() {
        this.to('devicefarm:CreateVPCEConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a user-generated device pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteDevicePool.html
     */
    toDeleteDevicePool() {
        this.to('devicefarm:DeleteDevicePool');
        return this;
    }
    /**
     * Grants permission to delete a user-generated instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteInstanceProfile.html
     */
    toDeleteInstanceProfile() {
        this.to('devicefarm:DeleteInstanceProfile');
        return this;
    }
    /**
     * Grants permission to delete a user-generated network profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/DeleteNetworkProfile.html
     */
    toDeleteNetworkProfile() {
        this.to('devicefarm:DeleteNetworkProfile');
        return this;
    }
    /**
     * Grants permission to delete a mobile testing project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        this.to('devicefarm:DeleteProject');
        return this;
    }
    /**
     * Grants permission to delete a completed remote access session and its results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteRemoteAccessSession.html
     */
    toDeleteRemoteAccessSession() {
        this.to('devicefarm:DeleteRemoteAccessSession');
        return this;
    }
    /**
     * Grants permission to delete a run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteRun.html
     */
    toDeleteRun() {
        this.to('devicefarm:DeleteRun');
        return this;
    }
    /**
     * Grants permission to delete a desktop testing project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteTestGridProject.html
     */
    toDeleteTestGridProject() {
        this.to('devicefarm:DeleteTestGridProject');
        return this;
    }
    /**
     * Grants permission to delete a user-uploaded file
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteUpload.html
     */
    toDeleteUpload() {
        this.to('devicefarm:DeleteUpload');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Virtual Private Cloud (VPC) endpoint configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeleteVPCEConfiguration.html
     */
    toDeleteVPCEConfiguration() {
        this.to('devicefarm:DeleteVPCEConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve the number of unmetered iOS and/or unmetered Android devices purchased by the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        this.to('devicefarm:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to retrieve the information of a unique device type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetDevice.html
     */
    toGetDevice() {
        this.to('devicefarm:GetDevice');
        return this;
    }
    /**
     * Grants permission to retireve the information of a device instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetDeviceInstance.html
     */
    toGetDeviceInstance() {
        this.to('devicefarm:GetDeviceInstance');
        return this;
    }
    /**
     * Grants permission to retireve the information of a device pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetDevicePool.html
     */
    toGetDevicePool() {
        this.to('devicefarm:GetDevicePool');
        return this;
    }
    /**
     * Grants permission to retrieve information about the compatibility of a test and/or app with a device pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetDevicePoolCompatibility.html
     */
    toGetDevicePoolCompatibility() {
        this.to('devicefarm:GetDevicePoolCompatibility');
        return this;
    }
    /**
     * Grants permission to retireve the information of an instance profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetInstanceProfile.html
     */
    toGetInstanceProfile() {
        this.to('devicefarm:GetInstanceProfile');
        return this;
    }
    /**
     * Grants permission to retireve the information of a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetJob.html
     */
    toGetJob() {
        this.to('devicefarm:GetJob');
        return this;
    }
    /**
     * Grants permission to retireve the information of a network profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetNetworkProfile.html
     */
    toGetNetworkProfile() {
        this.to('devicefarm:GetNetworkProfile');
        return this;
    }
    /**
     * Grants permission to retrieve the current status and future status of all offerings purchased by an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetOfferingStatus.html
     */
    toGetOfferingStatus() {
        this.to('devicefarm:GetOfferingStatus');
        return this;
    }
    /**
     * Grants permission to retrieve information about a mobile testing project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetProject.html
     */
    toGetProject() {
        this.to('devicefarm:GetProject');
        return this;
    }
    /**
     * Grants permission to retireve the link to a currently running remote access session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetRemoteAccessSession.html
     */
    toGetRemoteAccessSession() {
        this.to('devicefarm:GetRemoteAccessSession');
        return this;
    }
    /**
     * Grants permission to retireve the information of a run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetRun.html
     */
    toGetRun() {
        this.to('devicefarm:GetRun');
        return this;
    }
    /**
     * Grants permission to retireve the information of a testing suite
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetSuite.html
     */
    toGetSuite() {
        this.to('devicefarm:GetSuite');
        return this;
    }
    /**
     * Grants permission to retireve the information of a test case
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetTest.html
     */
    toGetTest() {
        this.to('devicefarm:GetTest');
        return this;
    }
    /**
     * Grants permission to retrieve information about a desktop testing project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetTestGridProject.html
     */
    toGetTestGridProject() {
        this.to('devicefarm:GetTestGridProject');
        return this;
    }
    /**
     * Grants permission to retireve the information of a test grid session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetTestGridSession.html
     */
    toGetTestGridSession() {
        this.to('devicefarm:GetTestGridSession');
        return this;
    }
    /**
     * Grants permission to retireve the information of an uploaded file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetUpload.html
     */
    toGetUpload() {
        this.to('devicefarm:GetUpload');
        return this;
    }
    /**
     * Grants permission to retireve the information of an Amazon Virtual Private Cloud (VPC) endpoint configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_GetVPCEConfiguration.html
     */
    toGetVPCEConfiguration() {
        this.to('devicefarm:GetVPCEConfiguration');
        return this;
    }
    /**
     * Grants permission to install an application to a device in a remote access session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_InstallToRemoteAccessSession.html
     */
    toInstallToRemoteAccessSession() {
        this.to('devicefarm:InstallToRemoteAccessSession');
        return this;
    }
    /**
     * Grants permission to list the artifacts in a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListArtifacts.html
     */
    toListArtifacts() {
        this.to('devicefarm:ListArtifacts');
        return this;
    }
    /**
     * Grants permission to list the information of device instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListDeviceInstances.html
     */
    toListDeviceInstances() {
        this.to('devicefarm:ListDeviceInstances');
        return this;
    }
    /**
     * Grants permission to list the information of device pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListDevicePools.html
     */
    toListDevicePools() {
        this.to('devicefarm:ListDevicePools');
        return this;
    }
    /**
     * Grants permission to list the information of unique device types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListDevices.html
     */
    toListDevices() {
        this.to('devicefarm:ListDevices');
        return this;
    }
    /**
     * Grants permission to list the information of device instance profiles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListInstanceProfiles.html
     */
    toListInstanceProfiles() {
        this.to('devicefarm:ListInstanceProfiles');
        return this;
    }
    /**
     * Grants permission to list the information of jobs within a run
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListJobs.html
     */
    toListJobs() {
        this.to('devicefarm:ListJobs');
        return this;
    }
    /**
     * Grants permission to list the information of network profiles within a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListNetworkProfiles.html
     */
    toListNetworkProfiles() {
        this.to('devicefarm:ListNetworkProfiles');
        return this;
    }
    /**
     * Grants permission to list the offering promotions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListOfferingPromotions.html
     */
    toListOfferingPromotions() {
        this.to('devicefarm:ListOfferingPromotions');
        return this;
    }
    /**
     * Grants permission to list all of the historical purchases, renewals, and system renewal transactions for an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListOfferingTransactions.html
     */
    toListOfferingTransactions() {
        this.to('devicefarm:ListOfferingTransactions');
        return this;
    }
    /**
     * Grants permission to list the products or offerings that the user can manage through the API
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListOfferings.html
     */
    toListOfferings() {
        this.to('devicefarm:ListOfferings');
        return this;
    }
    /**
     * Grants permission to list the information of mobile testing projects for an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListProjects.html
     */
    toListProjects() {
        this.to('devicefarm:ListProjects');
        return this;
    }
    /**
     * Grants permission to list the information of currently running remote access sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListRemoteAccessSessions.html
     */
    toListRemoteAccessSessions() {
        this.to('devicefarm:ListRemoteAccessSessions');
        return this;
    }
    /**
     * Grants permission to list the information of runs within a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListRuns.html
     */
    toListRuns() {
        this.to('devicefarm:ListRuns');
        return this;
    }
    /**
     * Grants permission to list the information of samples within a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListSamples.html
     */
    toListSamples() {
        this.to('devicefarm:ListSamples');
        return this;
    }
    /**
     * Grants permission to list the information of testing suites within a job
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListSuites.html
     */
    toListSuites() {
        this.to('devicefarm:ListSuites');
        return this;
    }
    /**
     * Grants permission to list the tags of a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('devicefarm:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the information of desktop testing projects for an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListTestGridProjects.html
     */
    toListTestGridProjects() {
        this.to('devicefarm:ListTestGridProjects');
        return this;
    }
    /**
     * Grants permission to list the session actions performed during a test grid session
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListTestGridSessionActions.html
     */
    toListTestGridSessionActions() {
        this.to('devicefarm:ListTestGridSessionActions');
        return this;
    }
    /**
     * Grants permission to list the artifacts generated by a test grid session
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListTestGridSessionArtifacts.html
     */
    toListTestGridSessionArtifacts() {
        this.to('devicefarm:ListTestGridSessionArtifacts');
        return this;
    }
    /**
     * Grants permission to list the sessions within a test grid project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListTestGridSessions.html
     */
    toListTestGridSessions() {
        this.to('devicefarm:ListTestGridSessions');
        return this;
    }
    /**
     * Grants permission to list the information of tests within a testing suite
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListTests.html
     */
    toListTests() {
        this.to('devicefarm:ListTests');
        return this;
    }
    /**
     * Grants permission to list the information of unique problems within a run
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListUniqueProblems.html
     */
    toListUniqueProblems() {
        this.to('devicefarm:ListUniqueProblems');
        return this;
    }
    /**
     * Grants permission to list the information of uploads within a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListUploads.html
     */
    toListUploads() {
        this.to('devicefarm:ListUploads');
        return this;
    }
    /**
     * Grants permission to list the information of Amazon Virtual Private Cloud (VPC) endpoint configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ListVPCEConfigurations.html
     */
    toListVPCEConfigurations() {
        this.to('devicefarm:ListVPCEConfigurations');
        return this;
    }
    /**
     * Grants permission to purchase offerings for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_PurchaseOffering.html
     */
    toPurchaseOffering() {
        this.to('devicefarm:PurchaseOffering');
        return this;
    }
    /**
     * Grants permission to set the quantity of devices to renew for an offering
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_RenewOffering.html
     */
    toRenewOffering() {
        this.to('devicefarm:RenewOffering');
        return this;
    }
    /**
     * Grants permission to schedule a run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_ScheduleRun.html
     */
    toScheduleRun() {
        this.to('devicefarm:ScheduleRun');
        return this;
    }
    /**
     * Grants permission to terminate a running job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_StopJob.html
     */
    toStopJob() {
        this.to('devicefarm:StopJob');
        return this;
    }
    /**
     * Grants permission to terminate a running remote access session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_StopRemoteAccessSession.html
     */
    toStopRemoteAccessSession() {
        this.to('devicefarm:StopRemoteAccessSession');
        return this;
    }
    /**
     * Grants permission to terminate a running test run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_StopRun.html
     */
    toStopRun() {
        this.to('devicefarm:StopRun');
        return this;
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('devicefarm:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('devicefarm:UntagResource');
        return this;
    }
    /**
     * Grants permission to modify an existing device instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateDeviceInstance.html
     */
    toUpdateDeviceInstance() {
        this.to('devicefarm:UpdateDeviceInstance');
        return this;
    }
    /**
     * Grants permission to modify an existing device pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateDevicePool.html
     */
    toUpdateDevicePool() {
        this.to('devicefarm:UpdateDevicePool');
        return this;
    }
    /**
     * Grants permission to modify an existing instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateInstanceProfile.html
     */
    toUpdateInstanceProfile() {
        this.to('devicefarm:UpdateInstanceProfile');
        return this;
    }
    /**
     * Grants permission to modify an existing network profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateNetworkProfile.html
     */
    toUpdateNetworkProfile() {
        this.to('devicefarm:UpdateNetworkProfile');
        return this;
    }
    /**
     * Grants permission to modify an existing mobile testing project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject() {
        this.to('devicefarm:UpdateProject');
        return this;
    }
    /**
     * Grants permission to modify an existing desktop testing project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateTestGridProject.html
     */
    toUpdateTestGridProject() {
        this.to('devicefarm:UpdateTestGridProject');
        return this;
    }
    /**
     * Grants permission to modify an existing upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateUpload.html
     */
    toUpdateUpload() {
        this.to('devicefarm:UpdateUpload');
        return this;
    }
    /**
     * Grants permission to modify an existing Amazon Virtual Private Cloud (VPC) endpoint configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_UpdateVPCEConfiguration.html
     */
    toUpdateVPCEConfiguration() {
        this.to('devicefarm:UpdateVPCEConfiguration');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Project.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:project:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type run to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Run.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRun(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:run:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Job.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:job:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type suite to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Suite.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSuite(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:suite:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type test to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Test.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTest(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:test:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type upload to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Upload.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUpload(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:upload:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type artifact to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Artifact.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onArtifact(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:artifact:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sample to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Sample.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSample(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:sample:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type networkprofile to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_NetworkProfile.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkprofile(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:networkprofile:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deviceinstance to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DeviceInstance.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeviceinstance(resourceId, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}::deviceinstance:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type session to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_RemoteAccessSession.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSession(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:session:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type devicepool to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_DevicePool.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevicepool(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:devicepool:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_Device.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevice(resourceId, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}::device:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instanceprofile to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_InstanceProfile.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstanceprofile(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:instanceprofile:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpceconfiguration to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_VPCEConfiguration.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpceconfiguration(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:vpceconfiguration:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type testgrid-project to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_TestGridProject.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestgridProject(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:testgrid-project:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type testgrid-session to the statement
     *
     * https://docs.aws.amazon.com/devicefarm/latest/APIReference/API_TestGridSession.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestgridSession(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:devicefarm:${Region}:${Account}:testgrid-session:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Devicefarm = Devicefarm;
//# sourceMappingURL=data:application/json;base64,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