"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Shield = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [shield](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsshield.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Shield extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [shield](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsshield.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'shield';
        this.accessLevelList = {
            "Write": [
                "AssociateDRTLogBucket",
                "AssociateDRTRole",
                "CreateProtection",
                "CreateSubscription",
                "DeleteProtection",
                "DeleteSubscription",
                "DisassociateDRTLogBucket",
                "DisassociateDRTRole",
                "UpdateEmergencyContactSettings",
                "UpdateSubscription"
            ],
            "Read": [
                "DescribeAttack",
                "DescribeDRTAccess",
                "DescribeEmergencyContactSettings",
                "DescribeProtection",
                "DescribeSubscription",
                "GetSubscriptionState"
            ],
            "List": [
                "ListAttacks",
                "ListProtections"
            ]
        };
    }
    /**
     * Authorizes the DDoS Response team to access the specified Amazon S3 bucket containing your flow logs
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetBucketPolicy
     * - s3:PutBucketPolicy
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTLogBucket.html
     */
    toAssociateDRTLogBucket() {
        this.to('shield:AssociateDRTLogBucket');
        return this;
    }
    /**
     * Authorizes the DDoS Response team using the specified role, to access your AWS account to assist with DDoS attack mitigation during potential attacks
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTRole.html
     */
    toAssociateDRTRole() {
        this.to('shield:AssociateDRTRole');
        return this;
    }
    /**
     * Activate DDoS protection service for a given resource ARN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateProtection.html
     */
    toCreateProtection() {
        this.to('shield:CreateProtection');
        return this;
    }
    /**
     * Activate subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateSubscription.html
     */
    toCreateSubscription() {
        this.to('shield:CreateSubscription');
        return this;
    }
    /**
     * Delete an existing protection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteProtection.html
     */
    toDeleteProtection() {
        this.to('shield:DeleteProtection');
        return this;
    }
    /**
     * Deactivate subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteSubscription.html
     */
    toDeleteSubscription() {
        this.to('shield:DeleteSubscription');
        return this;
    }
    /**
     * Get attack details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeAttack.html
     */
    toDescribeAttack() {
        this.to('shield:DescribeAttack');
        return this;
    }
    /**
     * Returns the current role and list of Amazon S3 log buckets used by the DDoS Response team to access your AWS account while assisting with attack mitigation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeDRTAccess.html
     */
    toDescribeDRTAccess() {
        this.to('shield:DescribeDRTAccess');
        return this;
    }
    /**
     * Lists the email addresses that the DRT can use to contact you during a suspected attack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeEmergencyContactSettings.html
     */
    toDescribeEmergencyContactSettings() {
        this.to('shield:DescribeEmergencyContactSettings');
        return this;
    }
    /**
     * Get protection details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeProtection.html
     */
    toDescribeProtection() {
        this.to('shield:DescribeProtection');
        return this;
    }
    /**
     * Get subscription details, such as start time
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeSubscription.html
     */
    toDescribeSubscription() {
        this.to('shield:DescribeSubscription');
        return this;
    }
    /**
     * Removes the DDoS Response team's access to the specified Amazon S3 bucket containing your flow logs
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteBucketPolicy
     * - s3:GetBucketPolicy
     * - s3:PutBucketPolicy
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTLogBucket.html
     */
    toDisassociateDRTLogBucket() {
        this.to('shield:DisassociateDRTLogBucket');
        return this;
    }
    /**
     * Removes the DDoS Response team's access to your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTRole.html
     */
    toDisassociateDRTRole() {
        this.to('shield:DisassociateDRTRole');
        return this;
    }
    /**
     * Get subscription state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_GetSubscriptionState.html
     */
    toGetSubscriptionState() {
        this.to('shield:GetSubscriptionState');
        return this;
    }
    /**
     * List all existing attacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListAttacks.html
     */
    toListAttacks() {
        this.to('shield:ListAttacks');
        return this;
    }
    /**
     * List all existing protections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListProtections.html
     */
    toListProtections() {
        this.to('shield:ListProtections');
        return this;
    }
    /**
     * Updates the details of the list of email addresses that the DRT can use to contact you during a suspected attack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateEmergencyContactSettings.html
     */
    toUpdateEmergencyContactSettings() {
        this.to('shield:UpdateEmergencyContactSettings');
        return this;
    }
    /**
     * Updates the details of an existing subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateSubscription.html
     */
    toUpdateSubscription() {
        this.to('shield:UpdateSubscription');
        return this;
    }
    /**
     * Adds a resource of type attack to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AttackDetail.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAttack(id, account, partition) {
        var arn = 'arn:${Partition}:shield::${Account}:attack/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type protection to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_Protection.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProtection(id, account, partition) {
        var arn = 'arn:${Partition}:shield::${Account}:protection/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Shield = Shield;
//# sourceMappingURL=data:application/json;base64,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