"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * Adds "principal" functionality to the Policy Statement
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        this.useNotPrincipals = false;
        this.principals = {};
    }
    /**
     * Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        if (this.hasPrincipals()) {
            this.explicitAllow = true;
        }
        const mode = this.useNotPrincipals ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.principals;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyPrincipals() { }
    /**
     * Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     */
    notPrincipals() {
        this.useNotPrincipals = true;
        return this;
    }
    /**
     * Checks weather a principal was applied to the policy
     */
    hasPrincipals() {
        return Object.keys(this.principals).length > 0;
    }
    /**
     * Adds a principal to the statement
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**
     * @param principal The principal string
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.principals)) {
            this.principals[prefix] = [];
        }
        this.principals[prefix].push(principal);
        return this;
    }
    /**
     * Adds any principal to the statement
     *
     * @param arn The ARN of the principal
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * Adds an account principal to the statement
     *
     * @param account ID of the AWS account
     */
    forAccount(account) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:root`);
    }
    /**
     * Adds a [federated](https://aws.amazon.com/identity/federation/) (web identity) principal to the statement
     *
     * @param provider ID of the AWS account
     */
    forFederated(provider) {
        return this.addPrincipal(PrincipalType.FEDERATED, provider);
    }
    /**
     * Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * Adds a federated [Amazon](https://login.amazon.com/) principal to the statement
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * Adds a canonical user principal to the statement
     *
     * @param userID The user ID
     *
     * You can [find the canonical user ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html#FindingCanonicalId) for your AWS account in the AWS Management Console. The canonical user ID for an AWS account is specific to the account. You can retrieve the canonical user ID for your AWS account as either the root user or an IAM user.
     */
    forCanonicalUser(userID) {
        return this.addPrincipal(PrincipalType.CANONICAL_USER, userID);
    }
    /**
     * Adds federated SAML principal to the statement
     *
     * @param account ID of the AWS account
     * @param providerName Name of the SAML provider
     */
    forSaml(account, providerName) {
        return this.forFederated(`arn:aws:iam::${account}:saml-provider/${providerName}`);
    }
    /**
     * Adds an IAM user principal to the statement
     *
     * @param account ID of the AWS account
     * @param user Name of the IAM user
     */
    forUser(account, user) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:user/${user}`);
    }
    /**
     * Adds an IAM role principal to the statement
     *
     * @param account ID of the AWS account
     * @param role Name of the IAM role
     */
    forRole(account, role) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:role/${role}`);
    }
    /**
     * Adds a specific assumed role session principal to the statement
     *
     * @param account ID of the AWS account
     * @param roleName Name of the IAM role
     * @param sessionName Name of the session. You cannot use a wildcard (`*`) to mean *all sessions*. Principals must always name a specific session
     */
    forAssumedRoleSession(account, roleName, sessionName) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:sts::${account}:assumed-role/${roleName}/${sessionName}`);
    }
    /**
     * Adds a service principal to the statement
     *
     * @param service Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`
     *
     * The service principal is defined by the service. To learn the service principal for a service, see the documentation for that service. For some services, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html) and look for the services that have **Yes** in the **Service-Linked Role** column. Choose a **Yes** with a link to view the service-linked role documentation for that service. View the **Service-Linked Role Permissions** section for that service to view the service principal.
     */
    forService(service) {
        return this.addPrincipal(PrincipalType.SERVICE, service);
    }
    /**
     * Grants public asses
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
//# sourceMappingURL=data:application/json;base64,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