import click
import os
import sys

from apolo_flow.cli.utils import wrap_async
from apolo_flow.types import LocalPath

from .root import Root


CFG_FILE = {"bash": LocalPath("~/.bashrc"), "zsh": LocalPath("~/.zshrc")}
SOURCE_CMD = {"bash": "bash_source", "zsh": "zsh_source"}

ACTIVATION_TEMPLATE = 'eval "$(_APOLO_FLOW_COMPLETE={cmd} {exe})"'


@click.group()
def completion() -> None:
    """
    Output shell completion code.
    """


@click.command()
@click.argument("shell", type=click.Choice(["bash", "zsh"]))
@wrap_async()
async def generate(root: Root, shell: str) -> None:
    """
    Provide instruction for shell completion generation.
    """
    root.console.print(f"Push the following line into your {CFG_FILE[shell]}")
    root.console.print(
        ACTIVATION_TEMPLATE.format(cmd=SOURCE_CMD[shell], exe=sys.argv[0])
    )


@click.command()
@click.argument("shell", type=click.Choice(["bash", "zsh"]))
@wrap_async()
async def patch(root: Root, shell: str) -> None:
    """
    Automatically patch shell configuration profile to enable completion
    """
    GENERATED_START = (
        b"# Start of generated by apolo-flow. Please do not edit this comment.\n"
    )
    GENERATED_END = (
        b"\n# End of generated by apolo-flow. Please do not edit this comment.\n"
    )

    profile_file = CFG_FILE[shell].expanduser()

    code = (
        GENERATED_START
        + os.fsencode(
            ACTIVATION_TEMPLATE.format(cmd=SOURCE_CMD[shell], exe=sys.argv[0])
        )
        + GENERATED_END
    )

    try:
        with profile_file.open("rb+") as profile:
            content = profile.read()
    except FileNotFoundError:
        content = b""

    start = content.find(GENERATED_START)
    if start != -1:
        end = content.find(GENERATED_END)
        if end == -1:
            raise click.ClickException(
                f"Malformed guarding comments. Please edit {profile_file} manually"
            )
        content = content[:start] + code + content[end + len(GENERATED_END) :]
    else:
        if content != b"":
            content += b"\n" + code
        else:
            content = code

    with profile_file.open("wb+") as profile:
        profile.write(content)
    root.console.print(f"Added completion configuration into '{profile_file}'")


completion.add_command(generate)
completion.add_command(patch)
