"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const crypto = require("crypto");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Hook into synthesis before it occurs and make any final adjustments.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi) {
            // Ignore IRestApi that are imported
            /*
             * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
             * Quoting from CloudFormation's docs - "If you create an AWS::ApiGateway::RestApi resource and its methods (using AWS::ApiGateway::Method) in
             * the same template as your deployment, the deployment must depend on the RestApi's methods. To create a dependency, add a DependsOn attribute
             * to the deployment. If you don't, AWS CloudFormation creates the deployment right after it creates the RestApi resource that doesn't contain
             * any methods, and AWS CloudFormation encounters the following error: The REST API doesn't contain any methods."
             */
            /*
             * Adding a dependency between LatestDeployment and Method construct, using ConstructNode.addDependencies(), creates additional dependencies
             * between AWS::ApiGateway::Deployment and the AWS::Lambda::Permission nodes (children under Method), causing cyclic dependency errors. Hence,
             * falling back to declaring dependencies between the underlying CfnResources.
             */
            this.api.methods.map(m => m.node.defaultChild).forEach(m => this.resource.addDependsOn(m));
        }
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = core_1.Stack.of(this).getLogicalId(this);
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    /**
     * Hooks into synthesis to calculate a logical ID that hashes all the components
     * add via `addToLogicalId`.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            this.addToLogicalId(core_1.Stack.of(this).resolve(cfnRestApiCF));
        }
        const stack = core_1.Stack.of(this);
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (this.hashComponents.length > 0) {
            const md5 = crypto.createHash('md5');
            this.hashComponents.map(x => stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            this.overrideLogicalId(this.originalLogicalId + md5.digest('hex'));
        }
        super.prepare();
    }
}
//# sourceMappingURL=data:application/json;base64,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