"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImagePipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const iam = require("aws-cdk-lib/aws-iam");
const imagebuilder = require("aws-cdk-lib/aws-imagebuilder");
const kms = require("aws-cdk-lib/aws-kms");
const sns = require("aws-cdk-lib/aws-sns");
const subscriptions = require("aws-cdk-lib/aws-sns-subscriptions");
const constructs_1 = require("constructs");
class ImagePipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let infrastructureConfig;
        let imageRecipe;
        this.imageRecipeComponents = [];
        // Construct code below
        const kmsKey = kms.Key.fromLookup(this, 'KmsKeyLookup', {
            aliasName: props.kmsKeyAlias,
        });
        const topic = new sns.Topic(this, 'ImageBuilderTopic', {
            displayName: 'Image Builder Notify',
            masterKey: kmsKey,
        });
        if (props.email != null) {
            topic.addSubscription(new subscriptions.EmailSubscription(props.email));
        }
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'IAM role used as part of an Image Builder pipeline',
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'));
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'));
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        if (typeof props.additionalPolicies !== 'undefined' && props.additionalPolicies.length >= 1) {
            for (const policy of props.additionalPolicies) {
                role.addManagedPolicy(policy);
            }
        }
        const profile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: props.profileName,
        });
        if (props.securityGroups == null || props.subnetId == null) {
            infrastructureConfig = new imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: props.profileName,
                name: props.infraConfigName,
                description: 'Example Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
            });
        }
        else {
            infrastructureConfig = new imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: props.profileName,
                name: props.infraConfigName,
                description: 'Example Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
                securityGroupIds: props.securityGroups,
                subnetId: props.subnetId,
            });
        }
        infrastructureConfig.addDependency(profile);
        /**
         * Image recipe configuration
         * UserData and EBS volume config are set to nothing if they are not given
         */
        const userDataOverrideScript = props.userDataScript ?? '';
        const ebsVolumeConfig = props.ebsVolumeConfiguration ?? {};
        imageRecipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            components: [],
            name: props.imageRecipe,
            parentImage: props.parentImage,
            version: props.imageRecipeVersion ?? '0.0.1',
            additionalInstanceConfiguration: {
                userDataOverride: userDataOverrideScript,
            },
            blockDeviceMappings: [{
                    ebs: ebsVolumeConfig,
                }],
        });
        props.componentDocuments.forEach((document, index) => {
            let component = new imagebuilder.CfnComponent(this, props.componentNames[index], {
                name: props.componentNames[index],
                platform: props.platform ?? 'Linux',
                version: props.componentVersions[index] ?? '0.0.1',
                data: fs_1.readFileSync(document).toString(),
            });
            // add the component to the Image Recipe
            this.imageRecipeComponents.push({ componentArn: component.attrArn });
            imageRecipe.components = this.imageRecipeComponents;
        });
        new imagebuilder.CfnImagePipeline(this, 'ImagePipeline', {
            infrastructureConfigurationArn: infrastructureConfig.attrArn,
            name: props.pipelineName,
            description: 'A sample image pipeline',
            imageRecipeArn: imageRecipe.attrArn,
        });
    }
}
exports.ImagePipeline = ImagePipeline;
_a = JSII_RTTI_SYMBOL_1;
ImagePipeline[_a] = { fqn: "cdk-image-pipeline.ImagePipeline", version: "0.1.49" };
//# sourceMappingURL=data:application/json;base64,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