# -*- coding: utf-8 -*-
import numpy as np
from numpy import ndarray
from numba import njit, prange
__cache = True


@njit(nogil=True, cache=__cache)
def shape_function_values(x, L):
    """
    Evaluates the shape functions at a point x in the range [-1, 1].
    """
    return np.array([[-0.5625 * x**3 + 0.5625 * x**2 + 0.0625 * x - 0.0625,
                      1.740234375 * x**7 - 1.423828125 * x**6 - 2.759765625 * x**5 + 2.37304687499999 * x**4 + 0.548828124999993 * x**3 - 0.474609375 * x**2 - 0.0292968749999963 * x + 0.0253906250000007,
                      1.740234375 * x**7 - 1.423828125 * x**6 - 2.759765625 * x**5 + 2.37304687499999 * x**4 + 0.548828124999993 * x**3 - 0.474609375 * x**2 - 0.0292968749999963 * x + 0.0253906250000007,
                      -0.5625 * x**3 + 0.5625 * x**2 + 0.0625 * x - 0.0625,
                      L * (-0.158203125 * x**7 + 0.158203125 * x**6 + 0.193359375 * x**5 - 0.193359375 * x**4 - 0.0371093749999995 * x**3 + 0.037109375 * x**2 + 0.00195312499999972 * x - 0.00195312500000006),
                      L * (0.158203125 * x**7 - 0.158203125 * x**6 - 0.193359375 * x**5 + 0.193359375 * x**4 + 0.0371093749999995 * x**3 - 0.037109375 * x**2 - 0.00195312499999972 * x + 0.00195312500000006)],
                     [1.6875 * (x - 1) * (1.0 * x - 0.333333333333333) * (x + 1),
                      4.271484375 * x**7 + 1.42382812500001 * x**6 - 10.916015625 * x**5 - 2.37304687500002 * x**4 + 9.017578125 * x**3 + 0.474609375000004 * x**2 - 2.37304687499999 * x + 0.474609375000001,
                      4.271484375 * x**7 + 1.42382812500001 * x**6 - 10.916015625 * x**5 - 2.37304687500002 * x**4 + 9.017578125 * x**3 + 0.474609375000004 * x**2 - 2.37304687499999 * x + 0.474609375000001,
                      1.6875 * (x - 1) * (1.0 * x - 0.333333333333333) * (x + 1),
                      L * (-1.423828125 * x**7 + 0.474609374999997 * x**6 + 3.005859375 * x**5 - 1.00195312499999 * x**4 - 1.74023437499999 * x**3 + 0.580078124999999 * x**2 + 0.158203124999997 * x - 0.0527343750000006),
                      L * (1.423828125 * x**7 - 0.474609374999997 * x**6 - 3.005859375 * x**5 + 1.00195312499999 * x**4 + 1.74023437499999 * x**3 - 0.580078124999999 * x**2 - 0.158203124999997 * x + 0.0527343750000006)],
                     [-1.6875 * (x - 1) * (1.0 * x + 0.333333333333333) * (x + 1),
                      -4.271484375 * x**7 + 1.42382812499999 * x**6 + 10.916015625 * x**5 - 2.37304687499998 * x**4 - 9.01757812499999 * x**3 + 0.474609374999996 * x**2 + 2.37304687499999 * x + 0.474609374999998,
                      -4.271484375 * x**7 + 1.42382812499999 * x**6 + 10.916015625 * x**5 - 2.37304687499998 * x**4 - 9.01757812499999 * x**3 + 0.474609374999996 * x**2 + 2.37304687499999 * x + 0.474609374999998,
                      -1.6875 * (x - 1) * (1.0 * x + 0.333333333333333) * (x + 1),
                      L * (-1.423828125 * x**7 - 0.474609375000002 * x**6 + 3.005859375 * x**5 + 1.001953125 * x**4 - 1.740234375 * x**3 - 0.580078125000001 * x**2 + 0.158203124999999 * x + 0.0527343749999997),
                      L * (1.423828125 * x**7 + 0.474609375000002 * x**6 - 3.005859375 * x**5 - 1.001953125 * x**4 + 1.740234375 * x**3 + 0.580078125000001 * x**2 - 0.158203124999999 * x - 0.0527343749999997)],
                     [0.5625 * (1.0 * x - 0.333333333333333) * (1.0 * x + 0.333333333333333) * (x + 1),
                      -1.740234375 * x**7 - 1.423828125 * x**6 + 2.759765625 * x**5 + 2.373046875 * x**4 - 0.548828125000001 * x**3 - 0.474609375000001 * x**2 + 0.0292968749999998 * x + 0.0253906249999998,
                      -1.740234375 * x**7 - 1.423828125 * x**6 + 2.759765625 * x**5 + 2.373046875 * x**4 - 0.548828125000001 * x**3 - 0.474609375000001 * x**2 + 0.0292968749999998 * x + 0.0253906249999998,
                      0.5625 * (1.0 * x - 0.333333333333333) * (1.0 * x + 0.333333333333333) * (x + 1),
                      L * (-0.158203125 * x**7 - 0.158203125 * x**6 + 0.193359375 * x**5 + 0.193359375 * x**4 - 0.0371093750000001 * x**3 - 0.037109375 * x**2 + 0.001953125 * x + 0.001953125),
                      L * (0.158203125 * x**7 + 0.158203125 * x**6 - 0.193359375 * x**5 - 0.193359375 * x**4 + 0.0371093750000001 * x**3 + 0.037109375 * x**2 - 0.001953125 * x - 0.001953125)]])


@njit(nogil=True, cache=__cache)
def shape_function_derivatives_1(x, L):
    """
    Evaluates the first derivatives of the shape
    functions at a point x in the range [-1, 1].
    """
    return np.array([[-1.6875 * x**2 + 1.125 * x + 0.0625,
                      12.181640625 * x**6 - 8.54296874999998 * x**5 - 13.798828125 * x**4 + 9.49218749999998 * x**3 + 1.64648437499998 * x**2 - 0.949218749999999 * x - 0.0292968749999963,
                      12.181640625 * x**6 - 8.54296874999998 * x**5 - 13.798828125 * x**4 + 9.49218749999998 * x**3 + 1.64648437499998 * x**2 - 0.949218749999999 * x - 0.0292968749999963,
                      -1.6875 * x**2 + 1.125 * x + 0.0625,
                      L * (-1.107421875 * x**6 + 0.949218749999998 * x**5 + 0.966796874999998 * x**4 - 0.773437499999998 * x**3 - 0.111328124999999 * x**2 + 0.0742187499999999 * x + 0.00195312499999972),
                      L * (1.107421875 * x**6 - 0.949218749999998 * x**5 - 0.966796874999998 * x**4 + 0.773437499999998 * x**3 + 0.111328124999999 * x**2 - 0.0742187499999999 * x - 0.00195312499999972)],
                     [5.0625 * x**2 - 1.125 * x - 1.6875,
                      29.900390625 * x**6 + 8.54296875000005 * x**5 - 54.580078125 * x**4 - 9.49218750000007 * x**3 + 27.052734375 * x**2 + 0.949218750000009 * x - 2.37304687499999,
                      29.900390625 * x**6 + 8.54296875000005 * x**5 - 54.580078125 * x**4 - 9.49218750000007 * x**3 + 27.052734375 * x**2 + 0.949218750000009 * x - 2.37304687499999,
                      5.0625 * x**2 - 1.125 * x - 1.6875,
                      L * (-9.96679687499999 * x**6 + 2.84765624999998 * x**5 + 15.029296875 * x**4 - 4.00781249999998 * x**3 - 5.22070312499998 * x**2 + 1.16015625 * x + 0.158203124999997),
                      L * (9.96679687499999 * x**6 - 2.84765624999998 * x**5 - 15.029296875 * x**4 + 4.00781249999998 * x**3 + 5.22070312499998 * x**2 - 1.16015625 * x - 0.158203124999997)],
                     [-5.0625 * x**2 - 1.125 * x + 1.6875,
                      -29.900390625 * x**6 + 8.54296874999993 * x**5 + 54.580078125 * x**4 - 9.49218749999993 * x**3 - 27.052734375 * x**2 + 0.949218749999992 * x + 2.37304687499999,
                      -29.900390625 * x**6 + 8.54296874999993 * x**5 + 54.580078125 * x**4 - 9.49218749999993 * x**3 - 27.052734375 * x**2 + 0.949218749999992 * x + 2.37304687499999,
                      -5.0625 * x**2 - 1.125 * x + 1.6875,
                      L * (-9.96679687500001 * x**6 - 2.84765625000001 * x**5 + 15.029296875 * x**4 + 4.00781250000002 * x**3 - 5.220703125 * x**2 - 1.16015625 * x + 0.158203124999999),
                      L * (9.96679687500001 * x**6 + 2.84765625000001 * x**5 - 15.029296875 * x**4 - 4.00781250000002 * x**3 + 5.220703125 * x**2 + 1.16015625 * x - 0.158203124999999)],
                     [1.6875 * x**2 + 1.125 * x - 0.0625,
                      -12.181640625 * x**6 - 8.54296875000001 * x**5 + 13.798828125 * x**4 + 9.49218750000001 * x**3 - 1.646484375 * x**2 - 0.949218750000001 * x + 0.0292968749999998,
                      -12.181640625 * x**6 - 8.54296875000001 * x**5 + 13.798828125 * x**4 + 9.49218750000001 * x**3 - 1.646484375 * x**2 - 0.949218750000001 * x + 0.0292968749999998,
                      1.6875 * x**2 + 1.125 * x - 0.0625,
                      L * (-1.107421875 * x**6 - 0.94921875 * x**5 + 0.966796875 * x**4 + 0.7734375 * x**3 - 0.111328125 * x**2 - 0.0742187500000001 * x + 0.001953125),
                      L * (1.107421875 * x**6 + 0.94921875 * x**5 - 0.966796875 * x**4 - 0.7734375 * x**3 + 0.111328125 * x**2 + 0.0742187500000001 * x - 0.001953125)]])


@njit(nogil=True, cache=__cache)
def shape_function_derivatives_2(x, L):
    """
    Evaluates the second derivatives of the shape
    functions at a point x in the range [-1, 1].
    """
    return np.array([[1.125 - 3.375 * x,
                      73.08984375 * x**5 - 42.7148437499999 * x**4 - 55.1953124999999 * x**3 + 28.4765624999999 * x**2 + 3.29296874999996 * x - 0.949218749999999,
                      73.08984375 * x**5 - 42.7148437499999 * x**4 - 55.1953124999999 * x**3 + 28.4765624999999 * x**2 + 3.29296874999996 * x - 0.949218749999999,
                      1.125 - 3.375 * x,
                      L * (-6.64453125 * x**5 + 4.74609374999999 * x**4 + 3.86718749999999 * x**3 - 2.3203125 * x**2 - 0.222656249999997 * x + 0.0742187499999999),
                      L * (6.64453125 * x**5 - 4.74609374999999 * x**4 - 3.86718749999999 * x**3 + 2.3203125 * x**2 + 0.222656249999997 * x - 0.0742187499999999)],
                     [10.125 * x - 1.125,
                      179.40234375 * x**5 + 42.7148437500003 * x**4 - 218.3203125 * x**3 - 28.4765625000002 * x**2 + 54.10546875 * x + 0.949218750000009,
                      179.40234375 * x**5 + 42.7148437500003 * x**4 - 218.3203125 * x**3 - 28.4765625000002 * x**2 + 54.10546875 * x + 0.949218750000009,
                      10.125 * x - 1.125,
                      L * (-59.80078125 * x**5 + 14.2382812499999 * x**4 + 60.1171874999999 * x**3 - 12.0234374999999 * x**2 - 10.44140625 * x + 1.16015625),
                      L * (59.80078125 * x**5 - 14.2382812499999 * x**4 - 60.1171874999999 * x**3 + 12.0234374999999 * x**2 + 10.44140625 * x - 1.16015625)],
                     [-10.125 * x - 1.125,
                      -179.40234375 * x**5 + 42.7148437499997 * x**4 + 218.3203125 * x**3 - 28.4765624999998 * x**2 - 54.1054687499999 * x + 0.949218749999992,
                      -179.40234375 * x**5 + 42.7148437499997 * x**4 + 218.3203125 * x**3 - 28.4765624999998 * x**2 - 54.1054687499999 * x + 0.949218749999992,
                      -10.125 * x - 1.125,
                      L * (-59.80078125 * x**5 - 14.2382812500001 * x**4 + 60.1171875 * x**3 + 12.0234375 * x**2 - 10.44140625 * x - 1.16015625),
                      L * (59.80078125 * x**5 + 14.2382812500001 * x**4 - 60.1171875 * x**3 - 12.0234375 * x**2 + 10.44140625 * x + 1.16015625)],
                     [3.375 * x + 1.125,
                      -73.08984375 * x**5 - 42.71484375 * x**4 + 55.1953125 * x**3 + 28.4765625 * x**2 - 3.29296875 * x - 0.949218750000001,
                      -73.08984375 * x**5 - 42.71484375 * x**4 + 55.1953125 * x**3 + 28.4765625 * x**2 - 3.29296875 * x - 0.949218750000001,
                      3.375 * x + 1.125,
                      L * (-6.64453125 * x**5 - 4.74609375 * x**4 + 3.8671875 * x**3 + 2.3203125 * x**2 - 0.22265625 * x - 0.0742187500000001),
                      L * (6.64453125 * x**5 + 4.74609375 * x**4 - 3.8671875 * x**3 - 2.3203125 * x**2 + 0.22265625 * x + 0.0742187500000001)]])


@njit(nogil=True, cache=__cache)
def shape_function_derivatives_3(x, L):
    """
    Evaluates the third derivatives of the shape
    functions at a point x in the range [-1, 1].
    """
    return np.array([[-3.37500000000000,
                      365.44921875 * x**4 - 170.859375 * x**3 - 165.5859375 * x**2 + 56.9531249999999 * x + 3.29296874999996,
                      365.44921875 * x**4 - 170.859375 * x**3 - 165.5859375 * x**2 + 56.9531249999999 * x + 3.29296874999996,
                      -3.37500000000000,
                      L * (-33.22265625 * x**4 + 18.984375 * x**3 + 11.6015625 * x**2 - 4.64062499999999 * x - 0.222656249999997),
                      L * (33.22265625 * x**4 - 18.984375 * x**3 - 11.6015625 * x**2 + 4.64062499999999 * x + 0.222656249999997)],
                     [10.1250000000000,
                      897.01171875 * x**4 + 170.859375000001 * x**3 - 654.9609375 * x**2 - 56.9531250000004 * x + 54.10546875,
                      897.01171875 * x**4 + 170.859375000001 * x**3 - 654.9609375 * x**2 - 56.9531250000004 * x + 54.10546875,
                      10.1250000000000,
                      L * (-299.00390625 * x**4 + 56.9531249999996 * x**3 + 180.3515625 * x**2 - 24.0468749999999 * x - 10.44140625),
                      L * (299.00390625 * x**4 - 56.9531249999996 * x**3 - 180.3515625 * x**2 + 24.0468749999999 * x + 10.44140625)],
                     [-10.1250000000000,
                      -897.01171875 * x**4 + 170.859374999999 * x**3 + 654.9609375 * x**2 - 56.9531249999996 * x - 54.1054687499999,
                      -897.01171875 * x**4 + 170.859374999999 * x**3 + 654.9609375 * x**2 - 56.9531249999996 * x - 54.1054687499999,
                      -10.1250000000000,
                      L * (-299.00390625 * x**4 - 56.9531250000002 * x**3 + 180.3515625 * x**2 + 24.0468750000001 * x - 10.44140625),
                      L * (299.00390625 * x**4 + 56.9531250000002 * x**3 - 180.3515625 * x**2 - 24.0468750000001 * x + 10.44140625)],
                     [3.37500000000000,
                      -365.44921875 * x**4 - 170.859375 * x**3 + 165.5859375 * x**2 + 56.9531250000001 * x - 3.29296875,
                      -365.44921875 * x**4 - 170.859375 * x**3 + 165.5859375 * x**2 + 56.9531250000001 * x - 3.29296875,
                      3.37500000000000,
                      L * (-33.22265625 * x**4 - 18.984375 * x**3 + 11.6015625 * x**2 + 4.640625 * x - 0.22265625),
                      L * (33.22265625 * x**4 + 18.984375 * x**3 - 11.6015625 * x**2 - 4.640625 * x + 0.22265625)]])


@njit(nogil=True, parallel=True, cache=__cache)
def shape_function_values_bulk(x: ndarray, L: ndarray):
    """
    Evaluates the shape functions at several points
    in the range [-1, 1].

    Parameters
    ----------
    x : 1d numpy float array
        The points of interest in the range [-1, -1]

    Returns
    -------
    numpy float array of shape (nE, nP, nNE, nDOF=6)
    """
    nP = x.shape[0]
    nE = L.shape[0]
    res = np.zeros((nE, nP, 4, 6), dtype=x.dtype)
    for iE in prange(nE):
        for iP in prange(nP):
            res[iE, iP] = shape_function_values(x[iP], L[iE])
    return res


@njit(nogil=True, cache=__cache)
def shape_function_derivatives(x, L):
    """
    Evaluates the derivatives of the shape
    functions at a point x in the range [-1, 1].

    Parameters
    ----------
    x : float
        The point of interest in the range [-1, -1]

    djac : float
        Determinant of the Jacobi matrix of local-global transformation
        between the master elment and the actual element.
        Default is 1.0.

    Returns
    -------
    numpy float array of shape (nNE, nDOF=6, 3)
    """
    res = np.zeros((4, 6, 3))
    res[:, :, 0] = shape_function_derivatives_1(x, L)
    res[:, :, 1] = shape_function_derivatives_2(x, L)
    res[:, :, 2] = shape_function_derivatives_3(x, L)
    return res


@njit(nogil=True, parallel=True, cache=__cache)
def shape_function_derivatives_bulk(x: ndarray, L: ndarray):
    """
    Evaluates the derivatives of the shape
    functions at several points in the range [-1, 1].

    Returns
    -------
    dshp (nE, nP, nNE, nDOF=6, 3)
    """
    nP = x.shape[0]
    nE = L.shape[0]
    res = np.zeros((nE, nP, 4, 6, 3), dtype=x.dtype)
    for iE in prange(nE):
        for iP in prange(nP):
            res[iE, iP] = shape_function_derivatives(x[iP], L[iE])
    return res
