# -*- coding: utf-8 -*-
import numpy as np
from numpy import ndarray
from numba import njit, prange
__cache = True


@njit(nogil=True, cache=__cache)
def shape_function_values(x, L):
    """
    Evaluates the shape functions at a point x in the range [-1, 1].
    """
    return np.array([[0.666666666666667 * x * (x - 1) * (1.0 * x - 0.5) * (1.0 * x + 0.5),
                      x**2 * (3.7037037037037 * x**7 - 3.25925925925926 * x**6 - 6.44444444444444 * x**5 + 5.77777777777778 *
                              x**4 + 2.52777777777778 * x**3 - 2.27777777777778 * x**2 - 0.287037037037037 * x + 0.259259259259259),
                      x**2 * (3.7037037037037 * x**7 - 3.25925925925926 * x**6 - 6.44444444444444 * x**5 + 5.77777777777778 *
                              x**4 + 2.52777777777778 * x**3 - 2.27777777777778 * x**2 - 0.287037037037037 * x + 0.259259259259259),
                      0.666666666666667 * x *
                      (x - 1) * (1.0 * x - 0.5) * (1.0 * x + 0.5),
                      -0.222222222222222 * L * x**2 *
                      (x - 1)**2 * (1.0 * x - 0.5)**2 *
                      (1.0 * x + 0.5)**2 * (x + 1),
                      0.222222222222222 * L * x**2 * (x - 1)**2 * (1.0 * x - 0.5)**2 * (1.0 * x + 0.5)**2 * (x + 1)],
                     [-2.66666666666667 * x * (x - 1) * (1.0 * x - 0.5) * (x + 1),
                      x**2 * (23.7037037037037 * x**7 - 4.74074074074074 * x**6 - 60.4444444444444 * x**5 + 14.2222222222222 *
                              x**4 + 49.7777777777778 * x**3 - 14.2222222222222 * x**2 - 13.037037037037 * x + 4.74074074074074),
                      x**2 * (23.7037037037037 * x**7 - 4.74074074074074 * x**6 - 60.4444444444444 * x**5 + 14.2222222222222 *
                              x**4 + 49.7777777777778 * x**3 - 14.2222222222222 * x**2 - 13.037037037037 * x + 4.74074074074074),
                      -2.66666666666667 * x *
                      (x - 1) * (1.0 * x - 0.5) * (x + 1),
                      -3.55555555555556 * L * x**2 *
                      (x - 1)**2 * (1.0 * x - 0.5)**2 *
                      (1.0 * x + 0.5) * (x + 1)**2,
                      3.55555555555556 * L * x**2 * (x - 1)**2 * (1.0 * x - 0.5)**2 * (1.0 * x + 0.5) * (x + 1)**2],
                     [4.0 * x**4 - 5.0 * x**2 + 1.0,
                      16.0 * x**8 - 40.0 * x**6 + 33.0 * x**4 - 10.0 * x**2 + 1.0,
                      16.0 * x**8 - 40.0 * x**6 + 33.0 * x**4 - 10.0 * x**2 + 1.0,
                      4.0 * x**4 - 5.0 * x**2 + 1.0,
                      L * x * (-8.0 * x**8 + 20.0 * x**6 -
                               16.5 * x**4 + 5.0 * x**2 - 0.5),
                      L * x * (8.0 * x**8 - 20.0 * x**6 + 16.5 * x**4 - 5.0 * x**2 + 0.5)],
                     [-2.66666666666667 * x * (x - 1) * (1.0 * x + 0.5) * (x + 1),
                      x**2 * (-23.7037037037037 * x**7 - 4.74074074074074 * x**6 + 60.4444444444444 * x**5 + 14.2222222222222 *
                              x**4 - 49.7777777777778 * x**3 - 14.2222222222222 * x**2 + 13.037037037037 * x + 4.74074074074074),
                      x**2 * (-23.7037037037037 * x**7 - 4.74074074074074 * x**6 + 60.4444444444444 * x**5 + 14.2222222222222 *
                              x**4 - 49.7777777777778 * x**3 - 14.2222222222222 * x**2 + 13.037037037037 * x + 4.74074074074074),
                      -2.66666666666667 * x *
                      (x - 1) * (1.0 * x + 0.5) * (x + 1),
                      -3.55555555555556 * L * x**2 *
                      (x - 1)**2 * (1.0 * x - 0.5) *
                      (1.0 * x + 0.5)**2 * (x + 1)**2,
                      3.55555555555556 * L * x**2 * (x - 1)**2 * (1.0 * x - 0.5) * (1.0 * x + 0.5)**2 * (x + 1)**2],
                     [0.666666666666667 * x * (1.0 * x - 0.5) * (1.0 * x + 0.5) * (x + 1),
                      x**2 * (-3.7037037037037 * x**7 - 3.25925925925926 * x**6 + 6.44444444444444 * x**5 + 5.77777777777778 *
                              x**4 - 2.52777777777778 * x**3 - 2.27777777777778 * x**2 + 0.287037037037037 * x + 0.259259259259259),
                      x**2 * (-3.7037037037037 * x**7 - 3.25925925925926 * x**6 + 6.44444444444444 * x**5 + 5.77777777777778 *
                              x**4 - 2.52777777777778 * x**3 - 2.27777777777778 * x**2 + 0.287037037037037 * x + 0.259259259259259),
                      0.666666666666667 * x *
                      (1.0 * x - 0.5) * (1.0 * x + 0.5) * (x + 1),
                      -0.222222222222222 * L * x**2 *
                      (x - 1) * (1.0 * x - 0.5)**2 *
                      (1.0 * x + 0.5)**2 * (x + 1)**2,
                      0.222222222222222 * L * x**2 * (x - 1) * (1.0 * x - 0.5)**2 * (1.0 * x + 0.5)**2 * (x + 1)**2]])


@njit(nogil=True, cache=__cache)
def shape_function_derivatives_1(x, L):
    """
    Evaluates the first derivatives of the shape
    functions at a point x in the range [-1, 1].
    """
    return np.array([[2.66666666666667 * x**3 - 2.0 * x**2 - 0.333333333333333 * x + 0.166666666666667,
                      x * (33.3333333333333 * x**7 - 26.0740740740741 * x**6 - 45.1111111111111 * x**5 + 34.6666666666667 *
                           x**4 + 12.6388888888889 * x**3 - 9.11111111111111 * x**2 - 0.861111111111111 * x + 0.518518518518519),
                      x * (33.3333333333333 * x**7 - 26.0740740740741 * x**6 - 45.1111111111111 * x**5 + 34.6666666666667 *
                           x**4 + 12.6388888888889 * x**3 - 9.11111111111111 * x**2 - 0.861111111111111 * x + 0.518518518518519),
                      2.66666666666667 * x**3 - 2.0 * x**2 - 0.333333333333333 * x + 0.166666666666667,
                      L * x * (-2.0 * x**7 + 1.77777777777778 * x**6 + 2.33333333333333 * x**5 - 2.0 *
                               x**4 - 0.625 * x**3 + 0.5 * x**2 + 0.0416666666666667 * x - 0.0277777777777778),
                      2.0 * L * x * (x - 1) * (1.0 * x - 0.5) * (1.0 * x + 0.5) * (1.0 * x**4 + 0.111111111111111 * x**3 - 0.805555555555556 * x**2 - 0.0277777777777778 * x + 0.0555555555555556)],
                     [-10.6666666666667 * x**3 + 4.0 * x**2 + 5.33333333333333 * x - 1.33333333333333,
                      x * (213.333333333333 * x**7 - 37.9259259259259 * x**6 - 423.111111111111 * x**5 + 85.3333333333333 *
                           x**4 + 248.888888888889 * x**3 - 56.8888888888889 * x**2 - 39.1111111111111 * x + 9.48148148148148),
                      x * (213.333333333333 * x**7 - 37.9259259259259 * x**6 - 423.111111111111 * x**5 + 85.3333333333333 *
                           x**4 + 248.888888888889 * x**3 - 56.8888888888889 * x**2 - 39.1111111111111 * x + 9.48148148148148),
                      -10.6666666666667 * x**3 + 4.0 * x**2 + 5.33333333333333 * x - 1.33333333333333,
                      -32.0 * L * x * (x - 1) * (1.0 * x - 0.5) * (x + 1) * (1.0 * x**4 + 0.0555555555555556 *
                                                                             x**3 - 0.722222222222222 * x**2 - 0.0555555555555556 * x + 0.0555555555555556),
                      32.0 * L * x * (x - 1) * (1.0 * x - 0.5) * (x + 1) * (1.0 * x**4 + 0.0555555555555556 * x**3 - 0.722222222222222 * x**2 - 0.0555555555555556 * x + 0.0555555555555556)],
                     [x * (16.0 * x**2 - 10.0),
                      x * (128.0 * x**6 - 240.0 * x**4 + 132.0 * x**2 - 20.0),
                      x * (128.0 * x**6 - 240.0 * x**4 + 132.0 * x**2 - 20.0),
                      x * (16.0 * x**2 - 10.0),
                      L * (-72.0 * x**8 + 140.0 * x**6 -
                           82.5 * x**4 + 15.0 * x**2 - 0.5),
                      L * (72.0 * x**8 - 140.0 * x**6 + 82.5 * x**4 - 15.0 * x**2 + 0.5)],
                     [-10.6666666666667 * x**3 - 4.0 * x**2 + 5.33333333333333 * x + 1.33333333333333,
                      x * (-213.333333333333 * x**7 - 37.9259259259259 * x**6 + 423.111111111111 * x**5 + 85.3333333333333 *
                           x**4 - 248.888888888889 * x**3 - 56.8888888888889 * x**2 + 39.1111111111111 * x + 9.48148148148148),
                      x * (-213.333333333333 * x**7 - 37.9259259259259 * x**6 + 423.111111111111 * x**5 + 85.3333333333333 *
                           x**4 - 248.888888888889 * x**3 - 56.8888888888889 * x**2 + 39.1111111111111 * x + 9.48148148148148),
                      -10.6666666666667 * x**3 - 4.0 * x**2 + 5.33333333333333 * x + 1.33333333333333,
                      -32.0 * L * x * (x - 1) * (1.0 * x + 0.5) * (x + 1) * (1.0 * x**4 - 0.0555555555555556 *
                                                                             x**3 - 0.722222222222222 * x**2 + 0.0555555555555556 * x + 0.0555555555555556),
                      32.0 * L * x * (x - 1) * (1.0 * x + 0.5) * (x + 1) * (1.0 * x**4 - 0.0555555555555556 * x**3 - 0.722222222222222 * x**2 + 0.0555555555555556 * x + 0.0555555555555556)],
                     [2.66666666666667 * x**3 + 2.0 * x**2 - 0.333333333333333 * x - 0.166666666666667,
                      x * (-33.3333333333333 * x**7 - 26.0740740740741 * x**6 + 45.1111111111111 * x**5 + 34.6666666666667 *
                           x**4 - 12.6388888888889 * x**3 - 9.11111111111111 * x**2 + 0.861111111111111 * x + 0.518518518518519),
                      x * (-33.3333333333333 * x**7 - 26.0740740740741 * x**6 + 45.1111111111111 * x**5 + 34.6666666666667 *
                           x**4 - 12.6388888888889 * x**3 - 9.11111111111111 * x**2 + 0.861111111111111 * x + 0.518518518518519),
                      2.66666666666667 * x**3 + 2.0 * x**2 - 0.333333333333333 * x - 0.166666666666667,
                      L * x * (-2.0 * x**7 - 1.77777777777778 * x**6 + 2.33333333333333 * x**5 + 2.0 *
                               x**4 - 0.625 * x**3 - 0.5 * x**2 + 0.0416666666666667 * x + 0.0277777777777778),
                      2.0 * L * x * (1.0 * x - 0.5) * (1.0 * x + 0.5) * (x + 1) * (1.0 * x**4 - 0.111111111111111 * x**3 - 0.805555555555556 * x**2 + 0.0277777777777778 * x + 0.0555555555555556)]])


@njit(nogil=True, cache=__cache)
def shape_function_derivatives_2(x, L):
    """
    Evaluates the second derivatives of the shape
    functions at a point x in the range [-1, 1].
    """
    return np.array([[8.0 * x**2 - 4.0 * x - 0.333333333333333,
                      266.666666666667 * x**7 - 182.518518518519 * x**6 - 270.666666666667 * x**5 + 173.333333333333 *
                      x**4 + 50.5555555555556 * x**3 - 27.3333333333333 *
                      x**2 - 1.72222222222222 * x + 0.518518518518518,
                      266.666666666667 * x**7 - 182.518518518519 * x**6 - 270.666666666667 * x**5 + 173.333333333333 *
                      x**4 + 50.5555555555556 * x**3 - 27.3333333333333 *
                      x**2 - 1.72222222222222 * x + 0.518518518518518,
                      8.0 * x**2 - 4.0 * x - 0.333333333333333,
                      L * (-16.0 * x**7 + 12.4444444444444 * x**6 + 14.0 * x**5 - 10.0 * x**4 -
                           2.5 * x**3 + 1.5 * x**2 + 0.0833333333333333 * x - 0.0277777777777778),
                      L * (16.0 * x**7 - 12.4444444444444 * x**6 - 14.0 * x**5 + 10.0 * x**4 + 2.5 * x**3 - 1.5 * x**2 - 0.0833333333333333 * x + 0.0277777777777778)],
                     [-32.0 * x**2 + 8.0 * x + 5.33333333333333,
                      1706.66666666667 * x**7 - 265.481481481481 * x**6 - 2538.66666666667 * x**5 + 426.666666666667 *
                      x**4 + 995.555555555555 * x**3 - 170.666666666667 *
                      x**2 - 78.2222222222222 * x + 9.48148148148148,
                      1706.66666666667 * x**7 - 265.481481481481 * x**6 - 2538.66666666667 * x**5 + 426.666666666667 *
                      x**4 + 995.555555555555 * x**3 - 170.666666666667 *
                      x**2 - 78.2222222222222 * x + 9.48148148148148,
                      -32.0 * x**2 + 8.0 * x + 5.33333333333333,
                      L * (-256.0 * x**7 + 99.5555555555555 * x**6 + 336.0 * x**5 - 120.0 * x**4 -
                           106.666666666667 * x**3 + 32.0 * x**2 + 5.33333333333333 * x - 0.888888888888889),
                      L * (256.0 * x**7 - 99.5555555555555 * x**6 - 336.0 * x**5 + 120.0 * x**4 + 106.666666666667 * x**3 - 32.0 * x**2 - 5.33333333333333 * x + 0.888888888888889)],
                     [48.0 * x**2 - 10.0,
                      896.0 * x**6 - 1200.0 * x**4 + 396.0 * x**2 - 20.0,
                      896.0 * x**6 - 1200.0 * x**4 + 396.0 * x**2 - 20.0,
                      48.0 * x**2 - 10.0,
                      L * x * (-576.0 * x**6 + 840.0 * x **
                               4 - 330.0 * x**2 + 30.0),
                      L * x * (576.0 * x**6 - 840.0 * x**4 + 330.0 * x**2 - 30.0)],
                     [-32.0 * x**2 - 8.0 * x + 5.33333333333333,
                      -1706.66666666667 * x**7 - 265.481481481482 * x**6 + 2538.66666666667 * x**5 + 426.666666666667 *
                      x**4 - 995.555555555555 * x**3 - 170.666666666667 *
                      x**2 + 78.2222222222222 * x + 9.48148148148148,
                      -1706.66666666667 * x**7 - 265.481481481482 * x**6 + 2538.66666666667 * x**5 + 426.666666666667 *
                      x**4 - 995.555555555555 * x**3 - 170.666666666667 *
                      x**2 + 78.2222222222222 * x + 9.48148148148148,
                      -32.0 * x**2 - 8.0 * x + 5.33333333333333,
                      L * (-256.0 * x**7 - 99.5555555555555 * x**6 + 336.0 * x**5 + 120.0 * x**4 -
                           106.666666666667 * x**3 - 32.0 * x**2 + 5.33333333333333 * x + 0.888888888888889),
                      L * (256.0 * x**7 + 99.5555555555555 * x**6 - 336.0 * x**5 - 120.0 * x**4 + 106.666666666667 * x**3 + 32.0 * x**2 - 5.33333333333333 * x - 0.888888888888889)],
                     [8.0 * x**2 + 4.0 * x - 0.333333333333333,
                      -266.666666666667 * x**7 - 182.518518518519 * x**6 + 270.666666666667 * x**5 + 173.333333333333 *
                      x**4 - 50.5555555555556 * x**3 - 27.3333333333333 *
                      x**2 + 1.72222222222222 * x + 0.518518518518518,
                      -266.666666666667 * x**7 - 182.518518518519 * x**6 + 270.666666666667 * x**5 + 173.333333333333 *
                      x**4 - 50.5555555555556 * x**3 - 27.3333333333333 *
                      x**2 + 1.72222222222222 * x + 0.518518518518518,
                      8.0 * x**2 + 4.0 * x - 0.333333333333333,
                      L * (-16.0 * x**7 - 12.4444444444444 * x**6 + 14.0 * x**5 + 10.0 * x**4 -
                           2.5 * x**3 - 1.5 * x**2 + 0.0833333333333333 * x + 0.0277777777777778),
                      L * (16.0 * x**7 + 12.4444444444444 * x**6 - 14.0 * x**5 - 10.0 * x**4 + 2.5 * x**3 + 1.5 * x**2 - 0.0833333333333333 * x - 0.0277777777777778)]])


@njit(nogil=True, cache=__cache)
def shape_function_derivatives_3(x, L):
    """
    Evaluates the third derivatives of the shape
    functions at a point x in the range [-1, 1].
    """
    return np.array([[16.0 * x - 4.0,
                      1866.66666666667 * x**6 - 1095.11111111111 * x**5 - 1353.33333333333 * x**4 +
                      693.333333333333 * x**3 + 151.666666666667 *
                      x**2 - 54.6666666666667 * x - 1.72222222222222,
                      1866.66666666667 * x**6 - 1095.11111111111 * x**5 - 1353.33333333333 * x**4 +
                      693.333333333333 * x**3 + 151.666666666667 *
                      x**2 - 54.6666666666667 * x - 1.72222222222222,
                      16.0 * x - 4.0,
                      L * (-112.0 * x**6 + 74.6666666666666 * x**5 + 70.0 * x**4 -
                           40.0 * x**3 - 7.5 * x**2 + 3.0 * x + 0.0833333333333333),
                      L * (112.0 * x**6 - 74.6666666666667 * x**5 - 70.0 * x**4 + 40.0 * x**3 + 7.5 * x**2 - 3.0 * x - 0.0833333333333333)],
                     [8.0 - 64.0 * x,
                      11946.6666666667 * x**6 - 1592.88888888889 * x**5 - 12693.3333333333 * x**4 +
                      1706.66666666667 * x**3 + 2986.66666666667 *
                      x**2 - 341.333333333333 * x - 78.2222222222222,
                      11946.6666666667 * x**6 - 1592.88888888889 * x**5 - 12693.3333333333 * x**4 +
                      1706.66666666667 * x**3 + 2986.66666666667 *
                      x**2 - 341.333333333333 * x - 78.2222222222222,
                      8.0 - 64.0 * x,
                      L * (-1792.0 * x**6 + 597.333333333333 * x**5 + 1680.0 * x **
                           4 - 480.0 * x**3 - 320.0 * x**2 + 64.0 * x + 5.33333333333333),
                      L * (1792.0 * x**6 - 597.333333333333 * x**5 - 1680.0 * x**4 + 480.0 * x**3 + 320.0 * x**2 - 64.0 * x - 5.33333333333333)],
                     [96.0 * x,
                      x * (5376.0 * x**4 - 4800.0 * x**2 + 792.0),
                      x * (5376.0 * x**4 - 4800.0 * x**2 + 792.0),
                      96.0 * x,
                      L * (-4032.0 * x**6 + 4200.0 *
                           x**4 - 990.0 * x**2 + 30.0),
                      L * (4032.0 * x**6 - 4200.0 * x**4 + 990.0 * x**2 - 30.0)],
                     [-64.0 * x - 8.0,
                      -11946.6666666667 * x**6 - 1592.88888888889 * x**5 + 12693.3333333333 * x**4 +
                      1706.66666666667 * x**3 - 2986.66666666667 *
                      x**2 - 341.333333333333 * x + 78.2222222222222,
                      -11946.6666666667 * x**6 - 1592.88888888889 * x**5 + 12693.3333333333 * x**4 +
                      1706.66666666667 * x**3 - 2986.66666666667 *
                      x**2 - 341.333333333333 * x + 78.2222222222222,
                      -64.0 * x - 8.0,
                      L * (-1792.0 * x**6 - 597.333333333333 * x**5 + 1680.0 * x **
                           4 + 480.0 * x**3 - 320.0 * x**2 - 64.0 * x + 5.33333333333333),
                      L * (1792.0 * x**6 + 597.333333333333 * x**5 - 1680.0 * x**4 - 480.0 * x**3 + 320.0 * x**2 + 64.0 * x - 5.33333333333333)],
                     [16.0 * x + 4.0,
                      -1866.66666666667 * x**6 - 1095.11111111111 * x**5 + 1353.33333333333 * x**4 +
                      693.333333333333 * x**3 - 151.666666666667 *
                      x**2 - 54.6666666666667 * x + 1.72222222222222,
                      -1866.66666666667 * x**6 - 1095.11111111111 * x**5 + 1353.33333333333 * x**4 +
                      693.333333333333 * x**3 - 151.666666666667 *
                      x**2 - 54.6666666666667 * x + 1.72222222222222,
                      16.0 * x + 4.0,
                      L * (-112.0 * x**6 - 74.6666666666666 * x**5 + 70.0 * x**4 +
                           40.0 * x**3 - 7.5 * x**2 - 3.0 * x + 0.0833333333333333),
                      L * (112.0 * x**6 + 74.6666666666666 * x**5 - 70.0 * x**4 - 40.0 * x**3 + 7.5 * x**2 + 3.0 * x - 0.0833333333333333)]])


@njit(nogil=True, parallel=True, cache=__cache)
def shape_function_values_bulk(x: ndarray, L: ndarray):
    """
    Evaluates the shape functions at several points
    in the range [-1, 1].

    Parameters
    ----------
    x : 1d numpy float array
        The points of interest in the range [-1, -1]

    Returns
    -------
    numpy float array of shape (nE, nP, nNE, nDOF=6)
    """
    nP = x.shape[0]
    nE = L.shape[0]
    res = np.zeros((nE, nP, 5, 6), dtype=x.dtype)
    for iE in prange(nE):
        for iP in prange(nP):
            res[iE, iP] = shape_function_values(x[iP], L[iE])
    return res


@njit(nogil=True, cache=__cache)
def shape_function_derivatives(x, L):
    """
    Evaluates the derivatives of the shape
    functions at a point x in the range [-1, 1].

    Parameters
    ----------
    x : float
        The point of interest in the range [-1, -1]

    djac : float
        Determinant of the Jacobi matrix of local-global transformation
        between the master elment and the actual element.
        Default is 1.0.

    Returns
    -------
    numpy float array of shape (nNE, nDOF=6, 3)
    """
    res = np.zeros((5, 6, 3))
    res[:, :, 0] = shape_function_derivatives_1(x, L)
    res[:, :, 1] = shape_function_derivatives_2(x, L)
    res[:, :, 2] = shape_function_derivatives_3(x, L)
    return res


@njit(nogil=True, parallel=True, cache=__cache)
def shape_function_derivatives_bulk(x: ndarray, L: ndarray):
    """
    Evaluates the derivatives of the shape
    functions at several points in the range [-1, 1].

    Returns
    -------
    dshp (nE, nP, nNE, nDOF=6, 3)
    """
    nP = x.shape[0]
    nE = L.shape[0]
    res = np.zeros((nE, nP, 5, 6, 3), dtype=x.dtype)
    for iE in prange(nE):
        for iP in prange(nP):
            res[iE, iP] = shape_function_derivatives(x[iP], L[iE])
    return res
