"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.QueueEncryption = void 0;
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing SQS queue provided an ARN
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1)
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const queueName = attrs.queueName || stack.parseArn(attrs.queueArn).resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${stack.region}.${stack.urlSuffix}/${stack.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = queueName.endsWith('.fifo') ? true : false;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,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