"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const sqs = require("../lib");
function testGrant(action, ...expectedActions) {
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue');
    const principal = new iam.User(stack, 'User');
    action(queue, principal);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': expectedActions,
                    'Effect': 'Allow',
                    'Resource': {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn',
                        ],
                    },
                },
            ],
            'Version': '2012-10-17',
        },
    }));
}
module.exports = {
    'default properties'(test) {
        const stack = new core_1.Stack();
        const q = new sqs.Queue(stack, 'Queue');
        test.deepEqual(q.fifo, false);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                },
            },
        });
        test.done();
    },
    'with a dead letter queue'(test) {
        const stack = new core_1.Stack();
        const dlq = new sqs.Queue(stack, 'DLQ');
        new sqs.Queue(stack, 'Queue', { deadLetterQueue: { queue: dlq, maxReceiveCount: 3 } });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'DLQ581697C4': {
                    'Type': 'AWS::SQS::Queue',
                },
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'RedrivePolicy': {
                            'deadLetterTargetArn': {
                                'Fn::GetAtt': [
                                    'DLQ581697C4',
                                    'Arn',
                                ],
                            },
                            'maxReceiveCount': 3,
                        },
                    },
                },
            },
        });
        test.done();
    },
    'message retention period must be between 1 minute to 14 days'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new sqs.Queue(stack, 'MyQueue', {
            retentionPeriod: core_1.Duration.seconds(30),
        }), /message retention period must be 60 seconds or more/);
        test.throws(() => new sqs.Queue(stack, 'AnotherQueue', {
            retentionPeriod: core_1.Duration.days(15),
        }), /message retention period must be 1209600 seconds of less/);
        test.done();
    },
    'message retention period can be provided as a parameter'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const parameter = new core_1.CfnParameter(stack, 'my-retention-period', {
            type: 'Number',
            default: 30,
        });
        // WHEN
        new sqs.Queue(stack, 'MyQueue', {
            retentionPeriod: core_1.Duration.seconds(parameter.valueAsNumber),
        });
        // THEN
        assert_1.expect(stack).toMatch({
            'Parameters': {
                'myretentionperiod': {
                    'Type': 'Number',
                    'Default': 30,
                },
            },
            'Resources': {
                'MyQueueE6CA6235': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'MessageRetentionPeriod': {
                            'Ref': 'myretentionperiod',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'addToPolicy will automatically create a policy for this queue'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'MyQueue');
        queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sqs:*'],
            principals: [new iam.ArnPrincipal('arn')],
        }));
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyQueueE6CA6235': {
                    'Type': 'AWS::SQS::Queue',
                },
                'MyQueuePolicy6BBEDDAC': {
                    'Type': 'AWS::SQS::QueuePolicy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sqs:*',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': 'arn',
                                    },
                                    'Resource': '*',
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'Queues': [
                            {
                                'Ref': 'MyQueueE6CA6235',
                            },
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'export and import': {
        'importing works correctly'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const imports = sqs.Queue.fromQueueArn(stack, 'Imported', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            // THEN
            // "import" returns an IQueue bound to `Fn::ImportValue`s.
            test.deepEqual(stack.resolve(imports.queueArn), 'arn:aws:sqs:us-east-1:123456789012:queue1');
            test.deepEqual(stack.resolve(imports.queueUrl), { 'Fn::Join': ['', ['https://sqs.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/', { Ref: 'AWS::AccountId' }, '/queue1']] });
            test.deepEqual(stack.resolve(imports.queueName), 'queue1');
            test.done();
        },
        'importing fifo and standard queues are detected correctly'(test) {
            const stack = new core_1.Stack();
            const stdQueue = sqs.Queue.fromQueueArn(stack, 'StdQueue', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            const fifoQueue = sqs.Queue.fromQueueArn(stack, 'FifoQueue', 'arn:aws:sqs:us-east-1:123456789012:queue2.fifo');
            test.deepEqual(stdQueue.fifo, false);
            test.deepEqual(fifoQueue.fifo, true);
            test.done();
        },
    },
    'grants': {
        'grantConsumeMessages'(test) {
            testGrant((q, p) => q.grantConsumeMessages(p), 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
            test.done();
        },
        'grantSendMessages'(test) {
            testGrant((q, p) => q.grantSendMessages(p), 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grantPurge'(test) {
            testGrant((q, p) => q.grantPurge(p), 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grant() is general purpose'(test) {
            testGrant((q, p) => q.grant(p, 'service:hello', 'service:world'), 'service:hello', 'service:world');
            test.done();
        },
        'grants also work on imported queues'(test) {
            const stack = new core_1.Stack();
            const queue = sqs.Queue.fromQueueAttributes(stack, 'Import', {
                queueArn: 'arn:aws:sqs:us-east-1:123456789012:queue1',
                queueUrl: 'https://queue-url',
            });
            const user = new iam.User(stack, 'User');
            queue.grantPurge(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                'sqs:PurgeQueue',
                                'sqs:GetQueueAttributes',
                                'sqs:GetQueueUrl',
                            ],
                            'Effect': 'Allow',
                            'Resource': 'arn:aws:sqs:us-east-1:123456789012:queue1',
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
    },
    'queue encryption': {
        'encryptionMasterKey can be set to a custom KMS key'(test) {
            const stack = new core_1.Stack();
            const key = new kms.Key(stack, 'CustomKey');
            const queue = new sqs.Queue(stack, 'Queue', { encryptionMasterKey: key });
            test.same(queue.encryptionMasterKey, key);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                'KmsMasterKeyId': { 'Fn::GetAtt': ['CustomKey1E6D0D07', 'Arn'] },
            }));
            test.done();
        },
        'a kms key will be allocated if encryption = kms but a master key is not specified'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                'KmsMasterKeyId': {
                    'Fn::GetAtt': [
                        'QueueKey39FCBAE6',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'it is possible to use a managed kms key'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS_MANAGED });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'Queue4A7E3555': {
                        'Type': 'AWS::SQS::Queue',
                        'Properties': {
                            'KmsMasterKeyId': 'alias/aws/sqs',
                        },
                    },
                },
            });
            test.done();
        },
        'grant also affects key on encrypted queue'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue', {
                encryption: sqs.QueueEncryption.KMS,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('someone'),
            });
            // WHEN
            queue.grantSendMessages(role);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                'sqs:SendMessage',
                                'sqs:GetQueueAttributes',
                                'sqs:GetQueueUrl',
                            ],
                            'Effect': 'Allow',
                            'Resource': { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] },
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:Encrypt',
                                'kms:ReEncrypt*',
                                'kms:GenerateDataKey*',
                            ],
                            'Effect': 'Allow',
                            'Resource': { 'Fn::GetAtt': ['QueueKey39FCBAE6', 'Arn'] },
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
    },
    'test ".fifo" suffixed queues register as fifo'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            queueName: 'MyQueue.fifo',
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'QueueName': 'MyQueue.fifo',
                        'FifoQueue': true,
                    },
                },
            },
        });
        test.done();
    },
    'test a fifo queue is observed when the "fifo" property is specified'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            fifo: true,
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'FifoQueue': true,
                    },
                },
            },
        });
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue');
        // THEN
        test.deepEqual(stack.resolve(queue.metricNumberOfMessagesSent()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'NumberOfMessagesSent',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.deepEqual(stack.resolve(queue.metricSentMessageSize()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'SentMessageSize',
            period: core_1.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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