# (c) 2022, NetApp, Inc
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
# Author: Laurent Nicolas, laurentn@netapp.com

""" unit tests for Ansible modules for ONTAP:
    utility to build REST responses and errors, and register them to use them in testcases.

    1) at the module level, define the REST responses:
       SRR = rest_responses()        if you're only interested in the default ones: 'empty', 'error', ...
       SRR = rest_responses(dict)    to use the default ones and augment them:
                                                a key identifies a response name, and the value is a tuple.

    3) in each test function, create a list of (event, response) using rest_response
        def test_create_aggr():
            register_responses([
                ('GET', 'cluster', SRR['is_rest']),
                ('POST', 'storage/aggregates', SRR['empty_good'])
            ])

    See ansible_collections/netapp/ontap/tests/unit/plugins/modules/test_na_ontap_aggregate_rest.py
    for an example.
"""

from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

# name: (html_code, dict, None or error string)
# dict is translated into an xml structure, num_records is None or an integer >= 0
_DEFAULT_RESPONSES = {
    # common responses
    'is_rest': (200, {}, None),
    'is_rest_95': (200, dict(version=dict(generation=9, major=5, minor=0, full='dummy_9_5_0')), None),
    'is_rest_96': (200, dict(version=dict(generation=9, major=6, minor=0, full='dummy_9_6_0')), None),
    'is_zapi': (400, {}, "Unreachable"),
    'empty_good': (200, {}, None),
    'end_of_sequence': (500, None, "Unexpected call to send_request"),
    'empty_records': (200, {'records': []}, None),
    'generic_error': (400, None, "Expected error"),
}


class rest_responses:
    ''' return an object that behaves like a read-only dictionary
        supports [key] to read an entry, and 'in' keyword to check key existence.
    '''
    def __init__(self, adict=None, allow_override=True):
        self.responses = dict(_DEFAULT_RESPONSES.items())
        if adict:
            for key, value in adict.items():
                if not allow_override and key in self.responses:
                    raise KeyError('duplicated key: %s' % key)
                self.responses[key] = value

    def _get_response(self, name):
        try:
            return self.responses[name]
        except KeyError:
            raise KeyError('%s not registered, list of valid keys: %s' % (name, self.responses.keys()))

    def __getitem__(self, name):
        return self._get_response(name)

    def __contains__(self, name):
        return name in self.responses
