"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2021.7.5"
__checksum__ = "e463c05114d3c720bf4b971bd0470e39b43696847f3cb24bc97e16fae98849ee"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), (16, 9), (25, 61), (86, 26), (112, 12), None, (124, 19), (143, 22), (165, 7), (172, 20), (192, 18), None, (210, 29), (239, 45), (284, 7), (291, 9), (300, 36), (336, 16), (352, 10), (362, 28), None, (390, 54), (444, 8), (452, 18), (470, 19), (489, 13), (502, 14), (516, 14), None, None, (530, 29), (559, 20), (579, 35), (614, 14), (628, 24), (652, 9), None, (661, 25), (686, 27), (713, 8), (721, 13), None, None, (734, 17), (751, 6), (757, 26), (783, 5), (788, 5), (793, 19), (812, 14), (826, 11), (837, 12), (849, 27), None, (876, 11), (887, 11), (898, 7), (905, 29), (934, 18), (952, 27), (979, 46), (1025, 25), (1050, 16), (1066, 8), (1074, 5), (1079, 22), (1101, 18), None, (1119, 36), (1155, 15), (1170, 8), (1178, 11), None, (1189, 5), (1194, 16), (1210, 14), (1224, 18), None, (1242, 14), (1256, 18), (1274, 48), (1322, 19), (1341, 5), (1346, 59), (1405, 14), (1419, 14), (1433, 20), None, (1453, 10), (1463, 13), (1476, 15), (1491, 19), None, (1510, 13), (1523, 19), (1542, 11), (1553, 4), (1557, 22), (1579, 10), (1589, 7), (1596, 14), (1610, 21), (1631, 11), (1642, 21), (1663, 12), (1675, 32), None, (1707, 10), (1717, 14), (1731, 12), (1743, 45), (1788, 15), None, (1803, 11), (1814, 23), (1837, 21), (1858, 26), (1884, 6), (1890, 6), (1896, 7), (1903, 5), (1908, 20), (1928, 23), (1951, 24), (1975, 13), (1988, 15), (2003, 19), (2022, 6), (2028, 61), (2089, 44), (2133, 12), (2145, 23), (2168, 16), (2184, 38), (2222, 6), (2228, 12), (2240, 44), (2284, 6), (2290, 41), (2331, 13), (2344, 23), (2367, 30), (2397, 16), (2413, 8), (2421, 15), (2436, 12), (2448, 19), (2467, 21), (2488, 15), None, (2503, 35), (2538, 21), (2559, 17), (2576, 19), (2595, 26), (2621, 5), (2626, 37), (2663, 26), (2689, 16), (2705, 10), (2715, 17), (2732, 23), (2755, 14), (2769, 17), (2786, 8), (2794, 8), (2802, 7), (2809, 29), (2838, 6), (2844, 18), (2862, 27), (2889, 20), (2909, 17), (2926, 19), (2945, 12), (2957, 40), (2997, 40), (3037, 12), (3049, 48), (3097, 25), (3122, 12), None, (3134, 8), (3142, 25), (3167, 19), (3186, 6), (3192, 23), None, (3215, 30), (3245, 33), (3278, 14), (3292, 12), (3304, 27), None, (3331, 26), (3357, 41), (3398, 50), (3448, 15), (3463, 20), (3483, 15), (3498, 21), (3519, 32), (3551, 24), (3575, 20), (3595, 17), (3612, 60), (3672, 19), (3691, 9), (3700, 12), (3712, 12), (3724, 11), (3735, 10), (3745, 48), (3793, 32), None, (3825, 25), (3850, 23), None, (3873, 8), (3881, 8), (3889, 7), None, (3896, 25), (3921, 17), None, (3938, 21), (3959, 35), (3994, 21), (4015, 10), (4025, 36), (4061, 20), (4081, 22), (4103, 23), (4126, 19), (4145, 12), (4157, 5), (4162, 30), (4192, 24), (4216, 14), (4230, 14), (4244, 47), (4291, 52), None, None, (4343, 51), (4394, 42), None, (4436, 14), None, (4450, 15), (4465, 8), (4473, 21), (4494, 6), (4500, 16), (4516, 17)], [(4533, 8190), (12723, 8882), (21605, 9190), (30795, 7781), (38576, 8255), (46831, 8029), (54860, 8934), (63794, 7875), (71669, 8820), (80489, 8082), (88571, 9388), (97959, 7949), (105908, 8642), (114550, 9958), (124508, 8431), (132939, 8673), (141612, 9110), (150722, 8044), (158766, 8341), (167107, 8078), (175185, 8805), (183990, 8633), (192623, 8887), (201510, 8332), (209842, 8798), (218640, 8343), (226983, 8834), (235817, 9034), (244851, 8022), (252873, 8523), (261396, 8761), (270157, 8239), (278396, 8528), (286924, 8851), (295775, 7767), (303542, 8709), (312251, 8453), (320704, 9146), (329850, 8730), (338580, 8782), (347362, 9432), (356794, 8029), (364823, 8149), (372972, 8248), (381220, 8404), (389624, 8345), (397969, 8379), (406348, 9486), (415834, 8300), (424134, 7979), (432113, 8332), (440445, 8761), (449206, 8875), (458081, 8706), (466787, 8854), (475641, 8456), (484097, 8890), (492987, 8426), (501413, 8916), (510329, 7528), (517857, 8469), (526326, 8526), (534852, 8465), (543317, 8836), (552153, 8672), (560825, 8872), (569697, 8174), (577871, 9152), (587023, 9002), (596025, 8676), (604701, 8603), (613304, 8066), (621370, 7708), (629078, 8829), (637907, 8577), (646484, 9090), (655574, 7957), (663531, 9111), (672642, 8828), (681470, 8099), (689569, 8911), (698480, 7319), (705799, 8328), (714127, 8804), (722931, 8167), (731098, 8554), (739652, 9186), (748838, 8547), (757385, 8799), (766184, 8714), (774898, 9571), (784469, 7982), (792451, 8428), (800879, 8463), (809342, 8450), (817792, 9090), (826882, 8738), (835620, 8326), (843946, 8259), (852205, 8107), (860312, 8117), (868429, 8681), (877110, 8292), (885402, 8300), (893702, 8166), (901868, 9083), (910951, 9067), (920018, 9042), (929060, 9710), (938770, 8973), (947743, 8830), (956573, 8935), (965508, 8561), (974069, 8398), (982467, 8915), (991382, 8842), (1000224, 8508), (1008732, 8152), (1016884, 8312), (1025196, 9260), (1034456, 9055), (1043511, 9009), (1052520, 8582), (1061102, 8821), (1069923, 9675), (1079598, 8253), (1087851, 7747), (1095598, 9155), (1104753, 8578), (1113331, 10119), (1123450, 9052), (1132502, 8103), (1140605, 8807), (1149412, 8548), (1157960, 8318), (1166278, 8621), (1174899, 8154), (1183053, 9055), (1192108, 8209), (1200317, 8371), (1208688, 8844), (1217532, 8798), (1226330, 7856), (1234186, 8271), (1242457, 9031), (1251488, 8332), (1259820, 8447), (1268267, 8497), (1276764, 8213), (1284977, 8933), (1293910, 8723), (1302633, 8695), (1311328, 8995), (1320323, 8204), (1328527, 8618), (1337145, 8683), (1345828, 8279), (1354107, 8637), (1362744, 8138), (1370882, 7673), (1378555, 7767), (1386322, 8795), (1395117, 9233), (1404350, 8295), (1412645, 8269), (1420914, 9297), (1430211, 8477), (1438688, 8061), (1446749, 9003), (1455752, 8599), (1464351, 7627), (1471978, 8449), (1480427, 9939), (1490366, 8035), (1498401, 8059), (1506460, 9012), (1515472, 8517), (1523989, 9025), (1533014, 8290), (1541304, 8083), (1549387, 10631), (1560018, 8888), (1568906, 8573), (1577479, 8791), (1586270, 9457), (1595727, 9565), (1605292, 7898), (1613190, 8730), (1621920, 8150), (1630070, 8565), (1638635, 9311), (1647946, 8137), (1656083, 8915), (1664998, 8754), (1673752, 8387), (1682139, 8564), (1690703, 8358), (1699061, 8387), (1707448, 8673), (1716121, 8348), (1724469, 8648), (1733117, 8067), (1741184, 9001), (1750185, 8546), (1758731, 9249), (1767980, 9034), (1777014, 7634), (1784648, 9014), (1793662, 8374), (1802036, 8802), (1810838, 8807), (1819645, 8992), (1828637, 8574), (1837211, 8859), (1846070, 8716), (1854786, 8490), (1863276, 8544), (1871820, 8240), (1880060, 8671), (1888731, 9012), (1897743, 8717), (1906460, 8007), (1914467, 9605), (1924072, 8489), (1932561, 8510), (1941071, 8470), (1949541, 8515), (1958056, 7896), (1965952, 8875), (1974827, 8678), (1983505, 9265), (1992770, 8658), (2001428, 8089), (2009517, 9065), (2018582, 8521), (2027103, 9500), (2036603, 8230), (2044833, 8234), (2053067, 7582), (2060649, 9021), (2069670, 8860), (2078530, 9139), (2087669, 8292), (2095961, 8696), (2104657, 8329), (2112986, 9050), (2122036, 8329), (2130365, 7911), (2138276, 8425), (2146701, 7974), (2154675, 8800), (2163475, 9108), (2172583, 9026), (2181609, 8282), (2189891, 8475), (2198366, 8576)], [(2206942, 933), (2207875, 760), (2208635, 811), (2209446, 1010), (2210456, 660), (2211116, 861), (2211977, 668), (2212645, 1000), (2213645, 721), (2214366, 789), (2215155, 656), (2215811, 678), (2216489, 827), (2217316, 885), (2218201, 1067), (2219268, 1021), (2220289, 1366), (2221655, 747), (2222402, 1008), (2223410, 873), (2224283, 791), (2225074, 929), (2226003, 1017), (2227020, 811), (2227831, 843), (2228674, 699), (2229373, 1099), (2230472, 1327), (2231799, 818), (2232617, 869), (2233486, 1077), (2234563, 892), (2235455, 697), (2236152, 842), (2236994, 997), (2237991, 985), (2238976, 847), (2239823, 891), (2240714, 832), (2241546, 1225), (2242771, 726), (2243497, 989), (2244486, 807), (2245293, 784), (2246077, 780), (2246857, 561), (2247418, 1086), (2248504, 1045), (2249549, 919), (2250468, 619), (2251087, 937), (2252024, 734), (2252758, 855), (2253613, 1109), (2254722, 1094), (2255816, 593), (2256409, 782), (2257191, 693), (2257884, 728), (2258612, 892), (2259504, 915), (2260419, 859), (2261278, 1159), (2262437, 1078), (2263515, 857), (2264372, 810), (2265182, 809), (2265991, 511), (2266502, 719), (2267221, 640), (2267861, 826), (2268687, 1001), (2269688, 689), (2270377, 880), (2271257, 700), (2271957, 797), (2272754, 714), (2273468, 775), (2274243, 882), (2275125, 586), (2275711, 919), (2276630, 724), (2277354, 991), (2278345, 701), (2279046, 804), (2279850, 568), (2280418, 778), (2281196, 859), (2282055, 943), (2282998, 875), (2283873, 1092), (2284965, 1262), (2286227, 919), (2287146, 943), (2288089, 846), (2288935, 554), (2289489, 1009), (2290498, 906), (2291404, 675), (2292079, 763), (2292842, 848), (2293690, 1042), (2294732, 980), (2295712, 608), (2296320, 706), (2297026, 940), (2297966, 533), (2298499, 589), (2299088, 1060), (2300148, 1032), (2301180, 821), (2302001, 852), (2302853, 803), (2303656, 801), (2304457, 959), (2305416, 873), (2306289, 715), (2307004, 627), (2307631, 801), (2308432, 744), (2309176, 1153), (2310329, 757), (2311086, 904), (2311990, 568), (2312558, 821), (2313379, 914), (2314293, 911), (2315204, 1072), (2316276, 767), (2317043, 1079), (2318122, 925), (2319047, 653), (2319700, 939), (2320639, 787), (2321426, 990), (2322416, 828), (2323244, 777), (2324021, 776), (2324797, 828), (2325625, 706), (2326331, 760), (2327091, 744), (2327835, 824), (2328659, 651), (2329310, 605), (2329915, 637), (2330552, 750), (2331302, 701), (2332003, 824), (2332827, 715), (2333542, 824), (2334366, 631), (2334997, 618), (2335615, 893), (2336508, 789), (2337297, 778), (2338075, 818), (2338893, 1084), (2339977, 880), (2340857, 664), (2341521, 1096), (2342617, 903), (2343520, 720), (2344240, 858), (2345098, 993), (2346091, 719), (2346810, 745), (2347555, 730), (2348285, 756), (2349041, 799), (2349840, 900), (2350740, 677), (2351417, 993), (2352410, 825), (2353235, 939), (2354174, 897), (2355071, 821), (2355892, 630), (2356522, 792), (2357314, 802), (2358116, 2027), (2360143, 626), (2360769, 843), (2361612, 777), (2362389, 1088), (2363477, 913), (2364390, 863), (2365253, 666), (2365919, 704), (2366623, 994), (2367617, 638), (2368255, 651), (2368906, 884), (2369790, 814), (2370604, 985), (2371589, 854), (2372443, 810), (2373253, 763), (2374016, 908), (2374924, 781), (2375705, 961), (2376666, 785), (2377451, 896), (2378347, 688), (2379035, 846), (2379881, 675), (2380556, 981), (2381537, 992), (2382529, 768), (2383297, 1080), (2384377, 780), (2385157, 933), (2386090, 1006), (2387096, 1146), (2388242, 945), (2389187, 846), (2390033, 1006), (2391039, 791), (2391830, 633), (2392463, 490), (2392953, 908), (2393861, 852), (2394713, 618), (2395331, 1117), (2396448, 623), (2397071, 824), (2397895, 969), (2398864, 997), (2399861, 925), (2400786, 746), (2401532, 962), (2402494, 819), (2403313, 971), (2404284, 674), (2404958, 700), (2405658, 619), (2406277, 698), (2406975, 486), (2407461, 900), (2408361, 1067), (2409428, 926), (2410354, 751), (2411105, 724), (2411829, 709), (2412538, 1046), (2413584, 616), (2414200, 625), (2414825, 1007), (2415832, 497), (2416329, 990), (2417319, 2215), (2419534, 727), (2420261, 785), (2421046, 1003), (2422049, 1151), (2423200, 515)], [(2423715, 48), None, (2423763, 35), (2423798, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (2423840, 42), None, (2423882, 25), (2423907, 44), (2423951, 22), (2423973, 18), None, None, None, None, (2423991, 26), None, None, None, None, (2424017, 21), (2424038, 25), None, None, (2424063, 26), None, None, None, None, (2424089, 71), (2424160, 21), (2424181, 23), None, None, None, None, (2424204, 48), None, None, None, None, None, (2424252, 31), None, None, None, None, (2424283, 42), None, (2424325, 22), None, (2424347, 21), None, (2424368, 26), (2424394, 42), None, None, (2424436, 77), (2424513, 27), None, None, None, None, (2424540, 21), (2424561, 21), None, None, (2424582, 34), (2424616, 42), None, None, None, (2424658, 25), None, None, (2424683, 21), None, None, None, None, None, (2424704, 24), (2424728, 21), None, None, (2424749, 26), None, (2424775, 18), None, (2424793, 54), None, None, None, None, None, None, (2424847, 26), None, None, None, (2424873, 20), None, None, (2424893, 64), (2424957, 42), (2424999, 17), (2425016, 17), (2425033, 26), None, (2425059, 26), None, None, None, (2425085, 26), (2425111, 20), (2425131, 26), None, (2425157, 42), (2425199, 63), None, None, None, (2425262, 40), (2425302, 48), None, None, None, (2425350, 47), None, None, None, None, None, None, None, (2425397, 42), None, (2425439, 80), None, (2425519, 9), None, (2425528, 21), (2425549, 42), None, None, (2425591, 65), (2425656, 82), None, None, (2425738, 72), None, None, (2425810, 24), (2425834, 21), None, None, None, None, None, (2425855, 42), (2425897, 21), (2425918, 21), None, (2425939, 42), (2425981, 25), None, (2426006, 38), (2426044, 21), (2426065, 56), None, None, (2426121, 21), (2426142, 19), (2426161, 26), None, (2426187, 16), None, (2426203, 21), None, None, (2426224, 38), None, (2426262, 22), (2426284, 21), (2426305, 21), (2426326, 21), None, (2426347, 63), None, (2426410, 21), (2426431, 42), None, (2426473, 17), None, None, None, None, (2426490, 21), (2426511, 21), None, None, (2426532, 21), None, None, (2426553, 21), None, (2426574, 26), None, (2426600, 50), None, None, None, (2426650, 50), (2426700, 26), (2426726, 21), (2426747, 21), (2426768, 19), None, (2426787, 35), (2426822, 26), (2426848, 23), (2426871, 39), (2426910, 42), None, None, None, None, None, None, (2426952, 21), None, None, None, (2426973, 21), None, None, (2426994, 90), None, (2427084, 239), (2427323, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
