import numpy as np
from collections import OrderedDict
import json
from six.moves.urllib.request import urlopen

from astrorapid.classify import Classify


def read_json(url):
    response = urlopen(url)
    return json.loads(response.read(), object_pairs_hook=OrderedDict)


def delete_indexes(deleteindexes, *args):
    newarrs = []
    for arr in args:
        newarr = np.delete(arr, deleteindexes)
        newarrs.append(newarr)

    return newarrs


def convert_lists_to_arrays(*args):
    output = []
    for arg in args:
        out_array = np.asarray(arg)
        output.append(out_array)

    return output


def read_lasair_json(object_name='ZTF18acsovsw'):
    """
    Read light curve from lasair website API based on object name.

    Parameters
    ----------
    object_name : str
        The LASAIR object name. E.g. object_name='ZTF18acsovsw'

    """

    url = 'https://lasair.roe.ac.uk/object/{}/json/'.format(object_name)

    data = read_json(url)

    objid = data['objectId']
    ra = data['objectData']['ramean']
    dec = data['objectData']['decmean']
    # lasair_classification = data['objectData']['classification']
    tns_info = data['objectData']['annotation']
    photoZ = None
    for cross_match in data['crossmatches']:
        print(cross_match)
        photoZ = cross_match['photoZ']
        separation_arcsec = cross_match['separationArcsec']
        catalogue_object_type = cross_match['catalogue_object_type']
    if photoZ is None:  # TODO: Get correct redshift
        try:
            if "z=" in tns_info:
                photoZ = tns_info.split('z=')[1]
                redshift = float(photoZ.split()[0])
                print("PHOTOZZZZZZZZZZZZ", redshift, tns_info)
            elif "Z=" in tns_info:
                photoZ = tns_info.split('Z=')[1]
                redshift = float(photoZ.split()[0])
                print("PHOTOZZZZZZZZZZZZ", redshift, tns_info)
            else:
                print("TRYING ARITRARY GUESS REDSHIFT = 0.1")
                redshift = 0.1  # return
        except Exception as e:
            print(e)
    else:
        redshift = photoZ
    print("Redshift is {}".format(redshift))
    mwebv = 0.2  # TODO: Get correct extinction

    mjd = []
    passband = []
    mag = []
    magerr = []
    photflag = []
    for cand in data['candidates']:
        mjd.append(cand['mjd'])
        passband.append(cand['fid'])
        mag.append(cand['magpsf'])
        if 'sigmapsf' in cand:
            magerr.append(cand['sigmapsf'])
            photflag.append(4096)
        else:
            magerr.append(0.1 * cand['magpsf'])  # magerr.append(None)  #
            photflag.append(0)

    mjd, passband, mag, magerr, photflag = convert_lists_to_arrays(mjd, passband, mag, magerr, photflag)
    zeropoint = np.array([26.2] * len(mjd))  # TODO: Get correct zeropoint

    deleteindexes = np.where(magerr == None)  # [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29]  #
    mjd, passband, mag, magerr, photflag, zeropoint = delete_indexes(deleteindexes, mjd, passband, mag, magerr, photflag, zeropoint)

    return mjd, passband, mag, magerr, photflag, zeropoint, ra, dec, objid, redshift, mwebv


def classify_lasair_light_curves(object_name='ZTF18acsovsw', plot=True):
    mjd, passband, mag, magerr, photflag, zeropoint, ra, dec, objid, redshift, mwebv = read_lasair_json(object_name)

    flux = 10. ** (-0.4 * (mag - zeropoint))
    fluxerr = np.abs(flux * magerr * (np.log(10.) / 2.5))

    passband = np.where((passband == 1) | (passband == '1'), 'g', passband)
    passband = np.where((passband == 2) | (passband == '2'), 'r', passband)

    # Set photflag detections when S/N > 5
    photflag2 = np.zeros(len(flux))
    photflag2[flux / fluxerr > 5] = 4096
    photflag2[np.where(mjd == min(mjd[photflag2 == 4096]))] = 6144

    mjd_first_detection = min(mjd[photflag == 4096])
    photflag[np.where(mjd == mjd_first_detection)] = 6144

    deleteindexes = np.where(((passband == 3) | (passband == '3')) | (mjd > mjd_first_detection) & (photflag == 0))
    print("Deleting indexes {} at mjd {} and passband {}".format(deleteindexes, mjd[deleteindexes], passband[deleteindexes]))
    mjd, passband, flux, fluxerr, zeropoint, photflag = delete_indexes(deleteindexes, mjd, passband, flux, fluxerr, zeropoint, photflag)

    light_curve_list = [(mjd, flux, fluxerr, passband, zeropoint, photflag, ra, dec, objid, redshift, mwebv)]

    classification = Classify(light_curve_list, known_redshift=True, bcut=False, zcut=None)
    predictions = classification.get_predictions()
    print(predictions)

    if plot:
        classification.plot_light_curves_and_classifications(step=True, use_interp_flux=False)
        classification.plot_light_curves_and_classifications(step=False, use_interp_flux=True)
        classification.plot_light_curves_and_classifications(step=False, use_interp_flux=False)
        # classification.plot_classification_animation_step()
        # classification.plot_classification_animation()

    return classification.orig_lc, classification.timesX, classification.y_predict


if __name__ == '__main__':
    # classify_lasair_light_curves(object_name='ZTF19aabbnzo')
    # classify_lasair_light_curves(object_name='ZTF18abukavn')
    # classify_lasair_light_curves(object_name='ZTF18aajupnt') # 2018dyk
    # classify_lasair_light_curves(object_name='ZTF18aabtxvd') # 2018zr
    # classify_lasair_light_curves(object_name='ZTF18acpdvos') # 2018hyz

    # classify_lasair_light_curves(object_name='ZTF18acchhpe')
    # classify_lasair_light_curves(object_name='ZTF18abwlupf')

    classify_lasair_light_curves(object_name='ZTF18abxftqm')

    # SN for name in ['ZTF18abwlupf', 'ZTF18abzsqjw', 'ZTF18abztcib', 'ZTF18abzqafh', 'ZTF18abzvpif', 'ZTF18abzbkby', 'ZTF18abzsdju', 'ZTF18acafztq', 'ZTF18acaeous', 'ZTF18abyiusv', 'ZTF18acahuph', 'ZTF18acajkff', 'ZTF18acajowp', 'ZTF18acajwos', 'ZTF18acakdvp', 'ZTF18acaekjn', 'ZTF18acaksoq', 'ZTF18abxsmlf', 'ZTF17aabuxzy', 'ZTF18acandwv', 'ZTF18abzpjae', 'ZTF18acakwsu', 'ZTF18acamvve', 'ZTF18acaimrb', 'ZTF18abzrgim', 'ZTF18acaipdc', 'ZTF18acarxol', 'ZTF18acakfha', 'ZTF18acamnru', 'ZTF18acbujhw', 'ZTF18acbvkwl', 'ZTF18aabtvzf', 'ZTF18acbvifx', 'ZTF18acbvxdz', 'ZTF18aadrhsi', 'ZTF18acbvzta', 'ZTF18acbwdwg', 'ZTF18acadvva', 'ZTF18acbwiic', 'ZTF18acbwjyx', 'ZTF18acbwkso', 'ZTF18acbwktg', 'ZTF18acbwkjw', 'ZTF18acbwhns', 'ZTF18acbwppb', 'ZTF18acbwssb', 'ZTF18acbwtnl', 'ZTF18acbwtnh', 'ZTF18acbwtmf', 'ZTF18acbwwjg', 'ZTF18acbwxdm', 'ZTF18acbwvsq', 'ZTF18acbwwpm', 'ZTF18acbwwlc', 'ZTF18acbwxcc', 'ZTF18acbwxou', 'ZTF18acbwtkh', 'ZTF18acbxhua', 'ZTF18acbxkop', 'ZTF18acatuts', 'ZTF18acbxsge', 'ZTF18acbxsap', 'ZTF18acbxubx', 'ZTF18acbxudr', 'ZTF18abxmbjy', 'ZTF18acbvdjv', 'ZTF17aaaslrl', 'ZTF18acbxzge', 'ZTF18acbyfwa', 'ZTF18acbvmwr', 'ZTF18acbyoft', 'ZTF18acbvmcl', 'ZTF18aaadyuc', 'ZTF18acbzppd', 'ZTF18acbwaax', 'ZTF18acbzqyi', 'ZTF18acbwfvt', 'ZTF18acbwgeg', 'ZTF18aaemivw', 'ZTF18acbwarv', 'ZTF18accbwow', 'ZTF18acbvyhx', 'ZTF18accdsxf', 'ZTF18acbbrig', 'ZTF18accfxnn', 'ZTF18acbxphq', 'ZTF18abtpite', 'ZTF18acchhpf', 'ZTF18acchvwh', 'ZTF18acchzyo', 'ZTF18acciaxb', 'ZTF18acbuwcq', 'ZTF18acbuemc', 'ZTF18acbufud', 'ZTF18acbvahw', 'ZTF18abvmkfh', 'ZTF18acbvhqi', 'ZTF18acbvhit', 'ZTF18acbvcmw', 'ZTF18acckpro', 'ZTF18acbvbox', 'ZTF17aachnrk', 'ZTF18acbuibx', 'ZTF18acbvijf', 'ZTF18acbvuli', 'ZTF18acbvnen', 'ZTF18acclexy', 'ZTF18acclshz', 'ZTF18acclmiq', 'ZTF18aabrzxo', 'ZTF18accnoli', 'ZTF18accntcd', 'ZTF18accepnx', 'ZTF18acbwyjt', 'ZTF18acbwopo', 'ZTF18acbwoym', 'ZTF18acbwtqi', 'ZTF18acapyww', 'ZTF18acavizq', 'ZTF18acbwxgn', 'ZTF18accqcab', 'ZTF18acbwvsp', 'ZTF18acbwqbd', 'ZTF18accqjow', 'ZTF18acauwik', 'ZTF18acbybjd', 'ZTF18acbxkpa', 'ZTF18accqket', 'ZTF18acahvwm', 'ZTF18acbxqyz', 'ZTF18accqzaj', 'ZTF18acbxshd', 'ZTF18acbxsjs', 'ZTF18acbxsec', 'ZTF18accrjwg', 'ZTF18acbyyff', 'ZTF18acbvmzg', 'ZTF17aaagxae', 'ZTF18abztrfp', 'ZTF18acbzpmu', 'ZTF18aaqkjuw', 'ZTF18aakeywu', 'ZTF18acbwfza', 'ZTF18acbznkf', 'ZTF18acbznbe', 'ZTF18acbwgfe', 'ZTF18aalpega', 'ZTF18acbzojv', 'ZTF18acbwawc', 'ZTF18acbwazl', 'ZTF18acbzpgy', 'ZTF18acbzoyh', 'ZTF18accxbta', 'ZTF18accxmbh', 'ZTF18acdvrfu', 'ZTF18acdvvhi', 'ZTF18acdways', 'ZTF18acdwazf', 'ZTF18acdvvej', 'ZTF18accekii', 'ZTF18abzqpjc', 'ZTF18acbbscf', 'ZTF18accefrw', 'ZTF18accebit', 'ZTF18acdwiwv', 'ZTF18accatdh', 'ZTF18accercc', 'ZTF18accqhag', 'ZTF18acbxmvy', 'ZTF18abzscns', 'ZTF18acbudzh', 'ZTF18acbuftp', 'ZTF18accjecm', 'ZTF18acchhpe', 'ZTF18accicnc', 'ZTF18accjuwd', 'ZTF18acbxwta', 'ZTF18accjqzu', 'ZTF18acebssa', 'ZTF18acbyhwg', 'ZTF18acecqjl', 'ZTF18acectzf', 'ZTF18accnnyu', 'ZTF18accnbgw', 'ZTF18acbznpp', 'ZTF18acbwfim', 'ZTF18acasuxc', 'ZTF18acaqcdg', 'ZTF18acbwnsh', 'ZTF18acapxrt', 'ZTF18aceguzr', 'ZTF18acegozh', 'ZTF18accnyyz', 'ZTF18acbwpak', 'ZTF18acdwohd', 'ZTF18abujubn', 'ZTF18acbwstu', 'ZTF18aceijsp', 'ZTF18acbxoie', 'ZTF18acchlkx', 'ZTF18acehilk', 'ZTF18accbuhi', 'ZTF18acbycjw', 'ZTF18abybyqw', 'ZTF17aacplzj', 'ZTF18accvkpt', 'ZTF18aceqrcy', 'ZTF18accwihs', 'ZTF18acbzvpg', 'ZTF18accnfrz', 'ZTF18acesjso', 'ZTF18acbzoxp', 'ZTF18acbzoxo', 'ZTF18acbwaxk', 'ZTF18accdymq', 'ZTF18accfjmp', 'ZTF18acesprs', 'ZTF17aabutmm', 'ZTF18acebefc', 'ZTF17aaakeap', 'ZTF18accvflv', 'ZTF18acewivc', 'ZTF18acexbir', 'ZTF17aaatzww', 'ZTF18abzssca', 'ZTF18acexxmw', 'ZTF18accnlji', 'ZTF18aceykjp', 'ZTF18acdvvgx', 'ZTF18accdtch', 'ZTF18aceyzwi', 'ZTF18acdwavm', 'ZTF18acaymqq', 'ZTF18acawznm', 'ZTF18acfejol', 'ZTF18acbvhzt', 'ZTF18aaapvkn', 'ZTF18acbwuah', 'ZTF18accpjkb', 'ZTF18acbwtng', 'ZTF18acctdhp', 'ZTF18acbxqis', 'ZTF18acbybzj', 'ZTF18acbuvpb', 'ZTF18achavgj', 'ZTF18aabjdkp', 'ZTF18acbvmjq', 'ZTF18accnmri', 'ZTF18acbznvp', 'ZTF17aacxkaq', 'ZTF18acbzpxx', 'ZTF18aceolcv', 'ZTF18acbwagb', 'ZTF18acbwgcd', 'ZTF18acbzvqn', 'ZTF18aamsgjq', 'ZTF18aceyjco', 'ZTF18acbwghc', 'ZTF18acaahzg', 'ZTF18accwjiu', 'ZTF18acbzqoc', 'ZTF18acbzqrk', 'ZTF18accdnep', 'ZTF18acgvgiq', 'ZTF18acbzwna', 'ZTF18abmfcdr', 'ZTF18accdxpa', 'ZTF18acbxnpg', 'ZTF18achepzk', 'ZTF18acfwmqj', 'ZTF18achkbum', 'ZTF18abnzunr', 'ZTF17aacfeuw', 'ZTF18achqmhh', 'ZTF18acgyksx', 'ZTF17aaayvyb', 'ZTF18accsane', 'ZTF18acchhxu', 'ZTF18acmwuum', 'ZTF18acbxzmb', 'ZTF18aceaape', 'ZTF18acbudwt', 'ZTF18acmwpue', 'ZTF18acchzkn', 'ZTF18acdzzyf', 'ZTF18aceathd', 'ZTF18aabezwv', 'ZTF18acmyfyu', 'ZTF18acbvgtj', 'ZTF17aacbnge', 'ZTF18achrter', 'ZTF18acexrlk', 'ZTF17aaatdlu', 'ZTF17aadnnft', 'ZTF18abzjwej', 'ZTF18achtnvk', 'ZTF18acbvtza', 'ZTF18acmzjkx', 'ZTF18acdvahc', 'ZTF18achtpcj', 'ZTF18acbzitw', 'ZTF18accvqmj', 'ZTF18acewwqf', 'ZTF18acbwdxy', 'ZTF18achoord', 'ZTF18acnbgrb', 'ZTF18acegbeo', 'ZTF18acbwabp', 'ZTF18aawmppg', 'ZTF18acnbptb', 'ZTF18aamrkmd', 'ZTF18acnmdao', 'ZTF18acgsdvk', 'ZTF18accdego', 'ZTF17aaciapo', 'ZTF18achdnip', 'ZTF18achdidy', 'ZTF18aceynvm', 'ZTF18achdfqm', 'ZTF18achyqcq', 'ZTF18acnnxzz', 'ZTF18accdwoz', 'ZTF18acewcbn', 'ZTF18acnlqud', 'ZTF18accngee', 'ZTF17aadkdks', 'ZTF18acnngyx', 'ZTF18acpeesd', 'ZTF18acnaydb', 'ZTF18aceyqpm', 'ZTF18acnbglg', 'ZTF18acbznnc', 'ZTF18acmzhmp', 'ZTF18acmzpbf', 'ZTF18acppddp', 'ZTF18acdvqhn', 'ZTF18aceykmc', 'ZTF18acpefbb', 'ZTF18acbzvzm', 'ZTF18acnbejq', 'ZTF18acnbenl', 'ZTF18aathofv', 'ZTF18aceyjfx', 'ZTF18acppzle', 'ZTF18acbzvuu', 'ZTF18ackvfyc', 'ZTF18acisruf', 'ZTF18acarupz', 'ZTF18acbxvom', 'ZTF18abzoryc', 'ZTF18acbvhfl', 'ZTF18aabjtqi', 'ZTF18achszin', 'ZTF17aacekrp', 'ZTF18acctsvr', 'ZTF18acefgee', 'ZTF18accvmgs', 'ZTF18acnlwws', 'ZTF18acnbgpw', 'ZTF18aadezmv', 'ZTF18acnbgvd', 'ZTF18aabklju', 'ZTF17aaaguld', 'ZTF18acptzwq', 'ZTF18acpuici', 'ZTF18acqqyah', 'ZTF18acqsfxh', 'ZTF18acnnfup', 'ZTF18aceyycq', 'ZTF18acpphpi', 'ZTF18acqwdla', 'ZTF18acaoelw', 'ZTF18acpfwmm', 'ZTF17aabqwns', 'ZTF18achlsqa', 'ZTF18aabezcx', 'ZTF18acmzsjl', 'ZTF18acqxsyi', 'ZTF18acrdlrp', 'ZTF18acbwgcu', 'ZTF18acploez', 'ZTF18achqdch', 'ZTF18acppwaf', 'ZTF18aaikyip', 'ZTF18acrhefh', 'ZTF18acqpobb', 'ZTF18acqsqry', 'ZTF18acnbgrk', 'ZTF18acrhehl', 'ZTF18acqphpd', 'ZTF18acrcwmj', 'ZTF18acppdfv', 'ZTF18acnbqdx', 'ZTF18acrdwag', 'ZTF18acrcpbe', 'ZTF18acrukmr', 'ZTF18acrukrr', 'ZTF18acrgiyy', 'ZTF18abzfjaa', 'ZTF18acnmqvh', 'ZTF18abvrmwx', 'ZTF18acptgzz', 'ZTF18acrvcqx', 'ZTF18acdvnym', 'ZTF18acqxcga', 'ZTF17aaaoqmq', 'ZTF18acpuxuh', 'ZTF18acrxfwp', 'ZTF18acnbbvk', 'ZTF18acqxwgl', 'ZTF18acrcyqw', 'ZTF18acrxiod', 'ZTF18aaacnpb', 'ZTF18acrhioo', 'ZTF18acrxpzx', 'ZTF18acrxpxb', 'ZTF18acpefgk', 'ZTF18acpwomj', 'ZTF18acepwhb', 'ZTF18acrdmmw', 'ZTF18acgvysg', 'ZTF18acrxvsn', 'ZTF18aazwtfj', 'ZTF18acnblbh', 'ZTF18acrxppd', 'ZTF18acrtykk', 'ZTF18acsjdxo', 'ZTF18acsjjqr', 'ZTF18acsjqvw', 'ZTF17aaaheis', 'ZTF18acsjrqw', 'ZTF17aabyktr', 'ZTF18acrewey', 'ZTF18acmxzeh', 'ZTF18acsjqga', 'ZTF18aaadxfi', 'ZTF18acrcenc', 'ZTF18acmypei', 'ZTF18acapgdp', 'ZTF18acrfvuf', 'ZTF18acskrzq', 'ZTF18acruyhw', 'ZTF18acrkaks', 'ZTF17aaceblj', 'ZTF17aadohxr', 'ZTF18acskkqy', 'ZTF18acrujgh', 'ZTF18acsjzcl', 'ZTF18acrxkru', 'ZTF18acefxpq', 'ZTF18acsoocj', 'ZTF18acnbeuu', 'ZTF18acpegrg', 'ZTF18acexyqt', 'ZTF18acefuhk', 'ZTF18acnncve', 'ZTF18aaevdnv', 'ZTF18aafwicl', 'ZTF18acsvycn', 'ZTF18abzpvwv', 'ZTF18acrjydf', 'ZTF18acsxqjx', 'ZTF18acrctos', 'ZTF18acsxsbf', 'ZTF18acsxsmi', 'ZTF18acsxucv', 'ZTF18acevgus', 'ZTF18acrcetn', 'ZTF18acmxkzj', 'ZTF18acsxvrp', 'ZTF18acsyujw', 'ZTF18acrciyr', 'ZTF18acrcizr', 'ZTF18acrcxek', 'ZTF18acszaiy', 'ZTF18acszayr', 'ZTF18acrtvmm', 'ZTF18acmyxfx', 'ZTF18acrvakb', 'ZTF17aabvtgd', 'ZTF18acrultm', 'ZTF18acsofdh', 'ZTF18actabky', 'ZTF18acrwheu', 'ZTF18acsyzla', 'ZTF18acpscso', 'ZTF18achmysr', 'ZTF18acsxrge', 'ZTF18acrknyn', 'ZTF18acrcuau', 'ZTF18acexqcz', 'ZTF18acrukpd', 'ZTF18acnmifq', 'ZTF18acenyfr', 'ZTF18acrlmvt', 'ZTF18acslgsa', 'ZTF18acujaic', 'ZTF18acpsrtc', 'ZTF18acszecy', 'ZTF18acsjpyq', 'ZTF18actyjmo', 'ZTF18acurbia', 'ZTF18acsovsw', 'ZTF18acrxpxd', 'ZTF18acuring', 'ZTF18acueemg', 'ZTF18acurlbl', 'ZTF18acrdtzn', 'ZTF18acuqyyn', 'ZTF18acurprq', 'ZTF18acsonov', 'ZTF18acuqxyf', 'ZTF18acuqskr', 'ZTF18acspjqd', 'ZTF17aaahyqu', 'ZTF18acuschq', 'ZTF18acpovry', 'ZTF18acszxqc', 'ZTF18acqzcdu', 'ZTF18acsouto', 'ZTF18acuefvt', 'ZTF18aajhfdm', 'ZTF18acqykvu', 'ZTF18acpeenk', 'ZTF18acpeemg', 'ZTF18acuwhvg', 'ZTF18acukeaq', 'ZTF18acukdyc', 'ZTF18aaaplxq', 'ZTF17aabwfrd', 'ZTF18abzlryq', 'ZTF18acuatde', 'ZTF17aaaieal', 'ZTF18acpgetv', 'ZTF17aaasznr', 'ZTF18acuimze', 'ZTF18acvbyjt', 'ZTF18actxhgn', 'ZTF18acrwgfp', 'ZTF18acvbpcv', 'ZTF18acvddao', 'ZTF18acubbwb', 'ZTF17aaagtee', 'ZTF18acuscgc', 'ZTF18acusbdy', 'ZTF18acspfaf', 'ZTF18acrvldh', 'ZTF18acurwua', 'ZTF18achdase', 'ZTF18acrncou', 'ZTF18acrgunw', 'ZTF18acvgrmd', 'ZTF18acpqahp', 'ZTF18aawxcbp', 'ZTF18aawmxdx', 'ZTF18acusrws', 'ZTF18acurljz', 'ZTF18acvgyst', 'ZTF18acrxgbu', 'ZTF18acsowoc', 'ZTF18acueeoo', 'ZTF18acurkik', 'ZTF18acrdtwo', 'ZTF18acebicb', 'ZTF18acujlwu', 'ZTF18acslbve', 'ZTF18aaanfpk', 'ZTF18acmzjqf', 'ZTF18aagiasb', 'ZTF18acurqaw', 'ZTF18acvvudh', 'ZTF18acvwcgq', 'ZTF18acvwchp', 'ZTF18acvwchw', 'ZTF18acvwcky', 'ZTF18aaiuynw', 'ZTF18acqppuj', 'ZTF18acvwcms', 'ZTF18acqpqph', 'ZTF18acvwddd', 'ZTF18acqsrzq', 'ZTF18acrxnme', 'ZTF18acvvved', 'ZTF18acvweal', 'ZTF18acsyanq', 'ZTF18acwbyuf', 'ZTF18acuwtcl', 'ZTF18acrexje', 'ZTF18acwfgbw', 'ZTF18acwhaxj', 'ZTF18acwdrfb', 'ZTF18aaaamjo', 'ZTF18acsxpmp', 'ZTF18acvplrz', 'ZTF18acptbqu', 'ZTF18acszixl', 'ZTF18aaaamoh', 'ZTF18accnceh', 'ZTF18acwwccb', 'ZTF18acvvuzi', 'ZTF18acwfzty', 'ZTF18acvgkio', 'ZTF18acvwgqy', 'ZTF18acvwgdu', 'ZTF18acvwbos', 'ZTF18acidwhi', 'ZTF17aaanppf', 'ZTF18acnnxdy', 'ZTF18acvcqqh', 'ZTF18acnmicp', 'ZTF18acrxnfr', 'ZTF18acubbuv', 'ZTF18acvvxfc', 'ZTF18acvguyi', 'ZTF18acqsqrg', 'ZTF18acvwchn', 'ZTF18acwyvhw', 'ZTF18acwyvet', 'ZTF18acuewyy', 'ZTF18aaquudq', 'ZTF18acxgujm', 'ZTF18acetnxh', 'ZTF18acxgumg', 'ZTF18acswton', 'ZTF18achaqmd', 'ZTF18acxgqij', 'ZTF17aaawcqg', 'ZTF18acseiei', 'ZTF18abvzkyr', 'ZTF18acslpba', 'ZTF18acszgpa', 'ZTF18acxhvkv', 'ZTF18acwrwrl', 'ZTF18aabfxpw', 'ZTF18abshnzy', 'ZTF17aabwgbf', 'ZTF18acurlbz', 'ZTF18acurlbt', 'ZTF18aabahuy', 'ZTF18acsoyiu', 'ZTF18acpeekw', 'ZTF18aawonvg', 'ZTF18acrnfww', 'ZTF18acynctq', 'ZTF18acynmcp', 'ZTF18acsxvpj', 'ZTF18aczawcw', 'ZTF18acsyeum', 'ZTF18acxijqw', 'ZTF18acymtph', 'ZTF17aaaakek', 'ZTF18actzpxw', 'ZTF18acrunrq', 'ZTF18acxyarg', 'ZTF18acvwhaz', 'ZTF18acsptko', 'ZTF18aczcsmg', 'ZTF18acwzyor', 'ZTF18acvgugm', 'ZTF18acvgqtf', 'ZTF18aclqiwo', 'ZTF18aczeesl', 'ZTF18acvgzab', 'ZTF18acurkzx', 'ZTF18aczeraw', 'ZTF18aczerlj', 'ZTF18acuswcl', 'ZTF18aczerpr', 'ZTF18acxexyu', 'ZTF18acsownk', 'ZTF18acsowkx', 'ZTF18acrxtrv', 'ZTF18acsxkov', 'ZTF18acynlmf', 'ZTF18acvbqaz', 'ZTF18acwugwx', 'ZTF18acurlbj', 'ZTF18acybdar', 'ZTF18aatjlhe', 'ZTF18acvgxmp', 'ZTF17aaaplzd', 'ZTF17aabtawv', 'ZTF18acbyeid', 'ZTF18acehmje', 'ZTF18acwtykp', 'ZTF18acyondd', 'ZTF18aczeryl', 'ZTF18aczeraq', 'ZTF18adaifzl', 'ZTF18acwvdoo', 'ZTF18acrtwlx', 'ZTF17aaahszr', 'ZTF18acvgwbt', 'ZTF18acxbksd', 'ZTF18adakzvj', 'ZTF18acwyqeg', 'ZTF18aczeqva', 'ZTF18adalgmc', 'ZTF18acueief', 'ZTF18acuegof', 'ZTF18acnnevs', 'ZTF18acwvehz', 'ZTF18adatknf', 'ZTF17aaaufly', 'ZTF17aabvtke', 'ZTF18aabgsjq', 'ZTF18aczdmwd', 'ZTF18achyngk', 'ZTF18acvdstg', 'ZTF18adaslwa', 'ZTF18adayssg', 'ZTF18adaysnz', 'ZTF18adaysgv', 'ZTF18adayxhm', 'ZTF18acrvdsj', 'ZTF18aaacbjk', 'ZTF18aabjhfp', 'ZTF18aczqzrj', 'ZTF18acurmhj', 'ZTF18aclcbvs', 'ZTF18acvggao', 'ZTF17aaantpu', 'ZTF18aafcibu', 'ZTF18acnbgca', 'ZTF18adbabsv', 'ZTF18abzikwz', 'ZTF18adbdhvb', 'ZTF18acywbgz', 'ZTF18adazblo', 'ZTF18adaylyq', 'ZTF18aczczav', 'ZTF18acaohsz', 'ZTF18aagpxsx', 'ZTF18adaykvg', 'ZTF18acwwkyu', 'ZTF18acyxxen', 'ZTF18accnbid', 'ZTF18adalarn', 'ZTF18acusnvl', 'ZTF18adaliyg', 'ZTF18acrhegn', 'ZTF18adayxxz', 'ZTF18acwzecq', 'ZTF18adalglq', 'ZTF18acxgoki', 'ZTF18acctgtf', 'ZTF18adarypz', 'ZTF18adaimlf', 'ZTF18adainmx', 'ZTF18acrhewe', 'ZTF18aahsyvy', 'ZTF18adbywzj', 'ZTF18adbzcry', 'ZTF18acvwmpw', 'ZTF18aczenvx', 'ZTF18aczeplb', 'ZTF18adazbol', 'ZTF18aavnedl', 'ZTF18adbifqw', 'ZTF18acuixmb', 'ZTF18adcbjec', 'ZTF18adbdngt', 'ZTF18adcbiuk', 'ZTF18aabeyio', 'ZTF18adccpmu', 'ZTF18aabgazt', 'ZTF18adbmbpa', 'ZTF18ackaxnw', 'ZTF18adaqipv', 'ZTF18acwtyfp', 'ZTF18adbdjij', 'ZTF18adbmkvr', 'ZTF18aczpdvz', 'ZTF18adcbymm', 'ZTF18adcbypn', 'ZTF18acsxwdi', 'ZTF18adcbytv', 'ZTF18adbdhni', 'ZTF18adbdhtn', 'ZTF18aczbkjn', 'ZTF18adccgir', 'ZTF19aaaasxu', 'ZTF18acwtydh', 'ZTF18acvrgwv', 'ZTF17aacfgoh', 'ZTF18aceulph', 'ZTF18acvqsht', 'ZTF18aczqtpt', 'ZTF18adailqf', 'ZTF18aabkjjk', 'ZTF18adcedvg', 'ZTF18adaymez', 'ZTF18acepuyx', 'ZTF18acbwfux', 'ZTF18aczevei', 'ZTF18adbhgwx', 'ZTF19aaacubf', 'ZTF19aaaegdw', 'ZTF18acuspbi', 'ZTF18adazigr', 'ZTF18adaktri', 'ZTF18acvgnft', 'ZTF18acwyaws', 'ZTF18acvgqsl', 'ZTF18acwwowy', 'ZTF19aaaeuxm', 'ZTF18adbabqs', 'ZTF18adaknbe', 'ZTF18aawngqo', 'ZTF19aaafgdr', 'ZTF18adbabqv', 'ZTF18adbabqc', 'ZTF18acsowfw', 'ZTF18acyxnyw', 'ZTF18acvwchg', 'ZTF18acyctqi', 'ZTF19aaafhyc', 'ZTF18aczuooo', 'ZTF18aajkgtr', 'ZTF19aaafhpj', 'ZTF19aaafhdx', 'ZTF18aczotxi', 'ZTF19aaafzah', 'ZTF18aczbgpm', 'ZTF18adaadmh', 'ZTF18adbmiyb', 'ZTF18adbcjlv', 'ZTF18abvmlow', 'ZTF18acxgnqj', 'ZTF18adbmbmh', 'ZTF18adbmbgf', 'ZTF19aaabtbz', 'ZTF18acsxkme', 'ZTF18adbcjkb', 'ZTF18aceatnu', 'ZTF18adbckox', 'ZTF18abwtbmg', 'ZTF18abwtbha', 'ZTF18actadvo', 'ZTF18acrflch', 'ZTF18actaqdw', 'ZTF18acebssd', 'ZTF18adbmrug', 'ZTF19aaaisig', 'ZTF18adasjei', 'ZTF18aaaasnn', 'ZTF18acytbjt', 'ZTF18acxhvzb', 'ZTF18adbcjin', 'ZTF17aadidxk', 'ZTF18adbdpln', 'ZTF18acvvroz', 'ZTF19aaadcbz', 'ZTF18adamhjn', 'ZTF18adasopt', 'ZTF18adaiunt', 'ZTF18acvwgfn', 'ZTF18aclrhwc', 'ZTF18aczdetq', 'ZTF18acvvpov', 'ZTF18acurjjx', 'ZTF18adayxje', 'ZTF18acnmlyy', 'ZTF19aaaonaw', 'ZTF18aaqlkgb', 'ZTF18actuhrs', 'ZTF18abgkjpd', 'ZTF18adbiftk', 'ZTF18acrnhtg', 'ZTF18adbadqp', 'ZTF18abedvgv', 'ZTF19aaapkgk', 'ZTF18adbabzm', 'ZTF18adbacwh', 'ZTF18acnoktl', 'ZTF19aaapnxn', 'ZTF18acyendr', 'ZTF18acnbpmd', 'ZTF18aczemzj', 'ZTF18aavjqvm', 'ZTF19aaapksl', 'ZTF18adbadid', 'ZTF18adalfyk', 'ZTF18adalfza', 'ZTF19aaapkhn', 'ZTF18adblnwv', 'ZTF18aczzrkn', 'ZTF19aaaanri', 'ZTF18abxrhmi', 'ZTF18acbxocm', 'ZTF18acdxfpu', 'ZTF18acrcdwn', 'ZTF18adcbyiw', 'ZTF19aaaamwp', 'ZTF18adbmhol', 'ZTF18adccxqw', 'ZTF18acrcgsi', 'ZTF19aabhkoh', 'ZTF19aaahulz', 'ZTF18adathpb', 'ZTF19aaahrff', 'ZTF19aaahryn', 'ZTF19aaaaxeu', 'ZTF18adbntwo', 'ZTF18adaruzw', 'ZTF17aadplpa', 'ZTF19aaajrme', 'ZTF18adasisj', 'ZTF18acvdcyf', 'ZTF19aaadfcp', 'ZTF19aaairqh', 'ZTF18aczwrcu', 'ZTF18acwutbr', 'ZTF18acrttfl', 'ZTF18acwtrfe', 'ZTF18abqzssj', 'ZTF18acsybmh', 'ZTF19aaaheft', 'ZTF18acxgobl', 'ZTF19aaahrno', 'ZTF18actafdw', 'ZTF19aaahsjp', 'ZTF18adbclkd', 'ZTF18acmtvsk', 'ZTF18acwwxjb', 'ZTF19aablomm', 'ZTF18acukgkk', 'ZTF18adbmmci', 'ZTF18acuauqm', 'ZTF18aczwdth', 'ZTF19aaadxmz', 'ZTF19aaaokub', 'ZTF19aaafldf', 'ZTF19aaafnhx', 'ZTF18adbhlet', 'ZTF19aabyrlk', 'ZTF19aabyheu', 'ZTF19aabyyhr', 'ZTF19aaafice', 'ZTF18acwwonp', 'ZTF18accvjxc', 'ZTF19aaaplqp', 'ZTF18acutfne', 'ZTF19aachpob', 'ZTF18adbijqe', 'ZTF19aacivvy', 'ZTF18acvilwk', 'ZTF18acwzawa', 'ZTF18acyyhgi', 'ZTF18adbijzc', 'ZTF18adcawvn', 'ZTF18acvqkxx', 'ZTF18adbdhwh', 'ZTF18acsxssp', 'ZTF19aacosqh', 'ZTF19aabhdko', 'ZTF18actzyzs', 'ZTF19aacpfqv', 'ZTF19aaabmng', 'ZTF18aczbkqp', 'ZTF18aceljis', 'ZTF19aaabmhd', 'ZTF19aaahoig', 'ZTF19aacolgi', 'ZTF18aczawvm', 'ZTF19aaaanyf', 'ZTF18acsyfnk', 'ZTF19aabgefm', 'ZTF19aacqeai', 'ZTF19aabhfdi', 'ZTF18acuxibg', 'ZTF18abzbprc', 'ZTF18acusegd', 'ZTF19aaaolce', 'ZTF18acvqugw', 'ZTF18acsmisj', 'ZTF18adazgdh', 'ZTF18aabazdr', 'ZTF19aacwljg', 'ZTF18aarbklo', 'ZTF18aajgowk', 'ZTF18acqywlx', 'ZTF18acyybvg', 'ZTF18acurklz', 'ZTF18adcakco', 'ZTF19aabyvwv', 'ZTF18acwzaui', 'ZTF19aabypir', 'ZTF19aabyppp', 'ZTF18achhdrz', 'ZTF18aczztqh', 'ZTF18adbcqwv', 'ZTF18adbmhzp', 'ZTF19aaahjhc', 'ZTF18adbcjdl', 'ZTF19aaaerbn', 'ZTF19aabyhwy', 'ZTF18aawohdr', 'ZTF19aadftkd', 'ZTF19aacilht']:
    # for name in ['ZTF18aaadpnd', 'ZTF18aaupmks', 'ZTF18aavqlnk', 'ZTF18aarcchg', 'ZTF18aajpjdi', 'ZTF18aaqcqvr', 'ZTF18aavmywq', 'ZTF18aaavscr', 'ZTF18aabpwwp', 'ZTF18aagzfzs', 'ZTF18aajrwen', 'ZTF18aaqehoc', 'ZTF18aavucmo', 'ZTF18aaunfqq', 'ZTF18aakctzv', 'ZTF18aahgakg', 'ZTF18aarzmtk', 'ZTF18aasttun', 'ZTF18aauvnhh', 'ZTF18aaveejk', 'ZTF18aaqedfj', 'ZTF18aavnvmp', 'ZTF18aaqctvg', 'ZTF18aajqcue', 'ZTF18aavnxar', 'ZTF18aaxkfos', 'ZTF18aaqqhis', 'ZTF18aacaxml', 'ZTF18aaccavf', 'ZTF18aataafd', 'ZTF18aasxvsg', 'ZTF18aauaayg', 'ZTF18aaxsakl', 'ZTF18aaxsbjy', 'ZTF18aaqgbzx', 'ZTF18aarpgds', 'ZTF18aaiyuqp', 'ZTF18aawtveo', 'ZTF18aaudfnk', 'ZTF18aajupnt', 'ZTF18aavsdxv', 'ZTF18aaxkjmm', 'ZTF18aatzyzn', 'ZTF18aaumrvc', 'ZTF18aaxavsk', 'ZTF17aaborap', 'ZTF18aawsamn', 'ZTF18aaxckpt', 'ZTF18aaxmnqg', 'ZTF18aaxxgnb', 'ZTF18aaxnqjb', 'ZTF18aaxnoxv', 'ZTF18aaxzhzf', 'ZTF18aatpisi', 'ZTF18aarpttw', 'ZTF18aauoloo', 'ZTF18aaxpixs', 'ZTF18aaizxwe', 'ZTF18aauduzx', 'ZTF18aawlhkh', 'ZTF18aaxvbmu', 'ZTF18aaxqfrr', 'ZTF18aavriwy', 'ZTF18aabduro', 'ZTF18aalpcph', 'ZTF18aajjokq', 'ZTF18aaxvpsw', 'ZTF18aaxmixe', 'ZTF18aaxxgkr', 'ZTF18aawvqma', 'ZTF18aacbjdd', 'ZTF18aamntui', 'ZTF18aarphti', 'ZTF18aaxxyhp', 'ZTF18aaowlig', 'ZTF18aauxtpt', 'ZTF18aaydtzf', 'ZTF18aaxkoez', 'ZTF18aajuewt', 'ZTF18aaydmkh', 'ZTF18aanhtig', 'ZTF18aaufprm', 'ZTF18aasadso', 'ZTF18aavrmcg', 'ZTF18aaxqort', 'ZTF18aawcwsx', 'ZTF18aayxupv', 'ZTF18aayivoq', 'ZTF18aaxqtps', 'ZTF18aaxrhwq', 'ZTF18aabxqge', 'ZTF18aakrnvd', 'ZTF18aaxxgic', 'ZTF18aayhzpp', 'ZTF18aaylbjm', 'ZTF18aatybbp', 'ZTF18aaziklu', 'ZTF18aaxjuwy', 'ZTF18aahmhxu', 'ZTF17aaarqox', 'ZTF18aarqxbw', 'ZTF18aaubcaj', 'ZTF18aapupwx', 'ZTF18aazlsvk', 'ZTF18aajumjz', 'ZTF18aanuveg', 'ZTF18aaojsvk', 'ZTF18aaupjtd', 'ZTF18aaxveek', 'ZTF18aaizyqk', 'ZTF18aakdzgg', 'ZTF18aaqfzoh', 'ZTF17aaawcwx', 'ZTF18aazwdak', 'ZTF17aabumwe', 'ZTF18abaphcx', 'ZTF17aaaedvh', 'ZTF18aayyimm', 'ZTF18aagvqkp', 'ZTF18aaylvql', 'ZTF18aanmmeg', 'ZTF18aaxtlox', 'ZTF18aavslda', 'ZTF18abbkixx', 'ZTF17aaaeuus', 'ZTF18abbodly', 'ZTF18aarltqh', 'ZTF18aayiqtf', 'ZTF17aacngdf', 'ZTF18abaqvko', 'ZTF18aayzpgg', 'ZTF18abaambh', 'ZTF18aaqzpne', 'ZTF18aamnrnj', 'ZTF18abbmbys', 'ZTF18abcbwnd', 'ZTF18abajspl', 'ZTF18aahkqok', 'ZTF18aayaarx', 'ZTF18aaxcfbt', 'ZTF18aabcdai', 'ZTF18aazhwnh', 'ZTF18abbfwtt', 'ZTF18aaunxpc', 'ZTF18abcezmh', 'ZTF18abcipbf', 'ZTF18aasnlpg', 'ZTF18aamftsi', 'ZTF18abbowqa', 'ZTF18abbuxcm', 'ZTF18abashqj', 'ZTF18abcreta', 'ZTF18abchhcn', 'ZTF18aazziao', 'ZTF18aaxljrh', 'ZTF18aazzfhx', 'ZTF18abcptmt', 'ZTF18aaqiklh', 'ZTF18abcrasg', 'ZTF18abashqr', 'ZTF18abblwsh', 'ZTF17aabwodp', 'ZTF18abcdzyc', 'ZTF18abcrzfz', 'ZTF18abcvush', 'ZTF18abawrxq', 'ZTF18abccpuw', 'ZTF18abdbuty', 'ZTF18abcxzqk', 'ZTF18aaseyca', 'ZTF18aansqun', 'ZTF18aajswer', 'ZTF18abcvufc', 'ZTF18abbvbru', 'ZTF18abalzzd', 'ZTF18abditme', 'ZTF18abcyawn', 'ZTF18abguhwj', 'ZTF18abehceu', 'ZTF18abfzkno', 'ZTF18abglpdy', 'ZTF18abgisys', 'ZTF18abcbvxc', 'ZTF18abespgb', 'ZTF18abeakbs', 'ZTF18aawurud', 'ZTF18abhhnnv', 'ZTF18aayezzl', 'ZTF18abcfcka', 'ZTF18abdkini', 'ZTF18abgulkc', 'ZTF18abhqdpv', 'ZTF18aascwkh', 'ZTF18abetewu', 'ZTF18abaqetc', 'ZTF18abflqxq', 'ZTF18abgitic', 'ZTF18aahbhvq', 'ZTF17aabvfha', 'ZTF18aapaohn', 'ZTF18abkjfzt', 'ZTF18abkjfzw', 'ZTF18abkhcrj', 'ZTF18aaofsmp', 'ZTF18abhakmn', 'ZTF18abjbnes', 'ZTF18aaqkdwf', 'ZTF18aavbdtl', 'ZTF18abjstcm', 'ZTF18ablhplf', 'ZTF18aaoxryq', 'ZTF18abjndls', 'ZTF18abfzixe', 'ZTF18abjvgwv', 'ZTF18abdzvgz', 'ZTF18aasgsbp', 'ZTF18abiirfq', 'ZTF18abklbgb', 'ZTF18abeibdj', 'ZTF17aaadjky', 'ZTF18ablowct', 'ZTF18abjwahi', 'ZTF18ablqfmf', 'ZTF18abeajml', 'ZTF18ablqlzp', 'ZTF18ablqqeb', 'ZTF18abkifng', 'ZTF18ablrlbm', 'ZTF18aayltmt', 'ZTF18abcpdns', 'ZTF18ablsxjo', 'ZTF18abkhcax', 'ZTF18abkhbrs', 'ZTF18abcfdzu', 'ZTF18aaxdddq', 'ZTF18ablongw', 'ZTF18abjhbss', 'ZTF18abeitow', 'ZTF18ablxdqz', 'ZTF18abfxhrt', 'ZTF18abgqvwv', 'ZTF18abgubgi', 'ZTF18abmjgyk', 'ZTF18abmjujg', 'ZTF18abjyjdz', 'ZTF18ablrljh', 'ZTF18abmbiza', 'ZTF18aamvand', 'ZTF18abmnwvc', 'ZTF18ablwbqn', 'ZTF17aaartnt', 'ZTF18ablszje', 'ZTF18aabesgz', 'ZTF18abmxfrc', 'ZTF17aabvcvr', 'ZTF18ablzvih', 'ZTF18ablsypo', 'ZTF18abbydbi', 'ZTF18aaymybb', 'ZTF18abmrhom', 'ZTF18abltbrd', 'ZTF18aanxtko', 'ZTF18abnnnzk', 'ZTF18abncizy', 'ZTF18abmdvcj', 'ZTF18abctyvq', 'ZTF18ablvjom', 'ZTF18abnchro', 'ZTF18aabqtsw', 'ZTF18aboemit', 'ZTF18abnugci', 'ZTF18abntxck', 'ZTF18aaqovfv', 'ZTF18abmogca', 'ZTF18abmwxvv', 'ZTF18abobkii', 'ZTF18abiitmq', 'ZTF18abahgfm', 'ZTF18aboaeqy', 'ZTF18abflcof', 'ZTF18abgugua', 'ZTF18abjgzxw', 'ZTF18abhozbd', 'ZTF18abeaymc', 'ZTF18abnuhuy', 'ZTF18abptsco', 'ZTF18abpttky', 'ZTF18abqjvyl', 'ZTF18abmkbqk', 'ZTF18abrzhik', 'ZTF18abnbhpu', 'ZTF18abccnyj', 'ZTF18abcctas', 'ZTF17aaazypn', 'ZTF18abncimo', 'ZTF18aabfmux', 'ZTF18absdeij', 'ZTF18aagrgbo', 'ZTF18abmwzpt', 'ZTF18aaymqbw', 'ZTF18ablmhjv', 'ZTF18abrlurr', 'ZTF18ablrlmu', 'ZTF18aamfybu', 'ZTF18abctdgj', 'ZTF18abokvkt', 'ZTF18abnygkb', 'ZTF18abmnvjc', 'ZTF18abrymgj', 'ZTF18aaakpjg', 'ZTF18abmddcb', 'ZTF18ablwzih', 'ZTF18abnvnqb', 'ZTF18abeachz', 'ZTF18abkvwzj', 'ZTF18aapgbfm', 'ZTF18ablpnuq', 'ZTF18abihxip', 'ZTF18aauherh', 'ZTF18abdlfev', 'ZTF18aarrqrn', 'ZTF18absyyig', 'ZTF18abdazfd', 'ZTF18absgnio', 'ZTF18abtdxke', 'ZTF18abteera', 'ZTF18abotcjv', 'ZTF18abotbuq', 'ZTF18abhnscs', 'ZTF18aawcesm', 'ZTF18abtjlsr', 'ZTF18abtjmns', 'ZTF18abnufuv', 'ZTF18abjmlty', 'ZTF18abbuxbk', 'ZTF18abryqnn', 'ZTF17aabvele', 'ZTF18absljwl', 'ZTF18abtnfnq', 'ZTF18abhowyg', 'ZTF18abscghc', 'ZTF18absqitc', 'ZTF18absqkfg', 'ZTF18abscyhs', 'ZTF18abnvklh', 'ZTF18abmqwgr', 'ZTF18abqblsk', 'ZTF18abshnwm', 'ZTF18abrzbnb', 'ZTF18abrzcdi', 'ZTF18abrqedj', 'ZTF18absrcps', 'ZTF18abslxpd', 'ZTF18abcwjii', 'ZTF18abuatdz', 'ZTF18abjlxwv', 'ZTF18abufdlx', 'ZTF18abskhcp', 'ZTF18abtsobl', 'ZTF18abtmbaz', 'ZTF18abtswjk', 'ZTF18abrzbuj', 'ZTF18aaabzcd', 'ZTF18absmqmw', 'ZTF18abeyoaa', 'ZTF18abdmghg', 'ZTF18abcwzhs', 'ZTF18abgxnrw', 'ZTF18abcgjpn', 'ZTF18abnwvmh', 'ZTF18abijktl', 'ZTF18abhpyor', 'ZTF18aborolm', 'ZTF18abswypg', 'ZTF18aaywflb', 'ZTF18abttaxw', 'ZTF18absbyio', 'ZTF18abotfiv', 'ZTF18abbqjbb', 'ZTF18absnayk', 'ZTF18abtnwpa', 'ZTF18abuqugw', 'ZTF18abthuby', 'ZTF18abukavn', 'ZTF18abuwxii', 'ZTF18abnzzvq', 'ZTF18abuatqi', 'ZTF18abnygwk', 'ZTF18abbuutk', 'ZTF18abufgdj', 'ZTF18abtswll', 'ZTF18abuhyjv', 'ZTF18abvgkcg', 'ZTF18aboztku', 'ZTF18aajsypz', 'ZTF18absnqyo', 'ZTF18aavqlqx', 'ZTF18abvivzm', 'ZTF18abugmrg', 'ZTF18abtphnh', 'ZTF17aaafrch', 'ZTF18abuhzfc', 'ZTF18abtfwhe', 'ZTF18abmjuya', 'ZTF18aagteda', 'ZTF18abeainm', 'ZTF18aakgzot', 'ZTF18abvfisg', 'ZTF18abvujhp', 'ZTF18abbmenz', 'ZTF18abdebec', 'ZTF18abvwsjn', 'ZTF18abtjrbt', 'ZTF18abskrcm', 'ZTF18aarsqmk', 'ZTF18abumdjo', 'ZTF18abnzksu', 'ZTF18abvfafa', 'ZTF18abwgaek', 'ZTF18abvzdvj', 'ZTF18abwkans', 'ZTF18abuqhje', 'ZTF17aabptoh', 'ZTF18abuuqkb', 'ZTF18abuvrnw', 'ZTF18abuxjrd', 'ZTF18abunhyu', 'ZTF18abwkhky', 'ZTF18abuahnw', 'ZTF18abtzzlv', 'ZTF18abvptts', 'ZTF18abvtbow', 'ZTF18abtnggb', 'ZTF18aazttzj', 'ZTF18abtogdl', 'ZTF18ablprcf', 'ZTF18aburkuo', 'ZTF18abxdtyz', 'ZTF18abehvba', 'ZTF18abslbyb', 'ZTF18abwkrvs', 'ZTF18abmwwni', 'ZTF17aacjaxh', 'ZTF18abvpnnb', 'ZTF18abqwixr', 'ZTF18abvrvdk', 'ZTF18abqaujs', 'ZTF17aabvwlq', 'ZTF18abltbxa', 'ZTF18abxyvxq', 'ZTF18abtlwoj', 'ZTF18abwfktb', 'ZTF18abuiytq', 'ZTF18abqyvzy', 'ZTF18aabgboh', 'ZTF17aabsopd', 'ZTF17aaatdgc', 'ZTF17aaaabds', 'ZTF18aamzpba', 'ZTF18abijaxd', 'ZTF18aaeyilr', 'ZTF18abuqoii', 'ZTF18abrwpvo', 'ZTF18abwlupf', 'ZTF18abolvzu', 'ZTF18absnxdb', 'ZTF17aaaeizh', 'ZTF18abvefxs', 'ZTF18abyfcns', 'ZTF18abyfvfr', 'ZTF18abchrps', 'ZTF18abufhtw', 'ZTF18acafztq', 'ZTF18abykeda', 'ZTF18abuifwq', 'ZTF18abtrlla', 'ZTF18abxftqm', 'ZTF17aacxpvk', 'ZTF18acaksoq', 'ZTF18abxsmlf', 'ZTF18ablvxdg', 'ZTF17aaaadiz', 'ZTF18acbvkwl', 'ZTF18acbvifx', 'ZTF18aadrhsi', 'ZTF18acbvzta', 'ZTF18acbwdwg', 'ZTF18abuszlm', 'ZTF18abmfwbn', 'ZTF18acbwhns', 'ZTF18acbwppb', 'ZTF18acbwwpm', 'ZTF18acbwxcc', 'ZTF18acbwxou', 'ZTF18abmefij', 'ZTF18acbxhua', 'ZTF18acbxkop', 'ZTF18abviyvv', 'ZTF18acbxsge', 'ZTF18abtlxdk', 'ZTF18abxmbjy', 'ZTF18abtbxou', 'ZTF18acbvmwr', 'ZTF17aadkbbu', 'ZTF18aaadyuc', 'ZTF18acbzqyi', 'ZTF18aaasgau', 'ZTF18abyfbhl', 'ZTF18acbwfvt', 'ZTF18aaemivw', 'ZTF18abwwdaz', 'ZTF18accdsxf', 'ZTF18accfxnn', 'ZTF18abtpite', 'ZTF18abmenfr', 'ZTF18acchzyo', 'ZTF18abvmkfh', 'ZTF18acckpro', 'ZTF18acbvbox', 'ZTF18abtswhs', 'ZTF18acbuibx', 'ZTF18acbvijf', 'ZTF18acclexy', 'ZTF18acbwopo', 'ZTF18acauwik', 'ZTF18acbyyff', 'ZTF18abwjzxr', 'ZTF18abmolpg', 'ZTF18acbvmzg', 'ZTF18aadkjmo', 'ZTF18abztrfp', 'ZTF18acbzpmu', 'ZTF18aaqkjuw', 'ZTF18aakeywu', 'ZTF18acbznkf', 'ZTF18aalpega', 'ZTF18acbzojv', 'ZTF18aayjflv', 'ZTF18acbzpgy', 'ZTF18aaujlmk', 'ZTF18acdvvhi', 'ZTF18acdways', 'ZTF18aaklpdo', 'ZTF18abasoah', 'ZTF18abzscns', 'ZTF18absqpbs', 'ZTF18accjecm', 'ZTF18acchhpe', 'ZTF18accicnc', 'ZTF18accjuwd', 'ZTF18acectzf', 'ZTF18accnnyu', 'ZTF18accnbgw', 'ZTF18acbznpp', 'ZTF18acbwfim', 'ZTF18abkwfxq', 'ZTF18abujubn', 'ZTF18aceijsp', 'ZTF18abgjezf', 'ZTF18acbxoie', 'ZTF18absgmhg', 'ZTF18acchlkx', 'ZTF18accvkpt', 'ZTF18accwihs', 'ZTF18acbzoxp', 'ZTF18accfjmp', 'ZTF18abudvzn', 'ZTF18abcplgw', 'ZTF17aaakeap', 'ZTF18accvflv', 'ZTF18aafxtrt', 'ZTF18accnlji', 'ZTF18accdtch', 'ZTF18abxjekh', 'ZTF18aceyzwi', 'ZTF18acfejol', 'ZTF18abxzsnv', 'ZTF17aaaruzy', 'ZTF18aaagyuv', 'ZTF18aaapvkn', 'ZTF18accpjkb', 'ZTF18abmrhkl', 'ZTF18acbwtng', 'ZTF18abupgps', 'ZTF18achavgj', 'ZTF18acbvmjq', 'ZTF17aacxkaq', 'ZTF18aceolcv', 'ZTF18acbzvqn', 'ZTF18aamsgjq', 'ZTF18acbzwna', 'ZTF18acbxnpg', 'ZTF18abqavrw', 'ZTF18abnczui', 'ZTF18acfwmqj', 'ZTF18abnzunr', 'ZTF17aacfeuw', 'ZTF17aaayvyb', 'ZTF18accsane', 'ZTF18acchhxu', 'ZTF18abtyrmh', 'ZTF18abcpddh', 'ZTF18abnzuqm', 'ZTF18aceaape', 'ZTF18acbudwt', 'ZTF18acdvahc', 'ZTF18acewwqf', 'ZTF18acbwdxy', 'ZTF18abttklf', 'ZTF18aavrleg', 'ZTF18acegbeo', 'ZTF18abxbnvq', 'ZTF18aawmppg', 'ZTF18acnbptb', 'ZTF18aahjvqd', 'ZTF18aamrkmd', 'ZTF18achdnip', 'ZTF18achdidy', 'ZTF18achdfqm', 'ZTF18acnngyx', 'ZTF18acpeesd', 'ZTF18aceyqpm', 'ZTF18acnbglg', 'ZTF18acdvqhn', 'ZTF18aceykmc', 'ZTF18acbzvzm', 'ZTF18acnbejq', 'ZTF18aathofv', 'ZTF18acppzle', 'ZTF18ackvfyc', 'ZTF18abbuzeu', 'ZTF18abnuwsj', 'ZTF18aaadsrj', 'ZTF18acefgee', 'ZTF18accvmgs', 'ZTF18acnbgpw', 'ZTF18aadezmv', 'ZTF18acnbgvd', 'ZTF18aabklju', 'ZTF18acpuici', 'ZTF18abwskcn', 'ZTF18acnnfup', 'ZTF18acaoelw', 'ZTF17aabuleq', 'ZTF18achlsqa', 'ZTF18abviwrt', 'ZTF18abvtcpk', 'ZTF18aabezcx', 'ZTF18acmzsjl', 'ZTF18acqxsyi', 'ZTF18acrdlrp', 'ZTF18achqdch', 'ZTF18acppwaf', 'ZTF18aaikyip', 'ZTF18acqsqry', 'ZTF18acnbgrk', 'ZTF18acppdfv', 'ZTF18acrukmr', 'ZTF18acrukrr', 'ZTF18abzfjaa', 'ZTF18acrvcqx', 'ZTF18aaeynun', 'ZTF17aaaoqmq', 'ZTF18acpuxuh', 'ZTF18acrxfwp', 'ZTF18aaehoph', 'ZTF17aaaocpm', 'ZTF18absltuj', 'ZTF18acrxiod', 'ZTF18aaacnpb', 'ZTF18aabeafu', 'ZTF18acgvysg', 'ZTF18aakchfv', 'ZTF18aazwtfj', 'ZTF18acnblbh', 'ZTF18acrtykk', 'ZTF18acsjjqr', 'ZTF18abmwubh', 'ZTF18abndmse', 'ZTF18aaadxfi', 'ZTF18acmypei', 'ZTF17aabhlvu', 'ZTF18abttkue', 'ZTF18acapgdp', 'ZTF18acrfvuf', 'ZTF17aaceblj', 'ZTF18acefxpq', 'ZTF18aaackvy', 'ZTF18acexyqt', 'ZTF17aacanek', 'ZTF18aaevdnv', 'ZTF18aafwicl', 'ZTF18abzpvwv', 'ZTF18acrctos', 'ZTF18acsxsmi', 'ZTF18acevgus', 'ZTF18acrciyr', 'ZTF18acrcizr', 'ZTF18acmyxfx', 'ZTF18acrvakb', 'ZTF17aabvtgd', 'ZTF18acrultm', 'ZTF18acsyzla', 'ZTF18acsxrge', 'ZTF18acrukpd', 'ZTF18acenyfr', 'ZTF18acrlmvt', 'ZTF18aaadsdv', 'ZTF17aabciaf', 'ZTF17aaaoiei', 'ZTF18acujaic', 'ZTF18absqivc', 'ZTF18acurbia', 'ZTF18acuring', 'ZTF18acueemg', 'ZTF18acrdtzn', 'ZTF18acuqyyn', 'ZTF18acuqxyf', 'ZTF18acuqskr', 'ZTF17aaahyqu', 'ZTF18acuschq', 'ZTF18acpovry', 'ZTF18acszxqc', 'ZTF18acsouto', 'ZTF18aajhfdm', 'ZTF18acpeemg', 'ZTF18acpgetv', 'ZTF18abnzrgv', 'ZTF18actxhgn', 'ZTF18acrwgfp', 'ZTF18abrejjg', 'ZTF18acvddao', 'ZTF18acubbwb', 'ZTF17aaagtee', 'ZTF18acrvldh', 'ZTF18abuyhea', 'ZTF18acurwua', 'ZTF18acrgunw', 'ZTF18acpqahp', 'ZTF18aawxcbp', 'ZTF18aawmxdx', 'ZTF18acurljz', 'ZTF18acrxgbu', 'ZTF18acsowoc', 'ZTF18acueeoo', 'ZTF18acrdtwo', 'ZTF18acebicb', 'ZTF18acslbve', 'ZTF18acvvudh', 'ZTF18acvwchw', 'ZTF18acvwcky', 'ZTF18aaiuynw', 'ZTF18acvwcms', 'ZTF18acvwddd', 'ZTF18acvvved', 'ZTF18acsyanq', 'ZTF18acrexje', 'ZTF18acwhaxj', 'ZTF18acszixl', 'ZTF18acvvuzi', 'ZTF18acvwgqy', 'ZTF18acvwgdu', 'ZTF18acvwbos', 'ZTF18acidwhi', 'ZTF17aaanppf', 'ZTF18acnnxdy', 'ZTF18acubbuv', 'ZTF18acwyvhw', 'ZTF18acuewyy', 'ZTF18aaquudq', 'ZTF18abxdqxa', 'ZTF18abtpzai', 'ZTF18acxgujm', 'ZTF18acetnxh', 'ZTF18abvzkyr', 'ZTF18acszgpa', 'ZTF18ablwmzz', 'ZTF17aabwgbf', 'ZTF18aajpbkx', 'ZTF18aavmxyf', 'ZTF18aaavqpw', 'ZTF18aabahuy', 'ZTF18acpeekw', 'ZTF18aawonvg', 'ZTF18acynmcp', 'ZTF18abmjnkk', 'ZTF18acsxvpj', 'ZTF18aczawcw', 'ZTF18acxijqw', 'ZTF18actzpxw', 'ZTF18acrunrq', 'ZTF18acxyarg', 'ZTF18acvwhaz', 'ZTF18aczcsmg', 'ZTF18acvgugm', 'ZTF18aclqiwo', 'ZTF18acvgzab', 'ZTF18aaanzns', 'ZTF18aczeraw', 'ZTF18aczerlj', 'ZTF18acuswcl', 'ZTF18acrxtrv', 'ZTF18acynlmf', 'ZTF18acwugwx', 'ZTF18acybdar', 'ZTF18acvgxmp', 'ZTF17aaaplzd', 'ZTF17aabtawv', 'ZTF18acbyeid', 'ZTF18acwtykp', 'ZTF18acvgwbt', 'ZTF18aczeqva', 'ZTF18achyngk', 'ZTF18acvdstg', 'ZTF18abujbvb', 'ZTF18adaysgv', 'ZTF18acrvdsj', 'ZTF18aaacbjk', 'ZTF18aabjhfp', 'ZTF18acurmhj', 'ZTF18aclcbvs', 'ZTF18acvggao', 'ZTF17aaantpu', 'ZTF18aafcibu', 'ZTF18acnbgca', 'ZTF18abzikwz', 'ZTF18acywbgz', 'ZTF17aabvxzo', 'ZTF18adaylyq', 'ZTF18acaohsz', 'ZTF18acyxxen', 'ZTF18accnbid', 'ZTF18adalarn', 'ZTF18adaliyg', 'ZTF18acrhegn', 'ZTF18acwzecq', 'ZTF18adalglq', 'ZTF18acctgtf', 'ZTF18adaimlf', 'ZTF18aahsyvy', 'ZTF18adbywzj', 'ZTF18aczenvx', 'ZTF18adazbol', 'ZTF17aaaplrc', 'ZTF18aavnedl', 'ZTF18adbifqw', 'ZTF18abtlwpp', 'ZTF18aabeyio', 'ZTF18aabgazt', 'ZTF18acwtyfp', 'ZTF18adbmkvr', 'ZTF18aczpdvz', 'ZTF18adcbymm', 'ZTF18adcbytv', 'ZTF18adbdhni', 'ZTF18adbdhtn', 'ZTF18acvrgwv', 'ZTF17aacfgoh', 'ZTF18aceulph', 'ZTF18acvqsht', 'ZTF18aabkjjk', 'ZTF18adcedvg', 'ZTF17aaapffi', 'ZTF18adaymez', 'ZTF18acepuyx', 'ZTF18acbwfux', 'ZTF18aczevei', 'ZTF19aaacubf', 'ZTF19aaaegdw', 'ZTF18acuspbi', 'ZTF18adazigr', 'ZTF18adaktri', 'ZTF18aabcxog', 'ZTF18acvgqsl', 'ZTF18adbabqs', 'ZTF18aawngqo', 'ZTF18aajmlnp', 'ZTF19aaafgdr', 'ZTF18adbabqv', 'ZTF18acsowfw', 'ZTF18acyxnyw', 'ZTF19aaafhyc', 'ZTF18aajkgtr', 'ZTF18aaavdos', 'ZTF19aaafhdx', 'ZTF19aaafzah', 'ZTF18adaadmh', 'ZTF18adbmiyb', 'ZTF18adbmbgf', 'ZTF18acsxkme', 'ZTF18abxfjtd', 'ZTF18adbcjkb', 'ZTF18aceatnu', 'ZTF18abwtbmg', 'ZTF18abwtbha', 'ZTF18actadvo', 'ZTF18abuanlw', 'ZTF18acebssd', 'ZTF18adasjei', 'ZTF18acytbjt', 'ZTF18acxhvzb', 'ZTF18adbcjin', 'ZTF18adamhjn', 'ZTF18aclrhwc', 'ZTF18aczdetq', 'ZTF18acvvpov', 'ZTF18acurjjx', 'ZTF18abgkjpd', 'ZTF18acrnhtg', 'ZTF18abedvgv', 'ZTF18aazabua', 'ZTF18adbabzm', 'ZTF18abdgczy', 'ZTF18adbacwh', 'ZTF18acnoktl', 'ZTF19aaapnxn', 'ZTF18acyendr', 'ZTF18aainmon', 'ZTF18aavjqvm', 'ZTF18adbadid', 'ZTF18adalfyk', 'ZTF18abxbjnr', 'ZTF19aaaanri', 'ZTF18abxrhmi', 'ZTF18abxmxtq', 'ZTF18acbxocm', 'ZTF19aaahrff', 'ZTF19aaahryn', 'ZTF19aaaaxeu', 'ZTF17aabgout', 'ZTF18acvdcyf', 'ZTF18acrttfl', 'ZTF18abqzssj', 'ZTF18acsybmh', 'ZTF19aaahrno', 'ZTF18actafdw', 'ZTF18abyydte', 'ZTF19aaahsjp', 'ZTF18acmtvsk', 'ZTF18acukgkk', 'ZTF18adbmmci', 'ZTF18aczwdth', 'ZTF19aaaokub', 'ZTF19aaafldf', 'ZTF19aaafnhx', 'ZTF18aayxuty', 'ZTF18aaccneo', 'ZTF19aabyyhr', 'ZTF19aaafice', 'ZTF18accvjxc', 'ZTF18absrlhv', 'ZTF19aaaplqp', 'ZTF18aatinzo', 'ZTF19aacivvy', 'ZTF18acwzawa', 'ZTF18aagvqxm', 'ZTF18acvqkxx', 'ZTF18adbdhwh', 'ZTF18acsxssp', 'ZTF19aacosqh', 'ZTF18actzyzs', 'ZTF18aceljis', 'ZTF19aaabmhd', 'ZTF19aacolgi', 'ZTF18aczawvm', 'ZTF19aaaanyf', 'ZTF18acsyfnk', 'ZTF19aabgefm', 'ZTF19aacqeai', 'ZTF19aabhfdi', 'ZTF18abzbprc', 'ZTF19aaaolce', 'ZTF18acvqugw', 'ZTF18acsmisj', 'ZTF18adazgdh', 'ZTF18aabazdr', 'ZTF18aarbklo', 'ZTF18aajgowk', 'ZTF18acqywlx', 'ZTF18acurklz', 'ZTF19aabyvwv', 'ZTF19aabyppp', 'ZTF18achhdrz', 'ZTF19aaahjhc', 'ZTF18adbcjdl', 'ZTF18aawohdr', 'ZTF18acuepcb', 'ZTF19aaafoaf', 'ZTF19aabzgoc', 'ZTF19aaapvoy', 'ZTF18abttnhc', 'ZTF18aahegog', 'ZTF19aadoevn', 'ZTF19aacxvyw', 'ZTF19aacivld', 'ZTF19aadyiao', 'ZTF19aacwqjs', 'ZTF19aadomgl', 'ZTF19aaeoqkk', 'ZTF18aacdbzx', 'ZTF19aadgmuq', 'ZTF19aadnxkc', 'ZTF19aaapjps', 'ZTF18abtuote', 'ZTF17aacedyd', 'ZTF18acqighv', 'ZTF18acepxke', 'ZTF18aaisvqg', 'ZTF18aaisjkm', 'ZTF19aaeomsn', 'ZTF19aadgceq', 'ZTF19aaeopbq', 'ZTF19aaciwam', 'ZTF19aadohat', 'ZTF19aaejtrn', 'ZTF18acaqdaa', 'ZTF18abcgryu', 'ZTF18aboaavg', 'ZTF18acakxfh', 'ZTF19aafmidf', 'ZTF19aacsojb', 'ZTF18acegcvd', 'ZTF18aaofdgl', 'ZTF19aailmac', 'ZTF19aailsad', 'ZTF19aailvhf', 'ZTF18acvdwhd', 'ZTF19aaaemtp', 'ZTF18acduygl', 'ZTF19aacsofi', 'ZTF19aaikxuj', 'ZTF19aaaplph', 'ZTF19aafmyow', 'ZTF18aaqoukh', 'ZTF19aaapljd']:
    #     try:
    #         classify_lasair_light_curves(object_name=name)
    #     except Exception as e:
    #         print(e)



