"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResponseType = exports.GatewayResponse = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
/**
 * Configure the response received by clients, produced from the API Gateway backend.
 *
 * @stability stable
 * @resource AWS::ApiGateway::GatewayResponse
 */
class GatewayResponse extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const gatewayResponseProps = {
            restApiId: props.restApi.restApiId,
            responseType: props.type.responseType,
            responseParameters: this.buildResponseParameters(props.responseHeaders),
            responseTemplates: props.templates,
            statusCode: props.statusCode,
        };
        const resource = new apigateway_generated_1.CfnGatewayResponse(this, 'Resource', gatewayResponseProps);
        const deployment = props.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({
                gatewayResponse: {
                    ...gatewayResponseProps,
                },
            });
        }
        this.node.defaultChild = resource;
    }
    buildResponseParameters(responseHeaders) {
        if (!responseHeaders) {
            return undefined;
        }
        const responseParameters = {};
        for (const [header, value] of Object.entries(responseHeaders)) {
            responseParameters[`gatewayresponse.header.${header}`] = value;
        }
        return responseParameters;
    }
}
exports.GatewayResponse = GatewayResponse;
/**
 * Supported types of gateway responses.
 *
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html
 * @stability stable
 */
class ResponseType {
    constructor(type) {
        this.responseType = type;
    }
    /**
     * A custom response type to suppport future cases.
     *
     * @stability stable
     */
    static of(type) {
        return new ResponseType(type.toUpperCase());
    }
}
exports.ResponseType = ResponseType;
/**
 * The gateway response for authorization failure.
 *
 * @stability stable
 */
ResponseType.ACCESS_DENIED = new ResponseType('ACCESS_DENIED');
/**
 * The gateway response for an invalid API configuration.
 *
 * @stability stable
 */
ResponseType.API_CONFIGURATION_ERROR = new ResponseType('API_CONFIGURATION_ERROR');
/**
 * The gateway response when a custom or Amazon Cognito authorizer failed to authenticate the caller.
 *
 * @stability stable
 */
ResponseType.AUTHORIZER_FAILURE = new ResponseType('AUTHORIZER_FAILURE');
/**
 * The gateway response for failing to connect to a custom or Amazon Cognito authorizer.
 *
 * @stability stable
 */
ResponseType.AUTHORIZER_CONFIGURATION_ERROR = new ResponseType('AUTHORIZER_CONFIGURATION_ERROR');
/**
 * The gateway response when the request parameter cannot be validated according to an enabled request validator.
 *
 * @stability stable
 */
ResponseType.BAD_REQUEST_PARAMETERS = new ResponseType('BAD_REQUEST_PARAMETERS');
/**
 * The gateway response when the request body cannot be validated according to an enabled request validator.
 *
 * @stability stable
 */
ResponseType.BAD_REQUEST_BODY = new ResponseType('BAD_REQUEST_BODY');
/**
 * The default gateway response for an unspecified response type with the status code of 4XX.
 *
 * @stability stable
 */
ResponseType.DEFAULT_4XX = new ResponseType('DEFAULT_4XX');
/**
 * The default gateway response for an unspecified response type with a status code of 5XX.
 *
 * @stability stable
 */
ResponseType.DEFAULT_5XX = new ResponseType('DEFAULT_5XX');
/**
 * The gateway response for an AWS authentication token expired error.
 *
 * @stability stable
 */
ResponseType.EXPIRED_TOKEN = new ResponseType('EXPIRED_TOKEN');
/**
 * The gateway response for an invalid AWS signature error.
 *
 * @stability stable
 */
ResponseType.INVALID_SIGNATURE = new ResponseType('INVALID_SIGNATURE');
/**
 * The gateway response for an integration failed error.
 *
 * @stability stable
 */
ResponseType.INTEGRATION_FAILURE = new ResponseType('INTEGRATION_FAILURE');
/**
 * The gateway response for an integration timed out error.
 *
 * @stability stable
 */
ResponseType.INTEGRATION_TIMEOUT = new ResponseType('INTEGRATION_TIMEOUT');
/**
 * The gateway response for an invalid API key submitted for a method requiring an API key.
 *
 * @stability stable
 */
ResponseType.INVALID_API_KEY = new ResponseType('INVALID_API_KEY');
/**
 * The gateway response for a missing authentication token error, including the cases when the client attempts to invoke an unsupported API method or resource.
 *
 * @stability stable
 */
ResponseType.MISSING_AUTHENTICATION_TOKEN = new ResponseType('MISSING_AUTHENTICATION_TOKEN');
/**
 * The gateway response for the usage plan quota exceeded error.
 *
 * @stability stable
 */
ResponseType.QUOTA_EXCEEDED = new ResponseType('QUOTA_EXCEEDED');
/**
 * The gateway response for the request too large error.
 *
 * @stability stable
 */
ResponseType.REQUEST_TOO_LARGE = new ResponseType('REQUEST_TOO_LARGE');
/**
 * The gateway response when API Gateway cannot find the specified resource after an API request passes authentication and authorization.
 *
 * @stability stable
 */
ResponseType.RESOURCE_NOT_FOUND = new ResponseType('RESOURCE_NOT_FOUND');
/**
 * The gateway response when usage plan, method, stage, or account level throttling limits exceeded.
 *
 * @stability stable
 */
ResponseType.THROTTLED = new ResponseType('THROTTLED');
/**
 * The gateway response when the custom or Amazon Cognito authorizer failed to authenticate the caller.
 *
 * @stability stable
 */
ResponseType.UNAUTHORIZED = new ResponseType('UNAUTHORIZED');
/**
 * The gateway response when a payload is of an unsupported media type, if strict passthrough behavior is enabled.
 *
 * @stability stable
 */
ResponseType.UNSUPPORTED_MEDIA_TYPE = new ResponseType('UNSUPPORTED_MEDIA_TYPE');
/**
 * The gateway response when a request is blocked by AWS WAF.
 *
 * @stability stable
 */
ResponseType.WAF_FILTERED = new ResponseType('WAF_FILTERED');
//# sourceMappingURL=data:application/json;base64,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