"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_canned_metrics_generated_1 = require("./apigateway-canned-metrics.generated");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi.
 *
 * @stability stable
 */
class RestApiBase extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        this.restApiName = (_a = props.restApiName) !== null && _a !== void 0 ? _a : id;
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     *
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     *
     * @stability stable
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName` configuration prop, or added via `addDomainName`.
     *
     * @stability stable
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     *
     * @stability stable
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     *
     * @param id The construct id.
     * @param options custom domain options.
     * @stability stable
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     *
     * @stability stable
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * Gets the "execute-api" ARN.
     *
     * @stability stable
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     *
     * @stability stable
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Add an ApiKey.
     *
     * @stability stable
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Returns the given named metric for this API.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApiGateway',
            metricName,
            dimensions: { ApiName: this.restApiName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of client-side errors captured in a given period.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricClientError(props) {
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics._4XxErrorSum, props);
    }
    /**
     * Metric for the number of server-side errors captured in a given period.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricServerError(props) {
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics._5XxErrorSum, props);
    }
    /**
     * Metric for the number of requests served from the API cache in a given period.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricCacheHitCount(props) {
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.cacheHitCountSum, props);
    }
    /**
     * Metric for the number of requests served from the backend in a given period, when API caching is enabled.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricCacheMissCount(props) {
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.cacheMissCountSum, props);
    }
    /**
     * Metric for the total number API requests in a given period.
     *
     * Default: samplecount over 5 minutes
     *
     * @stability stable
     */
    metricCount(props) {
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.countSum, {
            statistic: 'SampleCount',
            ...props,
        });
    }
    /**
     * Metric for the time between when API Gateway relays a request to the backend and when it receives a response from the backend.
     *
     * Default: average over 5 minutes.
     *
     * @stability stable
     */
    metricIntegrationLatency(props) {
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.integrationLatencyAverage, props);
    }
    /**
     * The time between when API Gateway receives a request from a client and when it returns a response to the client.
     *
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * Default: average over 5 minutes.
     *
     * @stability stable
     */
    metricLatency(props) {
        return this.cannedMetric(apigateway_canned_metrics_generated_1.ApiGatewayMetrics.latencyAverage, props);
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    /**
     * Associates a Deployment resource with this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        ignore(deployment);
    }
    /**
     * @stability stable
     */
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    /**
     * @stability stable
     */
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
    /**
     * @internal
     */
    _configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ApiName: this.restApiName }),
            ...props,
        }).attachTo(this);
    }
}
exports.RestApiBase = RestApiBase;
/**
 * (experimental) Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            endpointConfiguration: this._configureEndpoints(props),
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, {}, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @stability stable
 */
class RestApi extends RestApiBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi.
         *
         * @stability stable
         */
        this.methods = new Array();
        /**
         * This list of deployments bound to this RestApi
         */
        this.deployments = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this._configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     *
     * @stability stable
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing RestApi that can be configured with additional Methods and Resources.
     *
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     *
     * @stability stable
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Adds a new model.
     *
     * @stability stable
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     *
     * @stability stable
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
        // add this method as a dependency to all deployments defined for this api
        // when additional deployments are added, _attachDeployment is called and
        // this method will be added there.
        for (const dep of this.deployments) {
            dep._addMethodDependency(method);
        }
    }
    /**
     * Attaches a deployment to this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        this.deployments.push(deployment);
        // add all methods that were already defined as dependencies of this
        // deployment when additional methods are added, _attachMethod is called and
        // it will be added as a dependency to this deployment.
        for (const method of this.methods) {
            deployment._addMethodDependency(method);
        }
    }
    /**
     * Performs validation of the REST API.
     *
     * @stability stable
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
}
exports.RestApi = RestApi;
/**
 * @stability stable
 */
var ApiKeySourceType;
(function (ApiKeySourceType) {
    ApiKeySourceType["HEADER"] = "HEADER";
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
/**
 * @stability stable
 */
var EndpointType;
(function (EndpointType) {
    EndpointType["EDGE"] = "EDGE";
    EndpointType["REGIONAL"] = "REGIONAL";
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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