"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.MethodLoggingLevel = void 0;
const core_1 = require("@aws-cdk/core");
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * @stability stable
 */
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
/**
 * @stability stable
 */
class Stage extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined &&
                !core_1.Token.isUnresolved(accessLogFormat.toString()) &&
                !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     *
     * @param path The resource path.
     * @stability stable
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod,
                resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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