# -*- coding: utf-8 -*-
#
# Copyright (c) 2019-2021 Kevin De Bruycker and Tim Krappitz
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#

from collections import Counter

# Masses of proton and electron in u, obtained from (11/10/2018):
# https://physics.nist.gov/cuu/Constants/index.html
proton = 1.007276466879
electron = 0.000548579909070

ionising_species = { # '<element+charge>': [formula: Counter, charge: int, mass_correction: str]
    # Mass correction uses the virtual elements E for +electron or Me for -electron
    'none': [Counter(),
             1,
             ''],
    '+':    [Counter(),
             1,
             'Me'],
    'H+':   [Counter(dict(H=1)),
             1,
             'Me'],
    'NH4+': [Counter(dict(N=1, H=4)),
             1,
             'Me'],
    'Na+':  [Counter(dict(Na=1)),
             1,
             'Me'],
    'K+':   [Counter(dict(K=1)),
             1,
             'Me'],
    '-':    [Counter(),
             -1,
             'E'],
    '-H+':  [Counter(dict(H=-1)),
             -1,
             'E'],
    'Cl-':  [Counter(dict(Cl=1)),
             -1,
             'E'],
    'I-':   [Counter(dict(I=1)),
             -1,
             'E'],
}

monomers = {
    'EO':   Counter(dict(C=2, H=4, O=1)),
    'PO':   Counter(dict(C=3, H=6, O=1)),
    'MA':   Counter(dict(C=4, H=6, O=2)),
    'tBuA': Counter(dict(C=7, H=12, O=2)),
    'MMA':  Counter(dict(C=5, H=8, O=2)),
    'EMA':  Counter(dict(C=6, H=10, O=2)),
    'BMA':  Counter(dict(C=8, H=14, O=2)),
}

endgroups = {
    'BrH':      [Counter(dict(H=1)), Counter(dict(Br=1))],
    'EBr':      [Counter(dict(C=6, H=11, O=2)), Counter(dict(Br=1))],
    'BrD':      [Counter(dict(C=5, H=7, O=2)), Counter(dict(Br=1))],
    'ED':       [Counter(dict(C=6, H=11, O=2)), Counter(dict(C=5, H=7, O=2))],
    'EE':       [Counter(dict(C=6, H=11, O=2)), Counter(dict(C=6, H=11, O=2))],
    'EH':       [Counter(dict(C=6, H=11, O=2)), Counter(dict(H=1))],
    'EOH':      [Counter(dict(C=6, H=11, O=2)), Counter(dict(H=1, O=1))],
    'MMAD':     [Counter(dict(C=5, H=9, O=2)), Counter(dict(C=5, H=7, O=2))],
    'MMAH':     [Counter(dict(C=5, H=9, O=2)), Counter(dict(H=1))],
    'EMAD':     [Counter(dict(C=6, H=11, O=2)), Counter(dict(C=6, H=9, O=2))],
    'EMAH':     [Counter(dict(C=6, H=11, O=2)), Counter(dict(H=1))],
    'BMAD':     [Counter(dict(C=8, H=15, O=2)), Counter(dict(C=8, H=13, O=2))],
    'BMAH':     [Counter(dict(C=8, H=15, O=2)), Counter(dict(H=1))],
}

isotopic_abundances = {
    # 'Element' : [
    #     (mass, abundance),
    # ],
    #
    # Data obtained from (11/10/2018):
    # https://www.nist.gov/pml/atomic-weights-and-isotopic-compositions-relative-atomic-masses
    # Quote:
    # """
    # Developers and Contributors:
    # J. S. Coursey, D. J. Schwab, J. J. Tsai, and R. A. Dragoset
    # NIST Physical Measurement Laboratory
    #
    # The atomic weights are available for elements 1 through 118 and isotopic compositions or abundances are given
    # when appropriate. The atomic weights data were published by J. Meija et al in Atomic Weights of the Elements
    # 2013 (http://dx.doi.org/10.1515/pac-2015-0305), and the isotopic compositions data were published by M.
    # Berglund and M.E. Wieser in Isotopic Compositions of the Elements 2009
    # (http://www.ciaaw.org/pubs/TICE-2009.pdf). The relative atomic masses of the isotopes data were published by
    # M. Wang, G. Audi, A.H. Wapstra, F.G. Kondev, M. MacCormick, X. Xu1, and B. Pfeiffer in The AME2012 Atomic Mass
    # Evaluation.
    #
    # These data have been compiled from the above sources for the user's convenience and does not represent a
    # critical evaluation by the NIST Physical Measurement Laboratory.
    # """

    'H' : [
        (1.00782503223, 0.999885),
        (2.01410177812, 0.000115),
    ],
    'He' : [
        (3.0160293201, 0.00000134),
        (4.00260325413, 0.99999866),
    ],
    'Li' : [
        (6.0151228874, 0.0759),
        (7.0160034366, 0.9241),
    ],
    'Be' : [
        (9.012183065, 1.0000),
    ],
    'B' : [
        (10.01293695, 0.199),
        (11.00930536, 0.801),
    ],
    'C' : [
        (12.0000000, 0.9893),
        (13.00335483507, 0.0107),
    ],
    'N' : [
        (14.00307400443, 0.99636),
        (15.00010889888, 0.00364),
    ],
    'O' : [
        (15.99491461957, 0.99757),
        (16.99913175650, 0.00038),
        (17.99915961286, 0.00205),
    ],
    'F' : [
        (18.99840316273, 1.0000),
    ],
    'Ne' : [
        (19.9924401762, 0.9048),
        (20.993846685, 0.0027),
        (21.991385114, 0.0925),
    ],
    'Na' : [
        (22.9897692820, 1.0000),
    ],
    'Mg' : [
        (23.985041697, 0.7899),
        (24.985836976, 0.1000),
        (25.982592968, 0.1101),
    ],
    'Al' : [
        (26.98153853, 1.0000),
    ],
    'Si' : [
        (27.97692653465, 0.92223),
        (28.97649466490, 0.04685),
        (29.973770136, 0.03092),
    ],
    'P' : [
        (30.97376199842, 1.0000),
    ],
    'S' : [
        (31.9720711744, 0.9499),
        (32.9714589098, 0.0075),
        (33.967867004, 0.0425),
        (35.96708071, 0.0001),
    ],
    'Cl' : [
        (34.968852682, 0.7576),
        (36.965902602, 0.2424),
    ],
    'Ar' : [
        (35.967545105, 0.003336),
        (37.96273211, 0.000629),
        (39.9623831237, 0.996035),
    ],
    'K' : [
        (38.9637064864, 0.932581),
        (39.963998166, 0.000117),
        (40.9618252579, 0.067302),
    ],
    'Ca' : [
        (39.962590863, 0.96941),
        (41.95861783, 0.00647),
        (42.95876644, 0.00135),
        (43.95548156, 0.02086),
        (45.9536890, 0.00004),
        (47.95252276, 0.00187),
    ],
    'Sc' : [
        (44.95590828, 1.0000),
    ],
    'Ti' : [
        (45.95262772, 0.0825),
        (46.95175879, 0.0744),
        (47.94794198, 0.7372),
        (48.94786568, 0.0541),
        (49.94478689, 0.0518),
    ],
    'V' : [
        (49.94715601, 0.00250),
        (50.94395704, 0.99750),
    ],
    'Cr' : [
        (49.94604183, 0.04345),
        (51.94050623, 0.83789),
        (52.94064815, 0.09501),
        (53.93887916, 0.02365),
    ],
    'Mn' : [
        (54.93804391, 1.0000),
    ],
    'Fe' : [
        (53.93960899, 0.05845),
        (55.93493633, 0.91754),
        (56.93539284, 0.02119),
        (57.93327443, 0.00282),
    ],
    'Co' : [
        (58.93319429, 1.0000),
    ],
    'Ni' : [
        (57.93534241, 0.68077),
        (59.93078588, 0.26223),
        (60.93105557, 0.011399),
        (61.92834537, 0.036346),
        (63.92796682, 0.009255),
    ],
    'Cu' : [
        (62.92959772, 0.6915),
        (64.92778970, 0.3085),
    ],
    'Zn' : [
        (63.92914201, 0.4917),
        (65.92603381, 0.2773),
        (66.92712775, 0.0404),
        (67.92484455, 0.1845),
        (69.9253192, 0.0061),
    ],
    'Ga' : [
        (68.9255735, 0.60108),
        (70.92470258, 0.39892),
    ],
    'Ge' : [
        (69.92424875, 0.2057),
        (71.922075826, 0.2745),
        (72.923458956, 0.0775),
        (73.921177761, 0.3650),
        (75.921402726, 0.0773),
    ],
    'As' : [
        (74.92159457, 1.0000),
    ],
    'Se' : [
        (73.922475934, 0.0089),
        (75.919213704, 0.0937),
        (76.919914154, 0.0763),
        (77.91730928, 0.2377),
        (79.9165218, 0.4961),
        (81.9166995, 0.0873),
    ],
    'Br' : [
        (78.9183376, 0.5069),
        (80.9162897, 0.4931),
    ],
    'Kr' : [
        (77.92036494, 0.00355),
        (79.91637808, 0.02286),
        (81.91348273, 0.11593),
        (82.91412716, 0.11500),
        (83.9114977282, 0.56987),
        (85.9106106269, 0.17279),
    ],
    'Rb' : [
        (84.9117897379, 0.7217),
        (86.9091805310, 0.2783),
    ],
    'Sr' : [
        (83.9134191, 0.0056),
        (85.9092606, 0.0986),
        (86.9088775, 0.0700),
        (87.9056125, 0.8258),
    ],
    'Y' : [
        (88.9058403, 1.0000),
    ],
    'Zr' : [
        (89.9046977, 0.5145),
        (90.9056396, 0.1122),
        (91.9050347, 0.1715),
        (93.9063108, 0.1738),
        (95.9082714, 0.0280),
    ],
    'Nb' : [
        (92.9063730, 1.0000),
    ],
    'Mo' : [
        (91.90680796, 0.1453),
        (93.90508490, 0.0915),
        (94.90583877, 0.1584),
        (95.90467612, 0.1667),
        (96.90601812, 0.0960),
        (97.90540482, 0.2439),
        (99.9074718, 0.0982),
    ],
    'Ru' : [
        (95.90759025, 0.0554),
        (97.9052868, 0.0187),
        (98.9059341, 0.1276),
        (99.9042143, 0.1260),
        (100.9055769, 0.1706),
        (101.9043441, 0.3155),
        (103.9054275, 0.1862),
    ],
    'Rh' : [
        (102.9054980, 1.0000),
    ],
    'Pd' : [
        (101.9056022, 0.0102),
        (103.9040305, 0.1114),
        (104.9050796, 0.2233),
        (105.9034804, 0.2733),
        (107.9038916, 0.2646),
        (109.90517220, 0.1172),
    ],
    'Ag' : [
        (106.9050916, 0.51839),
        (108.9047553, 0.48161),
    ],
    'Cd' : [
        (105.9064599, 0.0125),
        (107.9041834, 0.0089),
        (109.90300661, 0.1249),
        (110.90418287, 0.1280),
        (111.90276287, 0.2413),
        (112.90440813, 0.1222),
        (113.90336509, 0.2873),
        (115.90476315, 0.0749),
    ],
    'In' : [
        (112.90406184, 0.0429),
        (114.903878776, 0.9571),
    ],
    'Sn' : [
        (111.90482387, 0.0097),
        (113.9027827, 0.0066),
        (114.903344699, 0.0034),
        (115.90174280, 0.1454),
        (116.90295398, 0.0768),
        (117.90160657, 0.2422),
        (118.90331117, 0.0859),
        (119.90220163, 0.3258),
        (121.9034438, 0.0463),
        (123.9052766, 0.0579),
    ],
    'Sb' : [
        (120.9038120, 0.5721),
        (122.9042132, 0.4279),
    ],
    'Te' : [
        (119.9040593, 0.0009),
        (121.9030435, 0.0255),
        (122.9042698, 0.0089),
        (123.9028171, 0.0474),
        (124.9044299, 0.0707),
        (125.9033109, 0.1884),
        (127.90446128, 0.3174),
        (129.906222748, 0.3408),
    ],
    'I' : [
        (126.9044719, 1.0000),
    ],
    'Xe' : [
        (123.9058920, 0.000952),
        (125.9042983, 0.000890),
        (127.9035310, 0.019102),
        (128.9047808611, 0.264006),
        (129.903509349, 0.040710),
        (130.90508406, 0.212324),
        (131.9041550856, 0.269086),
        (133.90539466, 0.104357),
        (135.907214484, 0.088573),
    ],
    'Cs' : [
        (132.9054519610, 1.0000),
    ],
    'Ba' : [
        (129.9063207, 0.00106),
        (131.9050611, 0.00101),
        (133.90450818, 0.02417),
        (134.90568838, 0.06592),
        (135.90457573, 0.07854),
        (136.90582714, 0.11232),
        (137.90524700, 0.71698),
    ],
    'La' : [
        (137.9071149, 0.0008881),
        (138.9063563, 0.9991119),
    ],
    'Ce' : [
        (135.90712921, 0.00185),
        (137.905991, 0.00251),
        (139.9054431, 0.88450),
        (141.9092504, 0.11114),
    ],
    'Pr' : [
        (140.9076576, 1.0000),
    ],
    'Nd' : [
        (141.9077290, 0.27152),
        (142.9098200, 0.12174),
        (143.9100930, 0.23798),
        (144.9125793, 0.08293),
        (145.9131226, 0.17189),
        (147.9168993, 0.05756),
        (149.9209022, 0.05638),
    ],
    'Sm' : [
        (143.9120065, 0.0307),
        (146.9149044, 0.1499),
        (147.9148292, 0.1124),
        (148.9171921, 0.1382),
        (149.9172829, 0.0738),
        (151.9197397, 0.2675),
        (153.9222169, 0.2275),
    ],
    'Eu' : [
        (150.9198578, 0.4781),
        (152.9212380, 0.5219),
    ],
    'Gd' : [
        (151.9197995, 0.0020),
        (153.9208741, 0.0218),
        (154.9226305, 0.1480),
        (155.9221312, 0.2047),
        (156.9239686, 0.1565),
        (157.9241123, 0.2484),
        (159.9270624, 0.2186),
    ],
    'Tb' : [
        (158.9253547, 1.0000),
    ],
    'Dy' : [
        (155.9242847, 0.00056),
        (157.9244159, 0.00095),
        (159.9252046, 0.02329),
        (160.9269405, 0.18889),
        (161.9268056, 0.25475),
        (162.9287383, 0.24896),
        (163.9291819, 0.28260),
    ],
    'Ho' : [
        (164.9303288, 1.0000),
    ],
    'Er' : [
        (161.9287884, 0.00139),
        (163.9292088, 0.01601),
        (165.9302995, 0.33503),
        (166.9320546, 0.22869),
        (167.9323767, 0.26978),
        (169.9354702, 0.14910),
    ],
    'Tm' : [
        (168.9342179, 1.0000),
    ],
    'Yb' : [
        (167.9338896, 0.00123),
        (169.9347664, 0.02982),
        (170.9363302, 0.1409),
        (171.9363859, 0.2168),
        (172.9382151, 0.16103),
        (173.9388664, 0.32026),
        (175.9425764, 0.12996),
    ],
    'Lu' : [
        (174.9407752, 0.97401),
        (175.9426897, 0.02599),
    ],
    'Hf' : [
        (173.9400461, 0.0016),
        (175.9414076, 0.0526),
        (176.9432277, 0.1860),
        (177.9437058, 0.2728),
        (178.9458232, 0.1362),
        (179.9465570, 0.3508),
    ],
    'Ta' : [
        (179.9474648, 0.0001201),
        (180.9479958, 0.9998799),
    ],
    'W' : [
        (179.9467108, 0.0012),
        (181.94820394, 0.2650),
        (182.95022275, 0.1431),
        (183.95093092, 0.3064),
        (185.9543628, 0.2843),
    ],
    'Re' : [
        (184.9529545, 0.3740),
        (186.9557501, 0.6260),
    ],
    'Os' : [
        (183.9524885, 0.0002),
        (185.9538350, 0.0159),
        (186.9557474, 0.0196),
        (187.9558352, 0.1324),
        (188.9581442, 0.1615),
        (189.9584437, 0.2626),
        (191.9614770, 0.4078),
    ],
    'Ir' : [
        (190.9605893, 0.373),
        (192.9629216, 0.627),
    ],
    'Pt' : [
        (189.9599297, 0.00012),
        (191.9610387, 0.00782),
        (193.9626809, 0.3286),
        (194.9647917, 0.3378),
        (195.96495209, 0.2521),
        (197.9678949, 0.07356),
    ],
    'Au' : [
        (196.96656879, 1.0000),
    ],
    'Hg' : [
        (195.9658326, 0.0015),
        (197.96676860, 0.0997),
        (198.96828064, 0.1687),
        (199.96832659, 0.2310),
        (200.97030284, 0.1318),
        (201.97064340, 0.2986),
        (203.97349398, 0.0687),
    ],
    'Tl' : [
        (202.9723446, 0.2952),
        (204.9744278, 0.7048),
    ],
    'Pb' : [
        (203.9730440, 0.014),
        (205.9744657, 0.241),
        (206.9758973, 0.221),
        (207.9766525, 0.524),
    ],
    'Bi' : [
        (208.9803991, 1.0000),
    ],
    'Th' : [
        (232.0380558, 1.0000),
    ],
    'Pa' : [
        (231.0358842, 1.0000),
    ],
    'U' : [
        (234.0409523, 0.000054),
        (235.0439301, 0.007204),
        (238.0507884, 0.992742),
    ],
}

atomic_weights = {
    'H' : 1.0079407540557772,
    'He' : 4.002601932120929,
    'Li' : 6.94003660291572,
    'Be' : 9.012183065,
    'B' : 10.811028046410001,
    'C' : 12.010735896735248,
    'N' : 14.006703211445798,
    'O' : 15.999404924318277,
    'F' : 18.99840316273,
    'Ne' : 20.18004638052026,
    'Na' : 22.989769282,
    'Mg' : 24.3050516198371,
    'Al' : 26.98153853,
    'Si' : 28.08549870570596,
    'P' : 30.97376199842,
    'S' : 32.06478740612707,
    'Cl' : 35.452937582608,
    'Ar' : 39.947798563582005,
    'K' : 39.098300910086,
    'Ca' : 40.078022511017735,
    'Sc' : 44.95590828,
    'Ti' : 47.866744962722,
    'V' : 50.941465037425004,
    'Cr' : 51.9961317554337,
    'Mn' : 54.93804391,
    'Fe' : 55.845144433865904,
    'Co' : 58.93319429,
    'Ni' : 58.69334710994765,
    'Cu' : 63.54603994583,
    'Zn' : 65.37778252952499,
    'Ga' : 69.7230660725936,
    'Ge' : 72.6275501646868,
    'As' : 74.92159457,
    'Se' : 78.95938855701361,
    'Br' : 79.90352778050999,
    'Kr' : 83.7979999953261,
    'Rb' : 85.46766359561973,
    'Sr' : 87.61664446962,
    'Y' : 88.9058403,
    'Zr' : 91.22364159706,
    'Nb' : 92.906373,
    'Mo' : 95.95978854118802,
    'Ru' : 101.06494013916,
    'Rh' : 102.905498,
    'Pd' : 106.41532750734,
    'Ag' : 107.868149634557,
    'Cd' : 112.411557818268,
    'In' : 114.81808662944559,
    'Sn' : 118.7101125930106,
    'Sb' : 121.75978367348,
    'Te' : 127.60312648466041,
    'I' : 126.9044719,
    'Xe' : 131.29276144779053,
    'Cs' : 132.905451961,
    'Ba' : 137.3268916286322,
    'La' : 138.90546887371266,
    'Ce' : 140.1157307378545,
    'Pr' : 140.9076576,
    'Nd' : 144.24159603182702,
    'Sm' : 150.36635571193,
    'Eu' : 151.96437812637998,
    'Gd' : 157.25213064687998,
    'Tb' : 158.9253547,
    'Dy' : 162.499472819424,
    'Ho' : 164.9303288,
    'Er' : 167.259082649669,
    'Tm' : 168.9342179,
    'Yb' : 173.05415016631704,
    'Lu' : 174.96681495785498,
    'Hf' : 178.4849787234,
    'Ta' : 180.9478756362269,
    'W' : 183.841777550513,
    'Re' : 186.20670454560002,
    'Os' : 190.22485962823998,
    'Ir' : 192.2160516521,
    'Pt' : 195.08445686493104,
    'Au' : 196.96656879,
    'Hg' : 200.59916703455602,
    'Tl' : 204.38341283936,
    'Pb' : 207.216908063,
    'Bi' : 208.9803991,
    'Th' : 232.0380558,
    'Pa' : 231.0358842,
    'U' : 238.0289104616574,
}