"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const semver = require("semver");
const my_package_json_1 = require("./private/my-package-json");
const schema_helpers_1 = require("./private/schema-helpers");
const PACKAGE_VERSION = my_package_json_1.loadMyPackageJson().version;
/**
 * Static class with loader routines
 *
 * This class mostly exists to put the schema structs into input position
 * (taken into a function), so that the jsii-diff checker will make sure all
 * structs are only allowed to be weakened in future updates. For example,
 * it is now allowed to add new required fields, since old CDK frameworks
 * would not be emitting those fields yet.
 *
 * At the same time, we might as well validate the structure so code doesn't
 * barf on invalid disk input.
 */
class AssetManifestSchema {
    /**
     * Validate the given structured object as a valid ManifestFile schema
     */
    static validate(file) {
        const obj = file;
        if (typeof obj !== 'object' || obj === null) {
            throw new Error(`Expected object, got '${obj}`);
        }
        schema_helpers_1.expectKey(obj, 'version', schema_helpers_1.isString);
        // Current tool must be >= the version used to write the manifest
        // (disregarding MVs) which we can verify by ^-prefixing the file version.
        if (!semver.satisfies(AssetManifestSchema.currentVersion(), `^${obj.version}`)) {
            throw new Error(`Need CDK Tools >= '${obj.version}' to read this file (current is '${AssetManifestSchema.currentVersion()}')`);
        }
        schema_helpers_1.expectKey(obj, 'files', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isFileAsset)), true);
        schema_helpers_1.expectKey(obj, 'dockerImages', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isDockerImageAsset)), true);
    }
    /**
     * Take a ManifestFile as input
     *
     * The presence of this method makes sure the struct is only ever weakened
     * in future releases.
     */
    static input(file) {
        this.validate(file);
    }
    /**
     * Return the version of the schema module
     */
    static currentVersion() {
        return PACKAGE_VERSION;
    }
}
exports.AssetManifestSchema = AssetManifestSchema;
function isFileAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'path', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'packaging', schema_helpers_1.isString, true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'bucketName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'objectKey', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
function isDockerImageAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'directory', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'dockerFile', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildTarget', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildArgs', schema_helpers_1.isMapOf(schema_helpers_1.isString), true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'repositoryName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'imageTag', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
//# sourceMappingURL=data:application/json;base64,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