"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventBus = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const archive_1 = require("./archive");
const events_generated_1 = require("./events.generated");
class EventBusBase extends core_1.Resource {
    /**
     * Create an EventBridge archive to send events to.
     *
     * When you create an archive, incoming events might not immediately start being sent to the archive.
     * Allow a short period of time for changes to take effect.
     *
     * @stability stable
     */
    archive(id, props) {
        return new archive_1.Archive(this, id, {
            sourceEventBus: this,
            description: props.description || `Event Archive for ${this.eventBusName} Event Bus`,
            eventPattern: props.eventPattern,
            retention: props.retention,
            archiveName: props.archiveName,
        });
    }
    /**
     * Grants an IAM Principal to send custom events to the eventBus so that they can be matched to rules.
     *
     * @stability stable
     */
    grantPutEventsTo(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: [this.eventBusArn],
        });
    }
}
/**
 * Define an EventBridge EventBus.
 *
 * @stability stable
 * @resource AWS::Events::EventBus
 */
class EventBus extends EventBusBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_events_EventBusProps(props);
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.string({ produce: () => core_1.Names.uniqueId(this) }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: this.physicalName,
            eventSourceName,
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name,
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusArn ARN of imported event bus.
     * @stability stable
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).splitArn(eventBusArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        return new ImportedEventBus(scope, id, {
            eventBusArn: eventBusArn,
            eventBusName: parts.resourceName || '',
            eventBusPolicy: '',
        });
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusName Name of imported event bus.
     * @stability stable
     */
    static fromEventBusName(scope, id, eventBusName) {
        const eventBusArn = core_1.Stack.of(scope).formatArn({
            resource: 'event-bus',
            service: 'events',
            resourceName: eventBusName,
        });
        return EventBus.fromEventBusAttributes(scope, id, {
            eventBusName: eventBusName,
            eventBusArn: eventBusArn,
            eventBusPolicy: '',
        });
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param attrs Imported event bus properties.
     * @stability stable
     */
    static fromEventBusAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_events_EventBusAttributes(attrs);
        return new ImportedEventBus(scope, id, attrs);
    }
    /**
     * (deprecated) Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     * @deprecated use grantAllPutEvents instead
     */
    static grantPutEvents(grantee) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-events.EventBus#grantPutEvents", "use grantAllPutEvents instead");
        // It's currently not possible to restrict PutEvents to specific resources.
        // See https://docs.aws.amazon.com/eventbridge/latest/userguide/permissions-reference-eventbridge.html
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    /**
     * Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    static grantAllPutEvents(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    static eventBusProps(defaultEventBusName, props = {}) {
        const { eventBusName, eventSourceName } = props;
        const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
        if (eventBusName !== undefined && eventSourceName !== undefined) {
            throw new Error('\'eventBusName\' and \'eventSourceName\' cannot both be provided');
        }
        if (eventBusName !== undefined) {
            if (!core_1.Token.isUnresolved(eventBusName)) {
                if (eventBusName === 'default') {
                    throw new Error('\'eventBusName\' must not be \'default\'');
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error('\'eventBusName\' must not contain \'/\'');
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
            }
            return { eventBusName };
        }
        if (eventSourceName !== undefined) {
            // Ex: aws.partner/PartnerName/acct1/repo1
            const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
            if (!eventSourceNameRegex.test(eventSourceName)) {
                throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
            }
            else if (!eventBusNameRegex.test(eventSourceName)) {
                throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
            }
            return { eventBusName: eventSourceName, eventSourceName };
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
_a = JSII_RTTI_SYMBOL_1;
EventBus[_a] = { fqn: "@aws-cdk/aws-events.EventBus", version: "1.138.0" };
class ImportedEventBus extends EventBusBase {
    constructor(scope, id, attrs) {
        const arnParts = core_1.Stack.of(scope).splitArn(attrs.eventBusArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        super(scope, id, {
            account: arnParts.account,
            region: arnParts.region,
        });
        this.eventBusArn = attrs.eventBusArn;
        this.eventBusName = attrs.eventBusName;
        this.eventBusPolicy = attrs.eventBusPolicy;
        this.eventSourceName = attrs.eventSourceName;
    }
}
//# sourceMappingURL=data:application/json;base64,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