"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = exports.Platform = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
const ecs = require("aws-cdk-lib/aws-ecs");
const configuration_1 = require("./configuration");
const environment_1 = require("./environment");
const extension_1 = require("./extension");
/**
 * Defines the platform for the AWS AppConfig Lambda extension.
 */
var Platform;
(function (Platform) {
    Platform["X86_64"] = "x86-64";
    Platform["ARM_64"] = "ARM64";
})(Platform || (exports.Platform = Platform = {}));
class ApplicationBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this._environments = [];
    }
    addEnvironment(id, options = {}) {
        return new environment_1.Environment(this, id, {
            application: this,
            ...options,
        });
    }
    addHostedConfiguration(id, options) {
        return new configuration_1.HostedConfiguration(this, id, {
            application: this,
            ...options,
        });
    }
    addSourcedConfiguration(id, options) {
        return new configuration_1.SourcedConfiguration(this, id, {
            application: this,
            ...options,
        });
    }
    addExistingEnvironment(environment) {
        this._environments.push(environment);
    }
    get environments() {
        return this._environments;
    }
    /**
     * Adds an extension defined by the action point and event destination
     * and also creates an extension association to an application.
     *
     * @param actionPoint The action point which triggers the event
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    on(actionPoint, eventDestination, options) {
        this.extensible.on(actionPoint, eventDestination, options);
    }
    /**
     * Adds a PRE_CREATE_HOSTED_CONFIGURATION_VERSION extension with the
     * provided event destination and also creates an extension association to an application.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preCreateHostedConfigurationVersion(eventDestination, options) {
        this.extensible.preCreateHostedConfigurationVersion(eventDestination, options);
    }
    /**
     * Adds a PRE_START_DEPLOYMENT extension with the provided event destination and
     * also creates an extension association to an application.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preStartDeployment(eventDestination, options) {
        this.extensible.preStartDeployment(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_START extension with the provided event destination and
     * also creates an extension association to an application.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStart(eventDestination, options) {
        this.extensible.onDeploymentStart(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_STEP extension with the provided event destination and
     * also creates an extension association to an application.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStep(eventDestination, options) {
        this.extensible.onDeploymentStep(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_BAKING extension with the provided event destination and
     * also creates an extension association to an application.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentBaking(eventDestination, options) {
        this.extensible.onDeploymentBaking(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_COMPLETE extension with the provided event destination and
     * also creates an extension association to an application.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentComplete(eventDestination, options) {
        this.extensible.onDeploymentComplete(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_ROLLED_BACK extension with the provided event destination and
     * also creates an extension association to an application.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentRolledBack(eventDestination, options) {
        this.extensible.onDeploymentRolledBack(eventDestination, options);
    }
    /**
     * Adds an extension association to the application.
     *
     * @param extension The extension to create an association for
     */
    addExtension(extension) {
        this.extensible.addExtension(extension);
    }
}
/**
 * An AWS AppConfig application.
 *
 * @resource AWS::AppConfig::Application
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-application.html
 */
class Application extends ApplicationBase {
    /**
     * Imports an AWS AppConfig application into the CDK using its Amazon Resource Name (ARN).
     *
     * @param scope The parent construct
     * @param id The name of the application construct
     * @param applicationArn The Amazon Resource Name (ARN) of the application
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const parsedArn = cdk.Stack.of(scope).splitArn(applicationArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const applicationId = parsedArn.resourceName;
        if (!applicationId) {
            throw new Error('Missing required application id from application ARN');
        }
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationId = applicationId;
                this.applicationArn = applicationArn;
                this.extensible = new extension_1.ExtensibleBase(scope, this.applicationArn);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an AWS AppConfig application into the CDK using its ID.
     *
     * @param scope The parent construct
     * @param id The name of the application construct
     * @param applicationId The ID of the application
     */
    static fromApplicationId(scope, id, applicationId) {
        const stack = cdk.Stack.of(scope);
        const applicationArn = stack.formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: applicationId,
        });
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationId = applicationId;
                this.applicationArn = applicationArn;
                this.extensible = new extension_1.ExtensibleBase(scope, this.applicationArn);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Retrieves the Lambda layer version Amazon Resource Name (ARN) for the AWS AppConfig Lambda extension.
     *
     * @param region The region for the Lambda layer (for example, 'us-east-1')
     * @param platform The platform for the Lambda layer (default is Platform.X86_64)
     * @returns Lambda layer version ARN
     */
    static getLambdaLayerVersionArn(region, platform) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_Platform(platform);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getLambdaLayerVersionArn);
            }
            throw error;
        }
        return lambdaLayerVersions[platform || Platform.X86_64][region];
    }
    /**
     * Adds the AWS AppConfig Agent as a container to the provided ECS task definition.
     *
     * @param taskDef The ECS task definition
     */
    static addAgentToEcs(taskDef) {
        taskDef.addContainer('AppConfigAgentContainer', {
            image: ecs.ContainerImage.fromRegistry('public.ecr.aws/aws-appconfig/aws-appconfig-agent:latest'),
            containerName: 'AppConfigAgentContainer',
        });
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        this.description = props.description;
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 64,
            separator: '-',
        });
        this._application = new aws_appconfig_1.CfnApplication(this, 'Resource', {
            name: this.name,
            description: this.description,
        });
        this.applicationId = this._application.ref;
        this.applicationArn = cdk.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: this.applicationId,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.applicationArn, this.name);
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.Application", version: "2.106.0-alpha.0" };
const lambdaLayerVersions = {
    [Platform.X86_64]: {
        'us-east-1': 'arn:aws:lambda:us-east-1:027255383542:layer:AWS-AppConfig-Extension:110',
        'us-east-2': 'arn:aws:lambda:us-east-2:728743619870:layer:AWS-AppConfig-Extension:79',
        'us-west-1': 'arn:aws:lambda:us-west-1:958113053741:layer:AWS-AppConfig-Extension:121',
        'us-west-2': 'arn:aws:lambda:us-west-2:359756378197:layer:AWS-AppConfig-Extension:143',
        'ca-central-1': 'arn:aws:lambda:ca-central-1:039592058896:layer:AWS-AppConfig-Extension:79',
        'eu-central-1': 'arn:aws:lambda:eu-central-1:066940009817:layer:AWS-AppConfig-Extension:91',
        'eu-central-2': 'arn:aws:lambda:eu-central-2:758369105281:layer:AWS-AppConfig-Extension:29',
        'eu-west-1': 'arn:aws:lambda:eu-west-1:434848589818:layer:AWS-AppConfig-Extension:108',
        'eu-west-2': 'arn:aws:lambda:eu-west-2:282860088358:layer:AWS-AppConfig-Extension:79',
        'eu-west-3': 'arn:aws:lambda:eu-west-3:493207061005:layer:AWS-AppConfig-Extension:80',
        'eu-north-1': 'arn:aws:lambda:eu-north-1:646970417810:layer:AWS-AppConfig-Extension:139',
        'eu-south-1': 'arn:aws:lambda:eu-south-1:203683718741:layer:AWS-AppConfig-Extension:71',
        'eu-south-2': 'arn:aws:lambda:eu-south-2:586093569114:layer:AWS-AppConfig-Extension:26',
        'cn-north-1': 'arn:aws-cn:lambda:cn-north-1:615057806174:layer:AWS-AppConfig-Extension:66',
        'cn-northwest-1': 'arn:aws-cn:lambda:cn-northwest-1:615084187847:layer:AWS-AppConfig-Extension:66',
        'ap-east-1': 'arn:aws:lambda:ap-east-1:630222743974:layer:AWS-AppConfig-Extension:71',
        'ap-northeast-1': 'arn:aws:lambda:ap-northeast-1:980059726660:layer:AWS-AppConfig-Extension:82',
        'ap-northeast-2': 'arn:aws:lambda:ap-northeast-2:826293736237:layer:AWS-AppConfig-Extension:91',
        'ap-northeast-3': 'arn:aws:lambda:ap-northeast-3:706869817123:layer:AWS-AppConfig-Extension:84',
        'ap-southeast-1': 'arn:aws:lambda:ap-southeast-1:421114256042:layer:AWS-AppConfig-Extension:89',
        'ap-southeast-2': 'arn:aws:lambda:ap-southeast-2:080788657173:layer:AWS-AppConfig-Extension:91',
        'ap-southeast-3': 'arn:aws:lambda:ap-southeast-3:418787028745:layer:AWS-AppConfig-Extension:60',
        'ap-southeast-4': 'arn:aws:lambda:ap-southeast-4:307021474294:layer:AWS-AppConfig-Extension:2',
        'ap-south-1': 'arn:aws:lambda:ap-south-1:554480029851:layer:AWS-AppConfig-Extension:92',
        'ap-south-2': 'arn:aws:lambda:ap-south-2:489524808438:layer:AWS-AppConfig-Extension:29',
        'sa-east-1': 'arn:aws:lambda:sa-east-1:000010852771:layer:AWS-AppConfig-Extension:110',
        'af-south-1': 'arn:aws:lambda:af-south-1:574348263942:layer:AWS-AppConfig-Extension:71',
        'me-central-1': 'arn:aws:lambda:me-central-1:662846165436:layer:AWS-AppConfig-Extension:31',
        'me-south-1': 'arn:aws:lambda:me-south-1:559955524753:layer:AWS-AppConfig-Extension:71',
        'us-gov-east-1': 'arn:aws-us-gov:lambda:us-gov-east-1:946561847325:layer:AWS-AppConfig-Extension:44',
        'us-gov-west-1': 'arn:aws-us-gov:lambda:us-gov-west-1:946746059096:layer:AWS-AppConfig-Extension:44',
    },
    [Platform.ARM_64]: {
        'us-east-1': 'arn:aws:lambda:us-east-1:027255383542:layer:AWS-AppConfig-Extension-Arm64:43',
        'us-east-2': 'arn:aws:lambda:us-east-2:728743619870:layer:AWS-AppConfig-Extension-Arm64:31',
        'us-west-2': 'arn:aws:lambda:us-west-2:359756378197:layer:AWS-AppConfig-Extension-Arm64:45',
        'eu-central-1': 'arn:aws:lambda:eu-central-1:066940009817:layer:AWS-AppConfig-Extension-Arm64:34',
        'eu-west-1': 'arn:aws:lambda:eu-west-1:434848589818:layer:AWS-AppConfig-Extension-Arm64:46',
        'eu-west-2': 'arn:aws:lambda:eu-west-2:282860088358:layer:AWS-AppConfig-Extension-Arm64:31',
        'ap-northeast-1': 'arn:aws:lambda:ap-northeast-1:980059726660:layer:AWS-AppConfig-Extension-Arm64:35',
        'ap-southeast-1': 'arn:aws:lambda:ap-southeast-1:421114256042:layer:AWS-AppConfig-Extension-Arm64:41',
        'ap-southeast-2': 'arn:aws:lambda:ap-southeast-2:080788657173:layer:AWS-AppConfig-Extension-Arm64:34',
        'ap-south-1': 'arn:aws:lambda:ap-south-1:554480029851:layer:AWS-AppConfig-Extension-Arm64:34',
    },
};
//# sourceMappingURL=data:application/json;base64,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