"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationSource = exports.ConfigurationContent = exports.LambdaValidator = exports.JsonSchemaValidator = exports.ConfigurationSourceType = exports.ValidatorType = exports.ConfigurationType = exports.SourcedConfiguration = exports.HostedConfiguration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const mimeTypes = require("mime-types");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const deployment_strategy_1 = require("./deployment-strategy");
const extension_1 = require("./extension");
const hash_1 = require("./private/hash");
class ConfigurationBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 128,
            separator: '-',
        });
        this.application = props.application;
        this.applicationId = this.application.applicationId;
        this.type = props.type;
        this.validators = props.validators;
        this.description = props.description;
        this.deployTo = props.deployTo;
        this.deploymentKey = props.deploymentKey;
        this.deploymentStrategy = props.deploymentStrategy || new deployment_strategy_1.DeploymentStrategy(this, 'DeploymentStrategy', {
            rolloutStrategy: deployment_strategy_1.RolloutStrategy.CANARY_10_PERCENT_20_MINUTES,
        });
    }
    /**
     * Adds an extension defined by the action point and event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param actionPoint The action point which triggers the event
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    on(actionPoint, eventDestination, options) {
        this.extensible.on(actionPoint, eventDestination, options);
    }
    /**
     * Adds a PRE_CREATE_HOSTED_CONFIGURATION_VERSION extension with the
     * provided event destination and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preCreateHostedConfigurationVersion(eventDestination, options) {
        this.extensible.preCreateHostedConfigurationVersion(eventDestination, options);
    }
    /**
     * Adds a PRE_START_DEPLOYMENT extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preStartDeployment(eventDestination, options) {
        this.extensible.preStartDeployment(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_START extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStart(eventDestination, options) {
        this.extensible.onDeploymentStart(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_STEP extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStep(eventDestination, options) {
        this.extensible.onDeploymentStep(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_BAKING extension with the provided event destination and
     * also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentBaking(eventDestination, options) {
        this.extensible.onDeploymentBaking(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_COMPLETE extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentComplete(eventDestination, options) {
        this.extensible.onDeploymentComplete(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_ROLLED_BACK extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentRolledBack(eventDestination, options) {
        this.extensible.onDeploymentRolledBack(eventDestination, options);
    }
    /**
     * Adds an extension association to the configuration profile.
     *
     * @param extension The extension to create an association for
     */
    addExtension(extension) {
        this.extensible.addExtension(extension);
    }
    addExistingEnvironmentsToApplication() {
        this.deployTo?.forEach((environment) => {
            if (!this.application.environments.includes(environment)) {
                this.application.addExistingEnvironment(environment);
            }
        });
    }
    deployConfigToEnvironments() {
        if (!this.deployTo || !this.versionNumber) {
            return;
        }
        this.application.environments.forEach((environment) => {
            if ((this.deployTo && !this.deployTo.includes(environment))) {
                return;
            }
            const logicalId = `Deployment${this.getDeploymentHash(environment)}`;
            new aws_appconfig_1.CfnDeployment(this, logicalId, {
                applicationId: this.application.applicationId,
                configurationProfileId: this.configurationProfileId,
                deploymentStrategyId: this.deploymentStrategy.deploymentStrategyId,
                environmentId: environment.environmentId,
                configurationVersion: this.versionNumber,
                description: this.description,
                kmsKeyIdentifier: this.deploymentKey?.keyArn,
            });
        });
    }
}
/**
 * A hosted configuration represents configuration stored in the AWS AppConfig hosted configuration store.
 */
class HostedConfiguration extends ConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_HostedConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, HostedConfiguration);
            }
            throw error;
        }
        this._cfnConfigurationProfile = new aws_appconfig_1.CfnConfigurationProfile(this, 'ConfigurationProfile', {
            applicationId: this.applicationId,
            locationUri: 'hosted',
            name: this.name,
            description: this.description,
            type: this.type,
            validators: this.validators,
        });
        this.configurationProfileId = this._cfnConfigurationProfile.ref;
        this.configurationProfileArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.configurationProfileArn, this.name);
        this.content = props.content.content;
        this.contentType = props.content.contentType;
        this.latestVersionNumber = props.latestVersionNumber;
        this.versionLabel = props.versionLabel;
        this._cfnHostedConfigurationVersion = new aws_appconfig_1.CfnHostedConfigurationVersion(this, 'Resource', {
            applicationId: this.applicationId,
            configurationProfileId: this.configurationProfileId,
            content: this.content,
            contentType: this.contentType,
            description: this.description,
            latestVersionNumber: this.latestVersionNumber,
            versionLabel: this.versionLabel,
        });
        this._cfnHostedConfigurationVersion.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.RETAIN);
        this.versionNumber = this._cfnHostedConfigurationVersion.ref;
        this.hostedConfigurationVersionArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}/hostedconfigurationversion/${this.versionNumber}`,
        });
        this.addExistingEnvironmentsToApplication();
        this.deployConfigToEnvironments();
    }
    getDeploymentHash(environment) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_IEnvironment(environment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDeploymentHash);
            }
            throw error;
        }
        const combinedString = `
      ${this.application.name}
      ${this.name}
      ${environment.name}
      ${this.content}
    `;
        return (0, hash_1.getHash)(combinedString);
    }
}
exports.HostedConfiguration = HostedConfiguration;
_a = JSII_RTTI_SYMBOL_1;
HostedConfiguration[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.HostedConfiguration", version: "2.106.0-alpha.0" };
/**
 * A sourced configuration represents configuration stored in an Amazon S3 bucket, AWS Secrets Manager secret, Systems Manager
 * (SSM) Parameter Store parameter, SSM document, or AWS CodePipeline.
 */
class SourcedConfiguration extends ConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_SourcedConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SourcedConfiguration);
            }
            throw error;
        }
        this.location = props.location;
        this.locationUri = this.location.locationUri;
        this.versionNumber = props.versionNumber;
        this.sourceKey = this.location.key;
        this.retrievalRole = props.retrievalRole || this.location.type != ConfigurationSourceType.CODE_PIPELINE
            ? new iam.Role(this, 'Role', {
                roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('appconfig.amazonaws.com'),
                inlinePolicies: {
                    ['AllowAppConfigReadFromSourcePolicy']: this.getPolicyForRole(),
                },
            })
            : undefined;
        this._cfnConfigurationProfile = new aws_appconfig_1.CfnConfigurationProfile(this, 'Resource', {
            applicationId: this.applicationId,
            locationUri: this.locationUri,
            name: this.name,
            description: this.description,
            retrievalRoleArn: this.retrievalRole?.roleArn,
            type: this.type,
            validators: this.validators,
        });
        this.configurationProfileId = this._cfnConfigurationProfile.ref;
        this.configurationProfileArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.configurationProfileArn, this.name);
        this.addExistingEnvironmentsToApplication();
        this.deployConfigToEnvironments();
    }
    getDeploymentHash(environment) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_IEnvironment(environment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDeploymentHash);
            }
            throw error;
        }
        const combinedString = `
      ${this.application.name}
      ${this.name}
      ${environment.name}
      ${this.versionNumber}
      ${this.location.type}
    `;
        return (0, hash_1.getHash)(combinedString);
    }
    getPolicyForRole() {
        const policy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
        });
        const document = new iam.PolicyDocument({
            statements: [policy],
        });
        if (this.location.type == ConfigurationSourceType.SSM_PARAMETER) {
            policy.addActions('ssm:GetParameter');
            policy.addResources(this.locationUri);
        }
        else if (this.location.type == ConfigurationSourceType.SSM_DOCUMENT) {
            policy.addActions('ssm:GetDocument');
            policy.addResources(aws_cdk_lib_1.Stack.of(this).formatArn({
                service: 'ssm',
                resource: 'document',
                resourceName: this.locationUri.split('://')[1],
            }));
        }
        else if (this.location.type == ConfigurationSourceType.S3) {
            const bucketAndObjectKey = this.locationUri.split('://')[1];
            const sep = bucketAndObjectKey.search('/');
            const bucketName = bucketAndObjectKey.substring(0, sep);
            const objectKey = bucketAndObjectKey.substring(sep + 1);
            policy.addActions('s3:GetObject', 's3:GetObjectMetadata', 's3:GetObjectVersion');
            policy.addResources(aws_cdk_lib_1.Stack.of(this).formatArn({
                region: '',
                account: '',
                service: 's3',
                arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                resource: `${bucketName}/${objectKey}`,
            }));
            const bucketPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: [
                    's3:GetBucketLocation',
                    's3:GetBucketVersioning',
                    's3:ListBucket',
                    's3:ListBucketVersions',
                ],
                resources: [
                    aws_cdk_lib_1.Stack.of(this).formatArn({
                        region: '',
                        account: '',
                        service: 's3',
                        arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                        resource: bucketName,
                    }),
                ],
            });
            const listBucketsPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:ListAllMyBuckets'],
                resources: ['*'],
            });
            document.addStatements(bucketPolicy, listBucketsPolicy);
        }
        else {
            policy.addActions('secretsmanager:GetSecretValue');
            policy.addResources(this.locationUri);
        }
        if (this.sourceKey) {
            const keyPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['kms:Decrypt'],
                resources: [this.sourceKey.keyArn],
            });
            document.addStatements(keyPolicy);
        }
        return document;
    }
}
exports.SourcedConfiguration = SourcedConfiguration;
_b = JSII_RTTI_SYMBOL_1;
SourcedConfiguration[_b] = { fqn: "@aws-cdk/aws-appconfig-alpha.SourcedConfiguration", version: "2.106.0-alpha.0" };
/**
 * The configuration type.
 */
var ConfigurationType;
(function (ConfigurationType) {
    /**
     * Freeform configuration profile. Allows you to store your data in the AWS AppConfig
     * hosted configuration store or another Systems Manager capability or AWS service that integrates
     * with AWS AppConfig.
     *
     * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-free-form-configurations-creating.html
     */
    ConfigurationType["FREEFORM"] = "AWS.Freeform";
    /**
     * Feature flag configuration profile. This configuration stores its data
     * in the AWS AppConfig hosted configuration store and the URI is simply hosted.
     */
    ConfigurationType["FEATURE_FLAGS"] = "AWS.AppConfig.FeatureFlags";
})(ConfigurationType || (exports.ConfigurationType = ConfigurationType = {}));
/**
 * The validator type.
 */
var ValidatorType;
(function (ValidatorType) {
    /**
     * JSON Scema validator.
     */
    ValidatorType["JSON_SCHEMA"] = "JSON_SCHEMA";
    /**
     * Validate using a Lambda function.
     */
    ValidatorType["LAMBDA"] = "LAMBDA";
})(ValidatorType || (exports.ValidatorType = ValidatorType = {}));
/**
 * The configuration source type.
 */
var ConfigurationSourceType;
(function (ConfigurationSourceType) {
    ConfigurationSourceType["S3"] = "S3";
    ConfigurationSourceType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ConfigurationSourceType["SSM_PARAMETER"] = "SSM_PARAMETER";
    ConfigurationSourceType["SSM_DOCUMENT"] = "SSM_DOCUMENT";
    ConfigurationSourceType["CODE_PIPELINE"] = "CODE_PIPELINE";
})(ConfigurationSourceType || (exports.ConfigurationSourceType = ConfigurationSourceType = {}));
/**
 * Defines a JSON Schema validator.
 */
class JsonSchemaValidator {
    /**
     * Defines a JSON Schema validator from a file.
     *
     * @param path The path to the file that defines the validator
     */
    static fromFile(path) {
        return {
            content: fs.readFileSync(path).toString(),
            type: ValidatorType.JSON_SCHEMA,
        };
    }
    /**
     * Defines a JSON Schema validator from inline code.
     *
     * @param code The inline code that defines the validator
     */
    static fromInline(code) {
        return {
            content: code,
            type: ValidatorType.JSON_SCHEMA,
        };
    }
}
exports.JsonSchemaValidator = JsonSchemaValidator;
_c = JSII_RTTI_SYMBOL_1;
JsonSchemaValidator[_c] = { fqn: "@aws-cdk/aws-appconfig-alpha.JsonSchemaValidator", version: "2.106.0-alpha.0" };
/**
 * Defines an AWS Lambda validator.
 */
class LambdaValidator {
    /**
     *  Defines an AWS Lambda validator from a Lambda function. This will call
     * `addPermission` to your function to grant AWS AppConfig permissions.
     *
     * @param func The function that defines the validator
     */
    static fromFunction(func) {
        if (!func.permissionsNode.tryFindChild('AppConfigPermission')) {
            func.addPermission('AppConfigPermission', {
                principal: new iam.ServicePrincipal('appconfig.amazonaws.com'),
            });
        }
        return {
            content: func.functionArn,
            type: ValidatorType.LAMBDA,
        };
    }
}
exports.LambdaValidator = LambdaValidator;
_d = JSII_RTTI_SYMBOL_1;
LambdaValidator[_d] = { fqn: "@aws-cdk/aws-appconfig-alpha.LambdaValidator", version: "2.106.0-alpha.0" };
/**
 * Defines the hosted configuration content.
 */
class ConfigurationContent {
    /**
     * Defines the hosted configuration content from a file.
     *
     * @param path The path to the file that defines configuration content
     * @param contentType The content type of the configuration
     */
    static fromFile(path, contentType) {
        return {
            content: fs.readFileSync(path).toString(),
            contentType: contentType || mimeTypes.lookup(path) || 'application/json',
        };
    }
    /**
     * Defines the hosted configuration content from inline code.
     *
     * @param content The inline code that defines the configuration content
     * @param contentType The content type of the configuration
     */
    static fromInline(content, contentType) {
        return {
            content,
            contentType: contentType || 'application/octet-stream',
        };
    }
    /**
     * Defines the hosted configuration content as JSON from inline code.
     *
     * @param content The inline code that defines the configuration content
     * @param contentType The content type of the configuration
     */
    static fromInlineJson(content, contentType) {
        return {
            content,
            contentType: contentType || 'application/json',
        };
    }
    /**
     * Defines the hosted configuration content as text from inline code.
     *
     * @param content The inline code that defines the configuration content
     */
    static fromInlineText(content) {
        return {
            content,
            contentType: 'text/plain',
        };
    }
    /**
     * Defines the hosted configuration content as YAML from inline code.
     *
     * @param content The inline code that defines the configuration content
     */
    static fromInlineYaml(content) {
        return {
            content,
            contentType: 'application/x-yaml',
        };
    }
}
exports.ConfigurationContent = ConfigurationContent;
_e = JSII_RTTI_SYMBOL_1;
ConfigurationContent[_e] = { fqn: "@aws-cdk/aws-appconfig-alpha.ConfigurationContent", version: "2.106.0-alpha.0" };
/**
 * Defines the integrated configuration sources.
 */
class ConfigurationSource {
    /**
     * Defines configuration content from an Amazon S3 bucket.
     *
     * @param bucket The S3 bucket where the configuration is stored
     * @param objectKey The path to the configuration
     * @param key The KMS Key that the bucket is encrypted with
     */
    static fromBucket(bucket, objectKey, key) {
        return {
            locationUri: bucket.s3UrlForObject(objectKey),
            type: ConfigurationSourceType.S3,
            key,
        };
    }
    /**
     * Defines configuration content from an AWS Secrets Manager secret.
     *
     * @param secret The secret where the configuration is stored
     */
    static fromSecret(secret) {
        return {
            locationUri: secret.secretArn,
            type: ConfigurationSourceType.SECRETS_MANAGER,
            key: secret.encryptionKey,
        };
    }
    /**
     * Defines configuration content from a Systems Manager (SSM) Parameter Store parameter.
     *
     * @param parameter The parameter where the configuration is stored
     * @param key The KMS Key that the secure string is encrypted with
     */
    static fromParameter(parameter, key) {
        return {
            locationUri: parameter.parameterArn,
            type: ConfigurationSourceType.SSM_PARAMETER,
            key,
        };
    }
    /**
     * Defines configuration content from a Systems Manager (SSM) document.
     *
     * @param document The SSM document where the configuration is stored
     */
    static fromCfnDocument(document) {
        return {
            locationUri: `ssm-document://${document.ref}`,
            type: ConfigurationSourceType.SSM_DOCUMENT,
        };
    }
    /**
     * Defines configuration content from AWS CodePipeline.
     *
     * @param pipeline The pipeline where the configuration is stored
     * @returns
     */
    static fromPipeline(pipeline) {
        return {
            locationUri: `codepipeline://${pipeline.pipelineName}`,
            type: ConfigurationSourceType.CODE_PIPELINE,
        };
    }
}
exports.ConfigurationSource = ConfigurationSource;
_f = JSII_RTTI_SYMBOL_1;
ConfigurationSource[_f] = { fqn: "@aws-cdk/aws-appconfig-alpha.ConfigurationSource", version: "2.106.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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