"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RolloutStrategy = exports.PredefinedDeploymentStrategyId = exports.GrowthType = exports.DeploymentStrategy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
/**
 * An AWS AppConfig deployment strategy.
 *
 * @resource AWS::AppConfig::DeploymentStrategy
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-deployment-strategy.html
 */
class DeploymentStrategy extends aws_cdk_lib_1.Resource {
    /**
     * Imports a deployment strategy into the CDK using its Amazon Resource Name (ARN).
     *
     * @param scope The parent construct
     * @param id The name of the deployment strategy construct
     * @param deploymentStrategyArn The Amazon Resource Name (ARN) of the deployment strategy
     */
    static fromDeploymentStrategyArn(scope, id, deploymentStrategyArn) {
        const parsedArn = aws_cdk_lib_1.Stack.of(scope).splitArn(deploymentStrategyArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        const deploymentStrategyId = parsedArn.resourceName;
        if (!deploymentStrategyId) {
            throw new Error('Missing required deployment strategy id from deployment strategy ARN');
        }
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.deploymentStrategyId = deploymentStrategyId;
                this.deploymentStrategyArn = deploymentStrategyArn;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: deploymentStrategyArn,
        });
    }
    /**
     * Imports a deployment strategy into the CDK using its ID.
     *
     * @param scope The parent construct
     * @param id The name of the deployment strategy construct
     * @param deploymentStrategyId The ID of the deployment strategy
     */
    static fromDeploymentStrategyId(scope, id, deploymentStrategyId) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const deploymentStrategyArn = stack.formatArn({
            service: 'appconfig',
            resource: 'deploymentstrategy',
            resourceName: deploymentStrategyId,
        });
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.deploymentStrategyId = deploymentStrategyId;
                this.deploymentStrategyArn = deploymentStrategyArn;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: deploymentStrategyArn,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.name,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_DeploymentStrategyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DeploymentStrategy);
            }
            throw error;
        }
        this.deploymentDurationInMinutes = props.rolloutStrategy.deploymentDuration.toMinutes();
        this.growthFactor = props.rolloutStrategy.growthFactor;
        this.description = props.description;
        this.finalBakeTimeInMinutes = props.rolloutStrategy.finalBakeTime?.toMinutes();
        this.growthType = props.rolloutStrategy.growthType;
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 64,
            separator: '-',
        });
        const resource = new aws_appconfig_1.CfnDeploymentStrategy(this, 'Resource', {
            name: this.name,
            deploymentDurationInMinutes: this.deploymentDurationInMinutes,
            growthFactor: this.growthFactor,
            replicateTo: 'NONE',
            description: this.description,
            finalBakeTimeInMinutes: this.finalBakeTimeInMinutes,
            growthType: this.growthType,
        });
        this._cfnDeploymentStrategy = resource;
        this.deploymentStrategyId = this._cfnDeploymentStrategy.ref;
        this.deploymentStrategyArn = this.stack.formatArn({
            service: 'appconfig',
            resource: 'deploymentstrategy',
            resourceName: this.deploymentStrategyId,
        });
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_a = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.DeploymentStrategy", version: "2.106.0-alpha.0" };
/**
 * Defines the growth type of the deployment strategy.
 */
var GrowthType;
(function (GrowthType) {
    /**
     * AWS AppConfig will process the deployment by increments of the growth factor
     * evenly distributed over the deployment.
     */
    GrowthType["LINEAR"] = "LINEAR";
    /**
     * AWS AppConfig will process the deployment exponentially using the following formula:
     * `G*(2^N)`. In this formula, `G` is the step percentage specified by the user and `N`
     * is the number of steps until the configuration is deployed to all targets.
     */
    GrowthType["EXPONENTIAL"] = "EXPONENTIAL";
})(GrowthType || (exports.GrowthType = GrowthType = {}));
/**
 * Defines the deployment strategy ID's of AWS AppConfig predefined strategies.
 *
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-deployment-strategy.html
 */
var PredefinedDeploymentStrategyId;
(function (PredefinedDeploymentStrategyId) {
    /**
     * **AWS Recommended**. This strategy processes the deployment exponentially using a 10% growth factor over 20 minutes.
     * AWS AppConfig recommends using this strategy for production deployments because it aligns with AWS best practices
     * for configuration deployments.
     */
    PredefinedDeploymentStrategyId["CANARY_10_PERCENT_20_MINUTES"] = "AppConfig.Canary10Percent20Minutes";
    /**
     * **Testing/Demonstration**. This strategy deploys the configuration to half of all targets every 30 seconds for a
     * one-minute deployment. AWS AppConfig recommends using this strategy only for testing or demonstration purposes because
     * it has a short duration and bake time.
     */
    PredefinedDeploymentStrategyId["LINEAR_50_PERCENT_EVERY_30_SECONDS"] = "AppConfig.Linear50PercentEvery30Seconds";
    /**
     * **AWS Recommended**. This strategy deploys the configuration to 20% of all targets every six minutes for a 30 minute deployment.
     * AWS AppConfig recommends using this strategy for production deployments because it aligns with AWS best practices
     * for configuration deployments.
     */
    PredefinedDeploymentStrategyId["LINEAR_20_PERCENT_EVERY_6_MINUTES"] = "AppConfig.Linear20PercentEvery6Minutes";
    /**
     * **Quick**. This strategy deploys the configuration to all targets immediately.
     */
    PredefinedDeploymentStrategyId["ALL_AT_ONCE"] = "AppConfig.AllAtOnce";
})(PredefinedDeploymentStrategyId || (exports.PredefinedDeploymentStrategyId = PredefinedDeploymentStrategyId = {}));
/**
 * Defines the rollout strategy for a deployment strategy and includes the growth factor,
 * deployment duration, growth type, and optionally final bake time.
 *
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-deployment-strategy.html
 */
class RolloutStrategy {
    /**
     * Build your own linear rollout strategy.
     */
    static linear(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_RolloutStrategyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.linear);
            }
            throw error;
        }
        return {
            growthFactor: props.growthFactor,
            deploymentDuration: props.deploymentDuration,
            growthType: GrowthType.LINEAR,
            finalBakeTime: props.finalBakeTime,
        };
    }
    /**
     * Build your own exponential rollout strategy.
     */
    static exponential(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_RolloutStrategyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.exponential);
            }
            throw error;
        }
        return {
            growthFactor: props.growthFactor,
            deploymentDuration: props.deploymentDuration,
            growthType: GrowthType.EXPONENTIAL,
            finalBakeTime: props.finalBakeTime,
        };
    }
}
exports.RolloutStrategy = RolloutStrategy;
_b = JSII_RTTI_SYMBOL_1;
RolloutStrategy[_b] = { fqn: "@aws-cdk/aws-appconfig-alpha.RolloutStrategy", version: "2.106.0-alpha.0" };
/**
 * **AWS Recommended**. This strategy processes the deployment exponentially using a 10% growth factor over 20 minutes.
 * AWS AppConfig recommends using this strategy for production deployments because it aligns with AWS best practices
 * for configuration deployments.
 */
RolloutStrategy.CANARY_10_PERCENT_20_MINUTES = RolloutStrategy.exponential({
    growthFactor: 10,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(20),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(10),
});
/**
 * **Testing/Demonstration**. This strategy deploys the configuration to half of all targets every 30 seconds for a
 * one-minute deployment. AWS AppConfig recommends using this strategy only for testing or demonstration purposes because
 * it has a short duration and bake time.
 */
RolloutStrategy.LINEAR_50_PERCENT_EVERY_30_SECONDS = RolloutStrategy.linear({
    growthFactor: 50,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(1),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(1),
});
/**
 * **AWS Recommended**. This strategy deploys the configuration to 20% of all targets every six minutes for a 30 minute deployment.
 * AWS AppConfig recommends using this strategy for production deployments because it aligns with AWS best practices
 * for configuration deployments.
 */
RolloutStrategy.LINEAR_20_PERCENT_EVERY_6_MINUTES = RolloutStrategy.linear({
    growthFactor: 20,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(30),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(30),
});
/**
 * **Quick**. This strategy deploys the configuration to all targets immediately.
 */
RolloutStrategy.ALL_AT_ONCE = RolloutStrategy.linear({
    growthFactor: 100,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(0),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(10),
});
//# sourceMappingURL=data:application/json;base64,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