"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitor = exports.MonitorType = exports.Environment = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
const iam = require("aws-cdk-lib/aws-iam");
const extension_1 = require("./extension");
class EnvironmentBase extends aws_cdk_lib_1.Resource {
    on(actionPoint, eventDestination, options) {
        this.extensible.on(actionPoint, eventDestination, options);
    }
    preCreateHostedConfigurationVersion(eventDestination, options) {
        this.extensible.preCreateHostedConfigurationVersion(eventDestination, options);
    }
    preStartDeployment(eventDestination, options) {
        this.extensible.preStartDeployment(eventDestination, options);
    }
    onDeploymentStart(eventDestination, options) {
        this.extensible.onDeploymentStart(eventDestination, options);
    }
    onDeploymentStep(eventDestination, options) {
        this.extensible.onDeploymentStep(eventDestination, options);
    }
    onDeploymentBaking(eventDestination, options) {
        this.extensible.onDeploymentBaking(eventDestination, options);
    }
    onDeploymentComplete(eventDestination, options) {
        this.extensible.onDeploymentComplete(eventDestination, options);
    }
    onDeploymentRolledBack(eventDestination, options) {
        this.extensible.onDeploymentRolledBack(eventDestination, options);
    }
    addExtension(extension) {
        this.extensible.addExtension(extension);
    }
}
/**
 * An AWS AppConfig environment.
 *
 * @resource AWS::AppConfig::Environment
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-environment.html
 */
class Environment extends EnvironmentBase {
    /**
     * Imports an environment into the CDK using its Amazon Resource Name (ARN).
     *
     * @param scope The parent construct
     * @param id The name of the environment construct
     * @param environmentArn The Amazon Resource Name (ARN) of the environment
     */
    static fromEnvironmentArn(scope, id, environmentArn) {
        const parsedArn = aws_cdk_lib_1.Stack.of(scope).splitArn(environmentArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Missing required /$/{applicationId}/environment//$/{environmentId} from environment ARN: ${parsedArn.resourceName}`);
        }
        const resourceName = parsedArn.resourceName.split('/');
        if (resourceName.length != 3 || !resourceName[0] || !resourceName[2]) {
            throw new Error('Missing required parameters for environment ARN: format should be /$/{applicationId}/environment//$/{environmentId}');
        }
        const applicationId = resourceName[0];
        const environmentId = resourceName[2];
        class Import extends EnvironmentBase {
            constructor() {
                super(...arguments);
                this.applicationId = applicationId;
                this.environmentId = environmentId;
                this.environmentArn = environmentArn;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: environmentArn,
        });
    }
    /**
     * Imports an environment into the CDK from its attributes.
     *
     * @param scope The parent construct
     * @param id The name of the environment construct
     * @param attrs The attributes of the environment
     */
    static fromEnvironmentAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_EnvironmentAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEnvironmentAttributes);
            }
            throw error;
        }
        const applicationId = attrs.application.applicationId;
        const environmentId = attrs.environmentId;
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const environmentArn = stack.formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${applicationId}/environment/${environmentId}`,
        });
        class Import extends EnvironmentBase {
            constructor() {
                super(...arguments);
                this.application = attrs.application;
                this.applicationId = attrs.application.applicationId;
                this.name = attrs.name;
                this.environmentId = environmentId;
                this.environmentArn = environmentArn;
                this.description = attrs.description;
                this.monitors = attrs.monitors;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: environmentArn,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.name,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_EnvironmentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Environment);
            }
            throw error;
        }
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 64,
            separator: '-',
        });
        this.application = props.application;
        this.applicationId = this.application.applicationId;
        this.description = props.description;
        this.monitors = props.monitors;
        const resource = new aws_appconfig_1.CfnEnvironment(this, 'Resource', {
            applicationId: this.applicationId,
            name: this.name,
            description: this.description,
            monitors: this.monitors?.map((monitor, index) => {
                return {
                    alarmArn: monitor.alarmArn,
                    ...(monitor.monitorType === MonitorType.CLOUDWATCH
                        ? { alarmRoleArn: monitor.alarmRoleArn || this.createAlarmRole(monitor.alarmArn, index).roleArn }
                        : { alarmRoleArn: monitor.alarmRoleArn }),
                };
            }),
        });
        this._cfnEnvironment = resource;
        this.environmentId = this._cfnEnvironment.ref;
        this.environmentArn = this.stack.formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/environment/${this.environmentId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.environmentArn, this.name);
        this.application.addExistingEnvironment(this);
    }
    createAlarmRole(alarmArn, index) {
        const policy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['cloudwatch:DescribeAlarms'],
            resources: [alarmArn],
        });
        const document = new iam.PolicyDocument({
            statements: [policy],
        });
        const role = new iam.Role(this, `Role${index}`, {
            roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('appconfig.amazonaws.com'),
            inlinePolicies: {
                ['AllowAppConfigMonitorAlarmPolicy']: document,
            },
        });
        return role;
    }
}
exports.Environment = Environment;
_a = JSII_RTTI_SYMBOL_1;
Environment[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.Environment", version: "2.106.0-alpha.0" };
/**
 * The type of Monitor.
 */
var MonitorType;
(function (MonitorType) {
    /**
     * A Monitor from a CloudWatch alarm.
     */
    MonitorType[MonitorType["CLOUDWATCH"] = 0] = "CLOUDWATCH";
    /**
     * A Monitor from a CfnEnvironment.MonitorsProperty construct.
     */
    MonitorType[MonitorType["CFN_MONITORS_PROPERTY"] = 1] = "CFN_MONITORS_PROPERTY";
})(MonitorType || (exports.MonitorType = MonitorType = {}));
/**
 * Defines monitors that will be associated with an AWS AppConfig environment.
 */
class Monitor {
    /**
     * Creates a Monitor from a CloudWatch alarm. If the alarm role is not specified, a role will
     * be generated.
     *
     * @param alarm The Amazon CloudWatch alarm.
     * @param alarmRole The IAM role for AWS AppConfig to view the alarm state.
     */
    static fromCloudWatchAlarm(alarm, alarmRole) {
        return {
            alarmArn: alarm.alarmArn,
            alarmRoleArn: alarmRole?.roleArn,
            monitorType: MonitorType.CLOUDWATCH,
        };
    }
    /**
     * Creates a Monitor from a CfnEnvironment.MonitorsProperty construct.
     *
     * @param monitorsProperty The monitors property.
     */
    static fromCfnMonitorsProperty(monitorsProperty) {
        return {
            alarmArn: monitorsProperty.alarmArn,
            alarmRoleArn: monitorsProperty.alarmRoleArn,
            monitorType: MonitorType.CFN_MONITORS_PROPERTY,
        };
    }
}
exports.Monitor = Monitor;
_b = JSII_RTTI_SYMBOL_1;
Monitor[_b] = { fqn: "@aws-cdk/aws-appconfig-alpha.Monitor", version: "2.106.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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