"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvocationType = exports.LambdaInvoke = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class LambdaInvoke extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (props.payloadResponseOnly &&
            (props.integrationPattern || props.invocationType || props.clientContext || props.qualifier)) {
            throw new Error("The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified.");
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.props.lambdaFunction.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
        if ((_b = props.retryOnServiceExceptions) !== null && _b !== void 0 ? _b : true) {
            // Best practice from https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
            this.addRetry({
                errors: ['Lambda.ServiceException', 'Lambda.AWSLambdaException', 'Lambda.SdkClientException'],
                interval: cdk.Duration.seconds(2),
                maxAttempts: 6,
                backoffRate: 2,
            });
        }
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        if (this.props.payloadResponseOnly) {
            return {
                Resource: this.props.lambdaFunction.functionArn,
                ...this.props.payload && { Parameters: sfn.FieldUtils.renderObject(this.props.payload.value) },
            };
        }
        else {
            return {
                Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
                Parameters: sfn.FieldUtils.renderObject({
                    FunctionName: this.props.lambdaFunction.functionArn,
                    Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                    InvocationType: this.props.invocationType,
                    ClientContext: this.props.clientContext,
                    Qualifier: this.props.qualifier,
                }),
            };
        }
    }
}
exports.LambdaInvoke = LambdaInvoke;
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * Invoke the function synchronously.
     *
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke the function asynchronously.
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * Validate parameter values and verify that the user or role has permission to invoke the function.
     */
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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