"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const batch = require("@aws-cdk/aws-batch");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let batchJobDefinition;
let batchJobQueue;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    batchJobDefinition = new batch.JobDefinition(stack, 'JobDefinition', {
        container: {
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, 'batchjob-image')),
        },
    });
    batchJobQueue = new batch.JobQueue(stack, 'JobQueue', {
        computeEnvironments: [
            {
                order: 1,
                computeEnvironment: new batch.ComputeEnvironment(stack, 'ComputeEnv', {
                    computeResources: { vpc: new ec2.Vpc(stack, 'vpc') },
                }),
            },
        ],
    });
});
test('Task with only the required parameters', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunBatchJob({
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::batch:submitJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobDefinition: { Ref: 'JobDefinition24FFE3ED' },
            JobName: 'JobName',
            JobQueue: { Ref: 'JobQueueEE3AD499' },
        },
    });
});
test('Task with all the parameters', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunBatchJob({
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            arraySize: 15,
            containerOverrides: {
                command: ['sudo', 'rm'],
                environment: { key: 'value' },
                instanceType: new ec2.InstanceType('MULTI'),
                memory: 1024,
                gpuCount: 1,
                vcpus: 10,
            },
            dependsOn: [{ jobId: '1234', type: 'some_type' }],
            payload: {
                foo: sfn.JsonPath.stringAt('$.bar'),
            },
            attempts: 3,
            timeout: cdk.Duration.seconds(60),
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::batch:submitJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobDefinition: { Ref: 'JobDefinition24FFE3ED' },
            JobName: 'JobName',
            JobQueue: { Ref: 'JobQueueEE3AD499' },
            ArrayProperties: { Size: 15 },
            ContainerOverrides: {
                Command: ['sudo', 'rm'],
                Environment: [{ Name: 'key', Value: 'value' }],
                InstanceType: 'MULTI',
                Memory: 1024,
                ResourceRequirements: [{ Type: 'GPU', Value: '1' }],
                Vcpus: 10,
            },
            DependsOn: [{ JobId: '1234', Type: 'some_type' }],
            Parameters: { 'foo.$': '$.bar' },
            RetryStrategy: { Attempts: 3 },
            Timeout: { AttemptDurationSeconds: 60 },
        },
    });
});
test('supports tokens', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunBatchJob({
            jobDefinition: batchJobDefinition,
            jobName: sfn.JsonPath.stringAt('$.jobName'),
            jobQueue: batchJobQueue,
            arraySize: sfn.JsonPath.numberAt('$.arraySize'),
            timeout: cdk.Duration.seconds(sfn.JsonPath.numberAt('$.timeout')),
            attempts: sfn.JsonPath.numberAt('$.attempts'),
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::batch:submitJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'JobDefinition': { Ref: 'JobDefinition24FFE3ED' },
            'JobName.$': '$.jobName',
            'JobQueue': { Ref: 'JobQueueEE3AD499' },
            'ArrayProperties': {
                'Size.$': '$.arraySize',
            },
            'RetryStrategy': {
                'Attempts.$': '$.attempts',
            },
            'Timeout': {
                'AttemptDurationSeconds.$': '$.timeout',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            }),
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call RunBatchJob./i);
});
test('Task throws if environment in containerOverrides contain env with name starting with AWS_BATCH', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                containerOverrides: {
                    environment: { AWS_BATCH_MY_NAME: 'MY_VALUE' },
                },
            }),
        });
    }).toThrow(/Invalid environment variable name: AWS_BATCH_MY_NAME. Environment variable names starting with 'AWS_BATCH' are reserved./i);
});
test('Task throws if arraySize is out of limits 2-10000', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                arraySize: 1,
            }),
        });
    }).toThrow(/arraySize must be between 2 and 10,000/);
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                arraySize: 10001,
            }),
        });
    }).toThrow(/arraySize must be between 2 and 10,000/);
});
test('Task throws if dependencies exceeds 20', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                dependsOn: [...Array(21).keys()].map(i => ({
                    jobId: `${i}`,
                    type: `some_type-${i}`,
                })),
            }),
        });
    }).toThrow(/dependencies must be 20 or less/);
});
test('Task throws if attempts is out of limits 1-10', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                attempts: 0,
            }),
        });
    }).toThrow(/attempts must be between 1 and 10/);
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                attempts: 11,
            }),
        });
    }).toThrow(/attempts must be between 1 and 10/);
});
test('Task throws if timeout is less than 60 sec', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunBatchJob({
                jobDefinition: batchJobDefinition,
                jobName: 'JobName',
                jobQueue: batchJobQueue,
                timeout: cdk.Duration.seconds(59),
            }),
        });
    }).toThrow(/timeout must be greater than 60 seconds/);
});
//# sourceMappingURL=data:application/json;base64,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