"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sns = require("@aws-cdk/aws-sns");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
test('Publish literal message to SNS topic', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', {
        task: new tasks.PublishToTopic(topic, {
            message: sfn.TaskInput.fromText('Publish this message'),
        }),
    });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sns:publish',
                ],
            ],
        },
        End: true,
        Parameters: {
            TopicArn: { Ref: 'TopicBFC7AF6E' },
            Message: 'Publish this message',
        },
    });
});
test('Publish JSON to SNS topic with task token', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', {
        task: new tasks.PublishToTopic(topic, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            message: sfn.TaskInput.fromObject({
                Input: 'Publish this message',
                Token: sfn.JsonPath.taskToken,
            }),
        }),
    });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sns:publish.waitForTaskToken',
                ],
            ],
        },
        End: true,
        Parameters: {
            TopicArn: { Ref: 'TopicBFC7AF6E' },
            Message: {
                'Input': 'Publish this message',
                'Token.$': '$$.Task.Token',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied but task token is not included in message', () => {
    expect(() => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sfn.Task(stack, 'Publish', {
            task: new tasks.PublishToTopic(topic, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                message: sfn.TaskInput.fromText('Publish this message'),
            }),
        });
        // THEN
    }).toThrow(/Task Token is missing in message/i);
});
test('Publish to topic with ARN from payload', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = sns.Topic.fromTopicArn(stack, 'Topic', sfn.JsonPath.stringAt('$.topicArn'));
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', {
        task: new tasks.PublishToTopic(topic, {
            message: sfn.TaskInput.fromText('Publish this message'),
        }),
    });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sns:publish',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TopicArn.$': '$.topicArn',
            'Message': 'Publish this message',
        },
    });
});
test('Task throws if SYNC is supplied as service integration pattern', () => {
    expect(() => {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        new sfn.Task(stack, 'Publish', {
            task: new tasks.PublishToTopic(topic, {
                integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
                message: sfn.TaskInput.fromText('Publish this message'),
            }),
        });
    }).toThrow(/Invalid Service Integration Pattern: SYNC is not supported to call SNS./i);
});
//# sourceMappingURL=data:application/json;base64,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