"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sqs = require("@aws-cdk/aws-sqs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const send_message_1 = require("../../lib/sqs/send-message");
describe('SqsSendMessage', () => {
    let stack;
    let queue;
    beforeEach(() => {
        // GIVEN
        stack = new cdk.Stack();
        queue = new sqs.Queue(stack, 'Queue');
    });
    test('default settings', () => {
        // WHEN
        const task = new send_message_1.SqsSendMessage(stack, 'SendMessage', {
            queue,
            messageBody: sfn.TaskInput.fromText('a simple message'),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sqs:sendMessage',
                    ],
                ],
            },
            End: true,
            Parameters: {
                QueueUrl: { Ref: 'Queue4A7E3555' },
                MessageBody: 'a simple message',
            },
        });
    });
    test('send message with deduplication and delay', () => {
        // WHEN
        const task = new send_message_1.SqsSendMessage(stack, 'Send', {
            queue,
            messageBody: sfn.TaskInput.fromText('Send this message'),
            messageDeduplicationId: sfn.JsonPath.stringAt('$.deduping'),
            comment: 'sending a message to my SQS queue',
            delay: cdk.Duration.seconds(30),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sqs:sendMessage',
                    ],
                ],
            },
            End: true,
            Parameters: {
                'QueueUrl': { Ref: 'Queue4A7E3555' },
                'MessageBody': 'Send this message',
                'MessageDeduplicationId.$': '$.deduping',
                'DelaySeconds': 30,
            },
            Comment: 'sending a message to my SQS queue',
        });
    });
    test('send message to SQS and wait for task token', () => {
        // WHEN
        const task = new send_message_1.SqsSendMessage(stack, 'Send', {
            queue,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            messageBody: sfn.TaskInput.fromObject({
                Input: 'Send this message',
                Token: sfn.JsonPath.taskToken,
            }),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sqs:sendMessage.waitForTaskToken',
                    ],
                ],
            },
            End: true,
            Parameters: {
                QueueUrl: { Ref: 'Queue4A7E3555' },
                MessageBody: {
                    'Input': 'Send this message',
                    'Token.$': '$$.Task.Token',
                },
            },
        });
    });
    test('Message body can come from state', () => {
        // WHEN
        const task = new send_message_1.SqsSendMessage(stack, 'Send', {
            queue,
            messageBody: sfn.TaskInput.fromDataAt('$.theMessage'),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sqs:sendMessage',
                    ],
                ],
            },
            End: true,
            Parameters: {
                'QueueUrl': { Ref: 'Queue4A7E3555' },
                'MessageBody.$': '$.theMessage',
            },
        });
    });
    test('send message with message body defined as an object', () => {
        // WHEN
        const task = new send_message_1.SqsSendMessage(stack, 'Send', {
            queue,
            messageBody: sfn.TaskInput.fromObject({
                literal: 'literal',
                SomeInput: sfn.JsonPath.stringAt('$.theMessage'),
            }),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sqs:sendMessage',
                    ],
                ],
            },
            End: true,
            Parameters: {
                QueueUrl: { Ref: 'Queue4A7E3555' },
                MessageBody: {
                    'literal': 'literal',
                    'SomeInput.$': '$.theMessage',
                },
            },
        });
    });
    test('message body can use references', () => {
        // WHEN
        const task = new send_message_1.SqsSendMessage(stack, 'Send', {
            queue,
            messageBody: sfn.TaskInput.fromObject({
                queueArn: queue.queueArn,
            }),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sqs:sendMessage',
                    ],
                ],
            },
            End: true,
            Parameters: {
                QueueUrl: { Ref: 'Queue4A7E3555' },
                MessageBody: {
                    queueArn: { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] },
                },
            },
        });
    });
    test('fails when WAIT_FOR_TASK_TOKEN integration pattern is used without supplying a task token in message body', () => {
        expect(() => {
            new send_message_1.SqsSendMessage(stack, 'Send', {
                queue,
                integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                messageBody: sfn.TaskInput.fromText('Send this message'),
            });
        }).toThrow(/Task Token is required in `messageBody` Use JsonPath.taskToken to set the token./);
    });
    test('fails when RUN_JOB integration pattern is used', () => {
        expect(() => {
            new send_message_1.SqsSendMessage(stack, 'Send', {
                queue,
                integrationPattern: sfn.IntegrationPattern.RUN_JOB,
                messageBody: sfn.TaskInput.fromText('Send this message'),
            });
        }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,WAIT_FOR_TASK_TOKEN. Received: RUN_JOB/);
    });
});
//# sourceMappingURL=data:application/json;base64,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