"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        const parts = core_1.Stack.of(this).parseArn(this.repositoryArn);
        return `${parts.account}.dkr.ecr.${parts.region}.amazonaws.com/${this.repositoryName}${tagSuffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                }
            }
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailImagePushed(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: ['*'],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, "ecr:BatchCheckLayerAvailability", "ecr:GetDownloadUrlForLayer", "ecr:BatchGetImage");
        iam.Grant.addToPrincipal({
            grantee,
            actions: ["ecr:GetAuthorizationToken"],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, "ecr:PutImage", "ecr:InitiateLayerUpload", "ecr:UploadLayerPart", "ecr:CompleteLayerUpload");
    }
}
exports.RepositoryBase = RepositoryBase;
/**
 * Define an ECR repository
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.anyValue({ produce: () => this.renderLifecyclePolicy() }),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a repository
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName, scope) {
        return core_1.Stack.of(scope).formatArn({
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName
        });
    }
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule) {
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: rule.rulePriority !== undefined ? rule.rulePriority : autoPrio++
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
}
exports.Repository = Repository;
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* IMAGE_COUNT_MORE_THAN */,
            countNumber: rule.maxImageAge !== undefined ? rule.maxImageAge.toDays() : rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire'
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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