"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const ecr = require("../lib");
module.exports = {
    'construct repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                Repo02AC86CF: {
                    Type: "AWS::ECR::Repository",
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                }
            }
        });
        test.done();
    },
    'tag-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagPrefixList: ['abc'], maxImageCount: 1 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"abc\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":1},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'add day-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = new ecr.Repository(stack, 'Repo');
        repo.addLifecycleRule({
            maxImageAge: cdk.Duration.days(5),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"sinceImagePushed\",\"countNumber\":5,\"countUnit\":\"days\"},\"action\":{\"type\":\"expire\"}}]}",
            }
        }));
        test.done();
    },
    'add count-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({
            maxImageCount: 5,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}",
            }
        }));
        test.done();
    },
    'mixing numbered and unnumbered rules'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['a'], maxImageCount: 5 });
        repo.addLifecycleRule({ rulePriority: 10, tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['b'], maxImageCount: 5 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":10,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"b\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}},{\"rulePriority\":11,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"a\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'tagstatus Any is automatically sorted to the back'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ maxImageCount: 5 });
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['important'], maxImageCount: 999 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"important\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":999},\"action\":{\"type\":\"expire\"}},{\"rulePriority\":2,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'lifecycle rules can be added upon initialization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo', {
            lifecycleRules: [
                { maxImageCount: 3 }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            "LifecyclePolicy": {
                // tslint:disable-next-line:max-line-length
                "LifecyclePolicyText": "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":3},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'calculate repository URI'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        const uri = repo.repositoryUri;
        // THEN
        const arnSplit = { 'Fn::Split': [':', { 'Fn::GetAtt': ['Repo02AC86CF', 'Arn'] }] };
        test.deepEqual(stack.resolve(uri), { 'Fn::Join': ['', [
                    { 'Fn::Select': [4, arnSplit] },
                    '.dkr.ecr.',
                    { 'Fn::Select': [3, arnSplit] },
                    '.amazonaws.com/',
                    { Ref: 'Repo02AC86CF' }
                ]] });
        test.done();
    },
    'import with concrete arn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo2 = ecr.Repository.fromRepositoryArn(stack, 'repo', 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        // THEN
        test.deepEqual(stack.resolve(repo2.repositoryArn), 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        test.deepEqual(stack.resolve(repo2.repositoryName), 'foo/bar/foo/fooo');
        test.done();
    },
    'fails if importing with token arn and no name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => ecr.Repository.fromRepositoryArn(stack, 'arn', cdk.Fn.getAtt('Boom', 'Boom').toString()), /\"repositoryArn\" is a late-bound value, and therefore \"repositoryName\" is required\. Use \`fromRepositoryAttributes\` instead/);
        test.done();
    },
    'import with token arn and repository name (see awslabs/aws-cdk#1232)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: cdk.Fn.getAtt('Boom', 'Arn').toString(),
            repositoryName: cdk.Fn.getAtt('Boom', 'Name').toString()
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), { 'Fn::GetAtt': ['Boom', 'Arn'] });
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.done();
    },
    'import only with a repository name (arn is deduced)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryName(stack, 'just-name', 'my-repo');
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/my-repo'
                ]
            ]
        });
        test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
        test.done();
    },
    'arnForLocalRepository can be used to render an ARN for a local repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repoName = cdk.Fn.getAtt('Boom', 'Name').toString();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: ecr.Repository.arnForLocalRepository(repoName, stack),
            repositoryName: repoName
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/',
                    { 'Fn::GetAtt': ['Boom', 'Name'] }
                ]]
        });
        test.done();
    },
    'resource policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addToResourcePolicy(new iam.PolicyStatement({ actions: ['*'] }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            RepositoryPolicyText: {
                Statement: [
                    {
                        Action: "*",
                        Effect: "Allow"
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'events': {
        'onImagePushed without imageTag creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onCloudTrailImagePushed('EventRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: 'ID' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "eventName": [
                            "PutImage",
                        ],
                        "requestParameters": {
                            "repositoryName": [
                                {
                                    "Ref": "Repo02AC86CF"
                                }
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        }
    },
    'removal policy is "Retain" by default'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            "Type": "AWS::ECR::Repository",
            "DeletionPolicy": "Retain"
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    '"Delete" removal policy can be set explicitly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo', {
            removalPolicy: core_1.RemovalPolicy.DESTROY
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            "Type": "AWS::ECR::Repository",
            "DeletionPolicy": "Delete"
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'grant adds appropriate resource-*'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const repo = new ecr.Repository(stack, 'TestHarnessRepo');
        // WHEN
        repo.grantPull(new iam.AnyPrincipal());
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            "RepositoryPolicyText": {
                "Statement": [
                    {
                        "Action": [
                            "ecr:BatchCheckLayerAvailability",
                            "ecr:GetDownloadUrlForLayer",
                            "ecr:BatchGetImage"
                        ],
                        "Effect": "Allow",
                        "Principal": "*",
                        "Resource": "*",
                    }
                ],
                "Version": "2012-10-17"
            }
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5yZXBvc2l0b3J5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5yZXBvc2l0b3J5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBdUY7QUFDdkYsd0NBQXlDO0FBQ3pDLHFDQUFzQztBQUN0Qyx3Q0FBcUQ7QUFFckQsOEJBQStCO0FBSS9CLGlCQUFTO0lBQ1Asc0JBQXNCLENBQUMsSUFBVTtRQUMvQixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFbEMsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDcEIsU0FBUyxFQUFFO2dCQUNULFlBQVksRUFBRTtvQkFDWixJQUFJLEVBQUUsc0JBQXNCO29CQUM1QixjQUFjLEVBQUUsUUFBUTtvQkFDeEIsbUJBQW1CLEVBQUUsUUFBUTtpQkFDOUI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0QkFBNEIsQ0FBQyxJQUFVO1FBQ3JDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRS9DLE9BQU87UUFDUCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxhQUFhLEVBQUUsQ0FBQyxLQUFLLENBQUMsRUFBRSxhQUFhLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUVwRSxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1lBQ3BELGVBQWUsRUFBRTtnQkFDZiwyQ0FBMkM7Z0JBQzNDLG1CQUFtQixFQUFFLGlNQUFpTTthQUN2TjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdDQUFnQyxDQUFDLElBQVU7UUFDekMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztZQUNwQixXQUFXLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1NBQ2xDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7WUFDcEQsZUFBZSxFQUFFO2dCQUNmLDJDQUEyQztnQkFDM0MsbUJBQW1CLEVBQUUsdUxBQXVMO2FBQzdNO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0NBQWtDLENBQUMsSUFBVTtRQUMzQyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUvQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLGdCQUFnQixDQUFDO1lBQ3BCLGFBQWEsRUFBRSxDQUFDO1NBQ2pCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7WUFDcEQsZUFBZSxFQUFFO2dCQUNmLDJDQUEyQztnQkFDM0MsbUJBQW1CLEVBQUUsa0tBQWtLO2FBQ3hMO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0NBQXNDLENBQUMsSUFBVTtRQUMvQyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUvQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLGFBQWEsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ25HLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLFlBQVksRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLGFBQWEsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXJILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7WUFDcEQsZUFBZSxFQUFFO2dCQUNmLDJDQUEyQztnQkFDM0MsbUJBQW1CLEVBQUUsaVhBQWlYO2FBQ3ZZO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbURBQW1ELENBQUMsSUFBVTtRQUM1RCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUvQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsYUFBYSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDNUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRSxDQUFDLFdBQVcsQ0FBQyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBRTdHLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7WUFDcEQsZUFBZSxFQUFFO2dCQUNmLDJDQUEyQztnQkFDM0MsbUJBQW1CLEVBQUUsNFZBQTRWO2FBQ2xYO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0RBQWtELENBQUMsSUFBVTtRQUMzRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQ2hDLGNBQWMsRUFBRTtnQkFDZCxFQUFFLGFBQWEsRUFBRSxDQUFDLEVBQUU7YUFDckI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1lBQ3BELGlCQUFpQixFQUFFO2dCQUNqQiwyQ0FBMkM7Z0JBQzNDLHFCQUFxQixFQUFFLGtLQUFrSzthQUMxTDtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBCQUEwQixDQUFDLElBQVU7UUFDbkMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFL0MsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUM7UUFFL0IsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUUsR0FBRyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsY0FBYyxFQUFFLEtBQUssQ0FBRSxFQUFFLENBQUUsRUFBRSxDQUFDO1FBQ3ZGLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFFLEVBQUUsRUFBRTtvQkFDckQsRUFBRSxZQUFZLEVBQUUsQ0FBRSxDQUFDLEVBQUUsUUFBUSxDQUFFLEVBQUU7b0JBQ2pDLFdBQVc7b0JBQ1gsRUFBRSxZQUFZLEVBQUUsQ0FBRSxDQUFDLEVBQUUsUUFBUSxDQUFFLEVBQUU7b0JBQ2pDLGlCQUFpQjtvQkFDakIsRUFBRSxHQUFHLEVBQUUsY0FBYyxFQUFFO2lCQUN4QixDQUFDLEVBQUMsQ0FBQyxDQUFDO1FBRUwsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBCQUEwQixDQUFDLElBQVU7UUFDbkMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsZ0VBQWdFLENBQUMsQ0FBQztRQUVoSSxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsRUFBRSxnRUFBZ0UsQ0FBQyxDQUFDO1FBQ3JILElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUV4RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0NBQStDLENBQUMsSUFBVTtRQUN4RCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsWUFBWTtRQUNaLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxFQUN4RyxrSUFBa0ksQ0FBQyxDQUFDO1FBRXRJLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzRUFBc0UsQ0FBQyxJQUFVO1FBQy9FLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixPQUFPO1FBQ1AsTUFBTSxJQUFJLEdBQUcsR0FBRyxDQUFDLFVBQVUsQ0FBQyx3QkFBd0IsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQ2xFLGFBQWEsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUMsUUFBUSxFQUFFO1lBQ3RELGNBQWMsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUMsUUFBUSxFQUFFO1NBQ3pELENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsTUFBTSxFQUFFLEtBQUssQ0FBRSxFQUFFLENBQUMsQ0FBQztRQUN2RixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBRSxFQUFFLENBQUMsQ0FBQztRQUN6RixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscURBQXFELENBQUMsSUFBVTtRQUM5RCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLEdBQUcsQ0FBQyxVQUFVLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUU5RSxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNoRCxVQUFVLEVBQUUsQ0FBRSxFQUFFLEVBQUU7b0JBQ2hCLE1BQU07b0JBQ04sRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3pCLE9BQU87b0JBQ1AsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUN0QixHQUFHO29CQUNILEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO29CQUN6QixxQkFBcUI7aUJBQUU7YUFDeEI7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzlELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyRUFBMkUsQ0FBQyxJQUFVO1FBQ3BGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7UUFFMUQsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLEdBQUcsQ0FBQyxVQUFVLENBQUMsd0JBQXdCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUNsRSxhQUFhLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDO1lBQ3BFLGNBQWMsRUFBRSxRQUFRO1NBQ3pCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBRSxFQUFFLENBQUMsQ0FBQztRQUN6RixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQ2xELFVBQVUsRUFBRSxDQUFFLEVBQUUsRUFBRTtvQkFDaEIsTUFBTTtvQkFDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtvQkFDekIsT0FBTztvQkFDUCxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQ3RCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3pCLGNBQWM7b0JBQ2QsRUFBRSxZQUFZLEVBQUUsQ0FBRSxNQUFNLEVBQUUsTUFBTSxDQUFFLEVBQUU7aUJBQUUsQ0FBRTtTQUN6QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsaUJBQWlCLENBQUMsSUFBVTtRQUMxQixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUvQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXRFLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7WUFDcEQsb0JBQW9CLEVBQUU7Z0JBQ3BCLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUUsR0FBRzt3QkFDWCxNQUFNLEVBQUUsT0FBTztxQkFDaEI7aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFLFlBQVk7YUFDdEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxRQUFRLEVBQUU7UUFDUiwwREFBMEQsQ0FBQyxJQUFVO1lBQ25FLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFFL0MsSUFBSSxDQUFDLHVCQUF1QixDQUFDLFdBQVcsRUFBRTtnQkFDeEMsTUFBTSxFQUFFO29CQUNOLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQ3ZDO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxtQkFBbUIsRUFBRTtnQkFDckQsY0FBYyxFQUFFO29CQUNkLFFBQVEsRUFBRTt3QkFDUixTQUFTO3FCQUNWO29CQUNELFFBQVEsRUFBRTt3QkFDUixXQUFXLEVBQUU7NEJBQ1gsVUFBVTt5QkFDWDt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsZ0JBQWdCLEVBQUU7Z0NBQ2hCO29DQUNFLEtBQUssRUFBRSxjQUFjO2lDQUN0Qjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRCxPQUFPLEVBQUUsU0FBUzthQUNuQixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELHVDQUF1QyxDQUFDLElBQVU7UUFDaEQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRWxDLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7WUFDcEQsTUFBTSxFQUFFLHNCQUFzQjtZQUM5QixnQkFBZ0IsRUFBRSxRQUFRO1NBQzNCLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFDckMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtDQUErQyxDQUFDLElBQVU7UUFDeEQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUNoQyxhQUFhLEVBQUUsb0JBQWEsQ0FBQyxPQUFPO1NBQ3JDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7WUFDcEQsTUFBTSxFQUFFLHNCQUFzQjtZQUM5QixnQkFBZ0IsRUFBRSxRQUFRO1NBQzNCLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFDckMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1DQUFtQyxDQUFDLElBQVU7UUFDNUMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBRTFELE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksR0FBRyxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUM7UUFFdkMsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQkFBc0IsRUFBRTtZQUNwRCxzQkFBc0IsRUFBRTtnQkFDdEIsV0FBVyxFQUFFO29CQUNYO3dCQUNFLFFBQVEsRUFBRTs0QkFDUixpQ0FBaUM7NEJBQ2pDLDRCQUE0Qjs0QkFDNUIsbUJBQW1CO3lCQUNwQjt3QkFDRCxRQUFRLEVBQUUsT0FBTzt3QkFDakIsV0FBVyxFQUFFLEdBQUc7d0JBQ2hCLFVBQVUsRUFBRSxHQUFHO3FCQUNoQjtpQkFDRjtnQkFDRCxTQUFTLEVBQUUsWUFBWTthQUN4QjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZSwgaGF2ZVJlc291cmNlTGlrZSwgUmVzb3VyY2VQYXJ0IH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCBpYW0gPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtaWFtJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY29yZScpO1xuaW1wb3J0IHsgUmVtb3ZhbFBvbGljeSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgZWNyID0gcmVxdWlyZSgnLi4vbGliJyk7XG5cbi8vIHRzbGludDpkaXNhYmxlOm9iamVjdC1saXRlcmFsLWtleS1xdW90ZXNcblxuZXhwb3J0ID0ge1xuICAnY29uc3RydWN0IHJlcG9zaXRvcnknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBlY3IuUmVwb3NpdG9yeShzdGFjaywgJ1JlcG8nKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgUmVzb3VyY2VzOiB7XG4gICAgICAgIFJlcG8wMkFDODZDRjoge1xuICAgICAgICAgIFR5cGU6IFwiQVdTOjpFQ1I6OlJlcG9zaXRvcnlcIixcbiAgICAgICAgICBEZWxldGlvblBvbGljeTogXCJSZXRhaW5cIixcbiAgICAgICAgICBVcGRhdGVSZXBsYWNlUG9saWN5OiBcIlJldGFpblwiLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndGFnLWJhc2VkIGxpZmVjeWNsZSBwb2xpY3knKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHJlcG8gPSBuZXcgZWNyLlJlcG9zaXRvcnkoc3RhY2ssICdSZXBvJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgcmVwby5hZGRMaWZlY3ljbGVSdWxlKHsgdGFnUHJlZml4TGlzdDogWydhYmMnXSwgbWF4SW1hZ2VDb3VudDogMSB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQ1I6OlJlcG9zaXRvcnknLCB7XG4gICAgICBMaWZlY3ljbGVQb2xpY3k6IHtcbiAgICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgICAgICBMaWZlY3ljbGVQb2xpY3lUZXh0OiBcIntcXFwicnVsZXNcXFwiOlt7XFxcInJ1bGVQcmlvcml0eVxcXCI6MSxcXFwic2VsZWN0aW9uXFxcIjp7XFxcInRhZ1N0YXR1c1xcXCI6XFxcInRhZ2dlZFxcXCIsXFxcInRhZ1ByZWZpeExpc3RcXFwiOltcXFwiYWJjXFxcIl0sXFxcImNvdW50VHlwZVxcXCI6XFxcImltYWdlQ291bnRNb3JlVGhhblxcXCIsXFxcImNvdW50TnVtYmVyXFxcIjoxfSxcXFwiYWN0aW9uXFxcIjp7XFxcInR5cGVcXFwiOlxcXCJleHBpcmVcXFwifX1dfVwiXG4gICAgICB9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2FkZCBkYXktYmFzZWQgbGlmZWN5Y2xlIHBvbGljeScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVwbyA9IG5ldyBlY3IuUmVwb3NpdG9yeShzdGFjaywgJ1JlcG8nKTtcbiAgICByZXBvLmFkZExpZmVjeWNsZVJ1bGUoe1xuICAgICAgbWF4SW1hZ2VBZ2U6IGNkay5EdXJhdGlvbi5kYXlzKDUpLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUjo6UmVwb3NpdG9yeScsIHtcbiAgICAgIExpZmVjeWNsZVBvbGljeToge1xuICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgIExpZmVjeWNsZVBvbGljeVRleHQ6IFwie1xcXCJydWxlc1xcXCI6W3tcXFwicnVsZVByaW9yaXR5XFxcIjoxLFxcXCJzZWxlY3Rpb25cXFwiOntcXFwidGFnU3RhdHVzXFxcIjpcXFwiYW55XFxcIixcXFwiY291bnRUeXBlXFxcIjpcXFwic2luY2VJbWFnZVB1c2hlZFxcXCIsXFxcImNvdW50TnVtYmVyXFxcIjo1LFxcXCJjb3VudFVuaXRcXFwiOlxcXCJkYXlzXFxcIn0sXFxcImFjdGlvblxcXCI6e1xcXCJ0eXBlXFxcIjpcXFwiZXhwaXJlXFxcIn19XX1cIixcbiAgICAgIH1cbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkIGNvdW50LWJhc2VkIGxpZmVjeWNsZSBwb2xpY3knKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHJlcG8gPSBuZXcgZWNyLlJlcG9zaXRvcnkoc3RhY2ssICdSZXBvJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgcmVwby5hZGRMaWZlY3ljbGVSdWxlKHtcbiAgICAgIG1heEltYWdlQ291bnQ6IDUsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUNSOjpSZXBvc2l0b3J5Jywge1xuICAgICAgTGlmZWN5Y2xlUG9saWN5OiB7XG4gICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgTGlmZWN5Y2xlUG9saWN5VGV4dDogXCJ7XFxcInJ1bGVzXFxcIjpbe1xcXCJydWxlUHJpb3JpdHlcXFwiOjEsXFxcInNlbGVjdGlvblxcXCI6e1xcXCJ0YWdTdGF0dXNcXFwiOlxcXCJhbnlcXFwiLFxcXCJjb3VudFR5cGVcXFwiOlxcXCJpbWFnZUNvdW50TW9yZVRoYW5cXFwiLFxcXCJjb3VudE51bWJlclxcXCI6NX0sXFxcImFjdGlvblxcXCI6e1xcXCJ0eXBlXFxcIjpcXFwiZXhwaXJlXFxcIn19XX1cIixcbiAgICAgIH1cbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbWl4aW5nIG51bWJlcmVkIGFuZCB1bm51bWJlcmVkIHJ1bGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCByZXBvID0gbmV3IGVjci5SZXBvc2l0b3J5KHN0YWNrLCAnUmVwbycpO1xuXG4gICAgLy8gV0hFTlxuICAgIHJlcG8uYWRkTGlmZWN5Y2xlUnVsZSh7IHRhZ1N0YXR1czogZWNyLlRhZ1N0YXR1cy5UQUdHRUQsIHRhZ1ByZWZpeExpc3Q6IFsnYSddLCBtYXhJbWFnZUNvdW50OiA1IH0pO1xuICAgIHJlcG8uYWRkTGlmZWN5Y2xlUnVsZSh7IHJ1bGVQcmlvcml0eTogMTAsIHRhZ1N0YXR1czogZWNyLlRhZ1N0YXR1cy5UQUdHRUQsIHRhZ1ByZWZpeExpc3Q6IFsnYiddLCBtYXhJbWFnZUNvdW50OiA1IH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUjo6UmVwb3NpdG9yeScsIHtcbiAgICAgIExpZmVjeWNsZVBvbGljeToge1xuICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgIExpZmVjeWNsZVBvbGljeVRleHQ6IFwie1xcXCJydWxlc1xcXCI6W3tcXFwicnVsZVByaW9yaXR5XFxcIjoxMCxcXFwic2VsZWN0aW9uXFxcIjp7XFxcInRhZ1N0YXR1c1xcXCI6XFxcInRhZ2dlZFxcXCIsXFxcInRhZ1ByZWZpeExpc3RcXFwiOltcXFwiYlxcXCJdLFxcXCJjb3VudFR5cGVcXFwiOlxcXCJpbWFnZUNvdW50TW9yZVRoYW5cXFwiLFxcXCJjb3VudE51bWJlclxcXCI6NX0sXFxcImFjdGlvblxcXCI6e1xcXCJ0eXBlXFxcIjpcXFwiZXhwaXJlXFxcIn19LHtcXFwicnVsZVByaW9yaXR5XFxcIjoxMSxcXFwic2VsZWN0aW9uXFxcIjp7XFxcInRhZ1N0YXR1c1xcXCI6XFxcInRhZ2dlZFxcXCIsXFxcInRhZ1ByZWZpeExpc3RcXFwiOltcXFwiYVxcXCJdLFxcXCJjb3VudFR5cGVcXFwiOlxcXCJpbWFnZUNvdW50TW9yZVRoYW5cXFwiLFxcXCJjb3VudE51bWJlclxcXCI6NX0sXFxcImFjdGlvblxcXCI6e1xcXCJ0eXBlXFxcIjpcXFwiZXhwaXJlXFxcIn19XX1cIlxuICAgICAgfVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0YWdzdGF0dXMgQW55IGlzIGF1dG9tYXRpY2FsbHkgc29ydGVkIHRvIHRoZSBiYWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCByZXBvID0gbmV3IGVjci5SZXBvc2l0b3J5KHN0YWNrLCAnUmVwbycpO1xuXG4gICAgLy8gV0hFTlxuICAgIHJlcG8uYWRkTGlmZWN5Y2xlUnVsZSh7IG1heEltYWdlQ291bnQ6IDUgfSk7XG4gICAgcmVwby5hZGRMaWZlY3ljbGVSdWxlKHsgdGFnU3RhdHVzOiBlY3IuVGFnU3RhdHVzLlRBR0dFRCwgdGFnUHJlZml4TGlzdDogWydpbXBvcnRhbnQnXSwgbWF4SW1hZ2VDb3VudDogOTk5IH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUjo6UmVwb3NpdG9yeScsIHtcbiAgICAgIExpZmVjeWNsZVBvbGljeToge1xuICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgIExpZmVjeWNsZVBvbGljeVRleHQ6IFwie1xcXCJydWxlc1xcXCI6W3tcXFwicnVsZVByaW9yaXR5XFxcIjoxLFxcXCJzZWxlY3Rpb25cXFwiOntcXFwidGFnU3RhdHVzXFxcIjpcXFwidGFnZ2VkXFxcIixcXFwidGFnUHJlZml4TGlzdFxcXCI6W1xcXCJpbXBvcnRhbnRcXFwiXSxcXFwiY291bnRUeXBlXFxcIjpcXFwiaW1hZ2VDb3VudE1vcmVUaGFuXFxcIixcXFwiY291bnROdW1iZXJcXFwiOjk5OX0sXFxcImFjdGlvblxcXCI6e1xcXCJ0eXBlXFxcIjpcXFwiZXhwaXJlXFxcIn19LHtcXFwicnVsZVByaW9yaXR5XFxcIjoyLFxcXCJzZWxlY3Rpb25cXFwiOntcXFwidGFnU3RhdHVzXFxcIjpcXFwiYW55XFxcIixcXFwiY291bnRUeXBlXFxcIjpcXFwiaW1hZ2VDb3VudE1vcmVUaGFuXFxcIixcXFwiY291bnROdW1iZXJcXFwiOjV9LFxcXCJhY3Rpb25cXFwiOntcXFwidHlwZVxcXCI6XFxcImV4cGlyZVxcXCJ9fV19XCJcbiAgICAgIH1cbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbGlmZWN5Y2xlIHJ1bGVzIGNhbiBiZSBhZGRlZCB1cG9uIGluaXRpYWxpemF0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgZWNyLlJlcG9zaXRvcnkoc3RhY2ssICdSZXBvJywge1xuICAgICAgbGlmZWN5Y2xlUnVsZXM6IFtcbiAgICAgICAgeyBtYXhJbWFnZUNvdW50OiAzIH1cbiAgICAgIF1cbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQ1I6OlJlcG9zaXRvcnknLCB7XG4gICAgICBcIkxpZmVjeWNsZVBvbGljeVwiOiB7XG4gICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgXCJMaWZlY3ljbGVQb2xpY3lUZXh0XCI6IFwie1xcXCJydWxlc1xcXCI6W3tcXFwicnVsZVByaW9yaXR5XFxcIjoxLFxcXCJzZWxlY3Rpb25cXFwiOntcXFwidGFnU3RhdHVzXFxcIjpcXFwiYW55XFxcIixcXFwiY291bnRUeXBlXFxcIjpcXFwiaW1hZ2VDb3VudE1vcmVUaGFuXFxcIixcXFwiY291bnROdW1iZXJcXFwiOjN9LFxcXCJhY3Rpb25cXFwiOntcXFwidHlwZVxcXCI6XFxcImV4cGlyZVxcXCJ9fV19XCJcbiAgICAgIH1cbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NhbGN1bGF0ZSByZXBvc2l0b3J5IFVSSScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgcmVwbyA9IG5ldyBlY3IuUmVwb3NpdG9yeShzdGFjaywgJ1JlcG8nKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB1cmkgPSByZXBvLnJlcG9zaXRvcnlVcmk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgYXJuU3BsaXQgPSB7ICdGbjo6U3BsaXQnOiBbICc6JywgeyAnRm46OkdldEF0dCc6IFsgJ1JlcG8wMkFDODZDRicsICdBcm4nIF0gfSBdIH07XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh1cmkpLCB7ICdGbjo6Sm9pbic6IFsgJycsIFtcbiAgICAgIHsgJ0ZuOjpTZWxlY3QnOiBbIDQsIGFyblNwbGl0IF0gfSxcbiAgICAgICcuZGtyLmVjci4nLFxuICAgICAgeyAnRm46OlNlbGVjdCc6IFsgMywgYXJuU3BsaXQgXSB9LFxuICAgICAgJy5hbWF6b25hd3MuY29tLycsXG4gICAgICB7IFJlZjogJ1JlcG8wMkFDODZDRicgfVxuICAgIF1dfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW1wb3J0IHdpdGggY29uY3JldGUgYXJuJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCByZXBvMiA9IGVjci5SZXBvc2l0b3J5LmZyb21SZXBvc2l0b3J5QXJuKHN0YWNrLCAncmVwbycsICdhcm46YXdzOmVjcjp1cy1lYXN0LTE6NTg1Njk1MDM2MzA0OnJlcG9zaXRvcnkvZm9vL2Jhci9mb28vZm9vbycpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocmVwbzIucmVwb3NpdG9yeUFybiksICdhcm46YXdzOmVjcjp1cy1lYXN0LTE6NTg1Njk1MDM2MzA0OnJlcG9zaXRvcnkvZm9vL2Jhci9mb28vZm9vbycpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocmVwbzIucmVwb3NpdG9yeU5hbWUpLCAnZm9vL2Jhci9mb28vZm9vbycpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ZhaWxzIGlmIGltcG9ydGluZyB3aXRoIHRva2VuIGFybiBhbmQgbm8gbmFtZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOL1RIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBlY3IuUmVwb3NpdG9yeS5mcm9tUmVwb3NpdG9yeUFybihzdGFjaywgJ2FybicsIGNkay5Gbi5nZXRBdHQoJ0Jvb20nLCAnQm9vbScpLnRvU3RyaW5nKCkpLFxuICAgICAgL1xcXCJyZXBvc2l0b3J5QXJuXFxcIiBpcyBhIGxhdGUtYm91bmQgdmFsdWUsIGFuZCB0aGVyZWZvcmUgXFxcInJlcG9zaXRvcnlOYW1lXFxcIiBpcyByZXF1aXJlZFxcLiBVc2UgXFxgZnJvbVJlcG9zaXRvcnlBdHRyaWJ1dGVzXFxgIGluc3RlYWQvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdpbXBvcnQgd2l0aCB0b2tlbiBhcm4gYW5kIHJlcG9zaXRvcnkgbmFtZSAoc2VlIGF3c2xhYnMvYXdzLWNkayMxMjMyKScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVwbyA9IGVjci5SZXBvc2l0b3J5LmZyb21SZXBvc2l0b3J5QXR0cmlidXRlcyhzdGFjaywgJ1JlcG8nLCB7XG4gICAgICByZXBvc2l0b3J5QXJuOiBjZGsuRm4uZ2V0QXR0KCdCb29tJywgJ0FybicpLnRvU3RyaW5nKCksXG4gICAgICByZXBvc2l0b3J5TmFtZTogY2RrLkZuLmdldEF0dCgnQm9vbScsICdOYW1lJykudG9TdHJpbmcoKVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocmVwby5yZXBvc2l0b3J5QXJuKSwgeyAnRm46OkdldEF0dCc6IFsgJ0Jvb20nLCAnQXJuJyBdIH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocmVwby5yZXBvc2l0b3J5TmFtZSksIHsgJ0ZuOjpHZXRBdHQnOiBbICdCb29tJywgJ05hbWUnIF0gfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ltcG9ydCBvbmx5IHdpdGggYSByZXBvc2l0b3J5IG5hbWUgKGFybiBpcyBkZWR1Y2VkKScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVwbyA9IGVjci5SZXBvc2l0b3J5LmZyb21SZXBvc2l0b3J5TmFtZShzdGFjaywgJ2p1c3QtbmFtZScsICdteS1yZXBvJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShyZXBvLnJlcG9zaXRvcnlBcm4pLCB7XG4gICAgICAnRm46OkpvaW4nOiBbICcnLCBbXG4gICAgICAgICdhcm46JyxcbiAgICAgICAgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSxcbiAgICAgICAgJzplY3I6JyxcbiAgICAgICAgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgJzonLFxuICAgICAgICB7IFJlZjogJ0FXUzo6QWNjb3VudElkJyB9LFxuICAgICAgICAnOnJlcG9zaXRvcnkvbXktcmVwbycgXVxuICAgICAgXVxuICAgIH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocmVwby5yZXBvc2l0b3J5TmFtZSksICdteS1yZXBvJyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2FybkZvckxvY2FsUmVwb3NpdG9yeSBjYW4gYmUgdXNlZCB0byByZW5kZXIgYW4gQVJOIGZvciBhIGxvY2FsIHJlcG9zaXRvcnknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHJlcG9OYW1lID0gY2RrLkZuLmdldEF0dCgnQm9vbScsICdOYW1lJykudG9TdHJpbmcoKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCByZXBvID0gZWNyLlJlcG9zaXRvcnkuZnJvbVJlcG9zaXRvcnlBdHRyaWJ1dGVzKHN0YWNrLCAnUmVwbycsIHtcbiAgICAgIHJlcG9zaXRvcnlBcm46IGVjci5SZXBvc2l0b3J5LmFybkZvckxvY2FsUmVwb3NpdG9yeShyZXBvTmFtZSwgc3RhY2spLFxuICAgICAgcmVwb3NpdG9yeU5hbWU6IHJlcG9OYW1lXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShyZXBvLnJlcG9zaXRvcnlOYW1lKSwgeyAnRm46OkdldEF0dCc6IFsgJ0Jvb20nLCAnTmFtZScgXSB9KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHJlcG8ucmVwb3NpdG9yeUFybiksIHtcbiAgICAnRm46OkpvaW4nOiBbICcnLCBbXG4gICAgICAnYXJuOicsXG4gICAgICB7IFJlZjogJ0FXUzo6UGFydGl0aW9uJyB9LFxuICAgICAgJzplY3I6JyxcbiAgICAgIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAnOicsXG4gICAgICB7IFJlZjogJ0FXUzo6QWNjb3VudElkJyB9LFxuICAgICAgJzpyZXBvc2l0b3J5LycsXG4gICAgICB7ICdGbjo6R2V0QXR0JzogWyAnQm9vbScsICdOYW1lJyBdIH0gXSBdXG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Jlc291cmNlIHBvbGljeScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgcmVwbyA9IG5ldyBlY3IuUmVwb3NpdG9yeShzdGFjaywgJ1JlcG8nKTtcblxuICAgIC8vIFdIRU5cbiAgICByZXBvLmFkZFRvUmVzb3VyY2VQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoeyBhY3Rpb25zOiBbJyonXSB9KSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUNSOjpSZXBvc2l0b3J5Jywge1xuICAgICAgUmVwb3NpdG9yeVBvbGljeVRleHQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBcIipcIixcbiAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXZlbnRzJzoge1xuICAgICdvbkltYWdlUHVzaGVkIHdpdGhvdXQgaW1hZ2VUYWcgY3JlYXRlcyB0aGUgY29ycmVjdCBldmVudCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCByZXBvID0gbmV3IGVjci5SZXBvc2l0b3J5KHN0YWNrLCAnUmVwbycpO1xuXG4gICAgICByZXBvLm9uQ2xvdWRUcmFpbEltYWdlUHVzaGVkKCdFdmVudFJ1bGUnLCB7XG4gICAgICAgIHRhcmdldDoge1xuICAgICAgICAgIGJpbmQ6ICgpID0+ICh7IGFybjogJ0FSTicsIGlkOiAnSUQnIH0pXG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RXZlbnRzOjpSdWxlJywge1xuICAgICAgICBcIkV2ZW50UGF0dGVyblwiOiB7XG4gICAgICAgICAgXCJzb3VyY2VcIjogW1xuICAgICAgICAgICAgXCJhd3MuZWNyXCIsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBcImRldGFpbFwiOiB7XG4gICAgICAgICAgICBcImV2ZW50TmFtZVwiOiBbXG4gICAgICAgICAgICAgIFwiUHV0SW1hZ2VcIixcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcInJlcXVlc3RQYXJhbWV0ZXJzXCI6IHtcbiAgICAgICAgICAgICAgXCJyZXBvc2l0b3J5TmFtZVwiOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgXCJSZWZcIjogXCJSZXBvMDJBQzg2Q0ZcIlxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgXCJTdGF0ZVwiOiBcIkVOQUJMRURcIixcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfVxuICB9LFxuXG4gICdyZW1vdmFsIHBvbGljeSBpcyBcIlJldGFpblwiIGJ5IGRlZmF1bHQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBlY3IuUmVwb3NpdG9yeShzdGFjaywgJ1JlcG8nKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQ1I6OlJlcG9zaXRvcnknLCB7XG4gICAgICBcIlR5cGVcIjogXCJBV1M6OkVDUjo6UmVwb3NpdG9yeVwiLFxuICAgICAgXCJEZWxldGlvblBvbGljeVwiOiBcIlJldGFpblwiXG4gICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdcIkRlbGV0ZVwiIHJlbW92YWwgcG9saWN5IGNhbiBiZSBzZXQgZXhwbGljaXRseScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGVjci5SZXBvc2l0b3J5KHN0YWNrLCAnUmVwbycsIHtcbiAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUjo6UmVwb3NpdG9yeScsIHtcbiAgICAgIFwiVHlwZVwiOiBcIkFXUzo6RUNSOjpSZXBvc2l0b3J5XCIsXG4gICAgICBcIkRlbGV0aW9uUG9saWN5XCI6IFwiRGVsZXRlXCJcbiAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2dyYW50IGFkZHMgYXBwcm9wcmlhdGUgcmVzb3VyY2UtKicodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXBvID0gbmV3IGVjci5SZXBvc2l0b3J5KHN0YWNrLCAnVGVzdEhhcm5lc3NSZXBvJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgcmVwby5ncmFudFB1bGwobmV3IGlhbS5BbnlQcmluY2lwYWwoKSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUNSOjpSZXBvc2l0b3J5Jywge1xuICAgICAgXCJSZXBvc2l0b3J5UG9saWN5VGV4dFwiOiB7XG4gICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIkFjdGlvblwiOiBbXG4gICAgICAgICAgICAgIFwiZWNyOkJhdGNoQ2hlY2tMYXllckF2YWlsYWJpbGl0eVwiLFxuICAgICAgICAgICAgICBcImVjcjpHZXREb3dubG9hZFVybEZvckxheWVyXCIsXG4gICAgICAgICAgICAgIFwiZWNyOkJhdGNoR2V0SW1hZ2VcIlxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IFwiKlwiLFxuICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiBcIipcIixcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgfVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxufTtcbiJdfQ==