# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['molgri', 'molgri.scripts', 'scripts']

package_data = \
{'': ['*'], 'molgri': ['examples/*']}

install_requires = \
['matplotlib>=3.0',
 'mendeleev',
 'networkx>=2.6',
 'numpy>=1.0',
 'pandas>=1.0',
 'scipy>=1.0',
 'seaborn>=0.12']

entry_points = \
{'console_scripts': ['molgri-grid = '
                     'molgri.scripts.generate_grid:run_generate_grid',
                     'molgri-io = molgri.scripts.set_up_io:parse_and_create',
                     'molgri-pt = molgri.scripts.generate_pt:run_generate_pt']}

setup_kwargs = {
    'name': 'molgri',
    'version': '1.0.5',
    'description': 'Generate molecular pseudotrajectories based on rotational grids.',
    'long_description': '# molecularRotationalGrids\nThe python package ```molgri``` has three main purposes: 1) generation of rotation grids, 2) analysis of\nsaid grids and 3) generation of pseudotrajectories (PTs). PTs are .gro files with several timesteps in\nwhich the interaction space of two molecules is systematically explored. We provide user-friendly,\none-line scripts for rotation grid generation and analysis as well as pseudotrajectory generation and\ngive instructions how to use PTS with external tools like VMD and GROMACS for further analysis.\n\nBelow, we show examples of rotation grids created with different algorithms and figures of\na protein-ion pseudotrajectory as well as some analysis plots. All plots and animations are created\ndirectly with the ```molgri``` package, except the PT plot where the output of ```molgri``` is drawn\nusing VMD.\n\n\n<p float="left">\n    <img src="/readme_images/ico_630_grid.gif" width="48%">\n    <img src="/readme_images/systemE_1000_grid.gif" width="48%">\n</p>\n\n<p float="left">\n    <img src="/readme_images/systemE_1000_uniformity.png" width="30%">\n    <img src="/readme_images/set_up_30_full_color.png" width="30%">\n    <img src="/readme_images/systemE_1000_convergence.png" width="30%">\n</p>\n\n\n\n## Installation\n\nThis is a python library that can be easily installed using:\n\n```\npip install molgri\n```\n\n\n\n## Running examples\n\nTo explore the capabilities of the package, the user is encouraged to run\nthe following example commands (the commands should all be executed in the\nsame directory, we recommend an initially empty directory).\n\n```\nmolgri-io --examples\nmolgri-grid -N 250 -algo ico --draw --animate --animate_ordering --statistics\nmolgri-pt -m1 H2O -m2 NH3 -rotgrid cube3D_15 -transgrid "range(1, 5, 2)"\n```\n\n****The first-line command ```molgri-io``` creates the 📂 input/ and\n📂 output/ folder structure. This command should be run in each new\ndirectory before running other commands. The optional\nflag ```--examples``` provides some sample input files that we will use later.\n\nThe second command ```molgri-grid``` is used to generate rotation grids. It is\nnecessary to specify the number of grid points ```-N``` and the algorithm \n```-algo``` (select from: systemE, randomE, randomQ, cube4D, cube3D, ico; we\nrecommend ico). Other flags describe optional figures and animations to save. All\ngenerated files can be found in the output/ folder.\n\nThe last command ```molgri-pt``` creates a pseudotrajectory. This scripts needs\ntwo file inputs that should be provided in input/. Both must be\n.gro files, each containing a single molecule. Due to the flag\n```-m1 H2O``` the program will look for a file input/H2O.gro\nand use it as a fixed molecule in the pseudotrajectory. The flag ```-m2```\ngives the name of the file with the other molecule, which will be mobile\nin the simulation. Finally, the user needs to specify the rotational grid\nin form ```-rotgrid algorithm_N``` (see algorithm names above) and the translational grid after the\nflag ```-transgrid``` in one of the\nfollowing formats: a list of distances (in nm), linspace(start, stop, num) \nor range(start, stop, step). The argument should be surrounded by quotation\nmarks. Some acceptable translation grid arguments would be:\n\n```-transgrid "(1, 3, 5)"``` -> use distances 1nm, 3nm and 5nm\n\n```-transgrid "linspace(1, 3, 5)"``` -> use 5 equally spaced points\nbetween 1nm and 3nm\n\n```-transgrid "range(1, 3, 0.5)"``` -> use distances between 1nm and 3nm\nin 0.5nm increments\n\nAll flags starting with ```--``` are optional and can be omitted for faster\ncalculations. Remember that you can always add the flag ```--help``` to get\nfurther instructions.****\n\n\n## Using outputs\n\nThe pseudotrajectory .gro files can be used as regularly generated .gro files. We show how they can be\ndisplayed with VMD or used for GROMACS calculations, but the user is free to use them as inputs to any other\ntool.\n\n#### Displaying pseudotrajectory\n\nTo display the example pseudotrajectory we created in the previous section with VMD, stay in the same\ndirectory and run\n\n```\nvmd output/pt_files/H2O_NH3_cube3D_15_full.gro\n```\n\nor on a windows computer\n\n```\nstart vmd output/pt_files/H2O_NH3_cube3D_15_full.gro\n```\n\nThen, to fully display a pseudotrajectory, it is often helpful to change the display style and to display\nseveral or all frames at once. We suggest using the following commands within the VMD command line:\n\n```\nmol modstyle 0 0 VDW\nmol drawframes 0 0 1:1:241\n```\n\nThe first one displays the molecules as spheres with van der Waals radii and the second draws frames of\nthe pseudotrajectory in a form &lt;start>:&lt;step>:&lt;stop>.\nA useful trick is to use the number of rotations as &lt;step> (in this case that would be 15) - this displays one structure per mass point\nwithout considering internal rotations. This number is also written in the name of the .gro file.\nIf you want to display all frames, you can use any large number\nfor &lt;num_frames>, it does not need to correspond exactly to the number of frames.\n\n#### Calculating energy along a pseudotrajectory\nOften, a pseudotrajectory is used to explore where reagions of high and low energies lie when molecules\napproach each other. Since a range of timesteps sampling important rotations and translations\nis already provided in a PT, there is no need to run a real\nsimulation. Therefore, the flag ```-rerun``` is always used while dealing with PTs in GROMACS. This\nsetting saves time that would else be used for running an integrator and propagating positions.\n\n\nTo use GROMACS with PTs, the user must also provide a topology file which includes both molecules used in a pseudotrajectory.\nWe will assume that this file is named topol.top. Lastly, we need a GROMACS run file that we will name\nmdrun.mdp. This file records GROMACS parameters and can be used as in normal simulations, but note that\nsome parameters (e.g. integrator, dt) are meaningless for a pseudotrajectory without actual dynamics.\nThen, the energy along a pseudotrajectory can be calculated as follows:\n\n```\ngmx22 grompp -f mdrun.mdp -c H2O_NH3_cube3D_15_full.gro -p topol.top -o result.tpr   \ngmx22 trjconv -f H2O_NH3_cube3D_15_full.gro -s result.tpr -o result.trr\ngmx22 mdrun -s result.tpr -rerun result.trr\ngmx22 energy -f ener.edr -o full_energy.xvg\n```\n\n\n## Complex applications: using python package\n\nUsers who would like to build custom grids,  pseudotrajectories or sets of rotations can import ```molgri```\nas a python package (following installation described above) and work with all provided modules. Documentation\nis provided in form of docstrings or available in a compiled version at our github repository \n[in docs folder](https://github.com/bkellerlab/molecularRotationalGrids/tree/main/docs/molgri).',
    'author': 'Hana Zupan',
    'author_email': 'hana.zupan@fu-berlin.de',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://www.bcp.fu-berlin.de/en/chemie/chemie/forschung/PhysTheoChem/agkeller/index.html',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7',
}


setup(**setup_kwargs)
