"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certbot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const r53 = require("@aws-cdk/aws-route53");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const subscriptions = require("@aws-cdk/aws-sns-subscriptions");
const cdk = require("@aws-cdk/core");
const oneTimeEvents = require("@renovosolutions/cdk-library-one-time-event");
class Certbot extends cdk.Construct {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        let bucket;
        // Create a bucket if one is not provided
        if (props.bucket === undefined) {
            bucket = new s3.Bucket(this, 'bucket', {
                objectOwnership: s3.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                removalPolicy: props.removalPolicy || cdk.RemovalPolicy.RETAIN,
                autoDeleteObjects: (_b = props.enableObjectDeletion) !== null && _b !== void 0 ? _b : false,
                versioned: true,
                lifecycleRules: [{
                        enabled: true,
                        abortIncompleteMultipartUploadAfter: cdk.Duration.days(1),
                    }],
                encryption: s3.BucketEncryption.S3_MANAGED,
                enforceSSL: true,
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            });
        }
        else {
            bucket = props.bucket;
        }
        const functionDir = path.join(__dirname, '../function');
        // Create an SNS topic if one is not provided and add the defined email to it
        let snsTopic = (_c = props.snsTopic) !== null && _c !== void 0 ? _c : new sns.Topic(this, 'topic');
        if (props.snsTopic === undefined) {
            snsTopic.addSubscription(new subscriptions.EmailSubscription(props.letsencryptEmail));
        }
        // Set property defaults
        let layers = (_d = props.layers) !== null && _d !== void 0 ? _d : [];
        let runOnDeploy = (_e = props.runOnDeploy) !== null && _e !== void 0 ? _e : true;
        let functionDescription = (_f = props.functionDescription) !== null && _f !== void 0 ? _f : 'Certbot Renewal Lambda for domain ' + props.letsencryptDomains.split(',')[0];
        let enableInsights = (_g = props.enableInsights) !== null && _g !== void 0 ? _g : false;
        let insightsARN = (_h = props.insightsARN) !== null && _h !== void 0 ? _h : 'arn:aws:lambda:' + cdk.Stack.of(this).region + ':580247275435:layer:LambdaInsightsExtension:14';
        // Set up role policies
        let managedPolicies = [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')];
        if (enableInsights) {
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
            layers.push(lambda.LayerVersion.fromLayerVersionArn(this, 'insightsLayer', insightsARN));
        }
        const snsPolicy = new iam.ManagedPolicy(this, 'snsPolicy', {
            description: 'Allow the Certbot function to notify an SNS topic upon completion.',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['sns:Publish'],
                    resources: [snsTopic.topicArn],
                }),
            ],
        });
        let hostedZones = [];
        props.hostedZoneNames.forEach((domainName) => {
            hostedZones.push(r53.HostedZone.fromLookup(this, 'zone' + domainName, {
                domainName,
                privateZone: false,
            }).hostedZoneArn);
        });
        const r53Policy = new iam.ManagedPolicy(this, 'r53Policy', {
            description: 'Allow the Certbot function to perform DNS verification.',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['route53:ListHostedZones'],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        'route53:GetChange',
                        'route53:ChangeResourceRecordSets',
                    ],
                    resources: ['arn:aws:route53:::change/*'].concat(hostedZones),
                }),
            ],
        });
        const acmPolicy = new iam.ManagedPolicy(this, 'acmPolicy', {
            description: 'Allow the Certbot function to import and list certificates.',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        'acm:ListCertificates',
                        'acm:ImportCertificate',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['acm:DescribeCertificate'],
                    resources: ['arn:aws:acm:' + cdk.Stack.of(this).region + ':' + cdk.Stack.of(this).account + ':certificate/*'],
                }),
            ],
        });
        managedPolicies.push(snsPolicy);
        managedPolicies.push(r53Policy);
        managedPolicies.push(acmPolicy);
        const role = new iam.Role(this, 'role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        bucket.grantWrite(role);
        // Create the Lambda function
        this.handler = new lambda.Function(this, 'handler', {
            runtime: lambda.Runtime.PYTHON_3_8,
            role,
            code: lambda.Code.fromAsset(functionDir + '/function.zip'),
            handler: 'index.handler',
            functionName: props.functionName,
            description: functionDescription,
            environment: {
                LETSENCRYPT_DOMAINS: props.letsencryptDomains,
                LETSENCRYPT_EMAIL: props.letsencryptEmail,
                CERTIFICATE_BUCKET: bucket.bucketName,
                OBJECT_PREFIX: props.objectPrefix || '',
                REISSUE_DAYS: (props.reIssueDays === undefined) ? '30' : String(props.reIssueDays),
                PREFERRED_CHAIN: props.preferredChain || 'None',
                NOTIFICATION_SNS_ARN: snsTopic.topicArn,
            },
            layers,
            timeout: props.timeout || cdk.Duration.seconds(180),
        });
        // Add function triggers
        new events.Rule(this, 'trigger', {
            schedule: props.schedule || events.Schedule.cron({ minute: '0', hour: '0', weekDay: '1' }),
            targets: [new targets.LambdaFunction(this.handler)],
        });
        if (runOnDeploy) {
            new events.Rule(this, 'triggerImmediate', {
                schedule: new oneTimeEvents.OnDeploy(this, 'schedule', {
                    offsetMinutes: props.runOnDeployWaitMinutes || 10,
                }).schedule,
                targets: [new targets.LambdaFunction(this.handler)],
            });
        }
    }
}
exports.Certbot = Certbot;
_a = JSII_RTTI_SYMBOL_1;
Certbot[_a] = { fqn: "@renovosolutions/cdk-library-certbot.Certbot", version: "1.1.8" };
//# sourceMappingURL=data:application/json;base64,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