"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? true;
        this.failOnWarning = props?.failOnWarning ?? false;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNagApp", version: "0.4.1" };
class PDKNagAspect extends cdk_nag_1.AwsSolutionsChecks {
    constructor(app) {
        super({
            verbose: true,
            reports: true,
        });
        this.app = app;
    }
    visit(node) {
        super.visit(node);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            console.info(`Unable to add Nag Suppression for path: ${path} as it does not exist.`);
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            return `${stack.stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNag", version: "0.4.1" };
//# sourceMappingURL=data:application/json;base64,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