# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['shpyx']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'shpyx',
    'version': '0.0.22',
    'description': 'Run shell commands in Python',
    'long_description': '<p align="center">\n  <img src="https://github.com/Apakottur/shpyx/blob/main/shpyx.png?raw=true" />\n</p>\n\n[![PyPI](https://img.shields.io/pypi/v/shpyx?logo=pypi&logoColor=white&style=for-the-badge)](https://pypi.org/project/shpyx/)\n[![Downloads](https://img.shields.io/pypi/dm/shpyx?logo=pypi&logoColor=white&style=for-the-badge)](https://pypi.org/project/shpyx/)\n[![Python](https://img.shields.io/pypi/pyversions/shpyx?logo=pypi&logoColor=white&style=for-the-badge)](https://pypi.org/project/shpyx/)\n\n**shpyx** is a simple, lightweight and typed library for running shell commands in Python.\n\nUse `shpyx.run` to run a shell command in a subprocess:\n\n```python\n>>> import shpyx\n>>> shpyx.run("echo 1").return_code\n0\n>>> shpyx.run("echo 1").stdout\n\'1\\n\'\n>>> shpyx.run("echo 1").stderr\n\'\'\n>>> shpyx.run("echo 1")\nShellCmdResult(cmd=\'echo 1\', stdout=\'1\\n\', stderr=\'\', all_output=\'1\\n\', return_code=0)\n```\n\n## Installation\n\nInstall with `pip`:\n\n```shell\npip install shpyx\n```\n\n## How Tos\n\n### Run a command\n\nIn string format:\n\n```python\n>>> shpyx.run("echo 1")\nShellCmdResult(cmd=\'echo 1\', stdout=\'1\\n\', stderr=\'\', all_output=\'1\\n\', return_code=0)\n```\n\nIn list format:\n\n```python\n>>> shpyx.run(["echo", ["1"])\nShellCmdResult(cmd=\'echo 1\', stdout=\'1\\n\', stderr=\'\', all_output=\'1\\n\', return_code=0)\n```\n\n### Run a command and print live output\n\n```python\n>>> shpyx.run("echo 1", log_output=True)\n1\nShellCmdResult(cmd=\'echo 1\', stdout=\'1\\n\', stderr=\'\', all_output=\'1\\n\', return_code=0)\n```\n\n### Run a command with shell specific logic\n\nWhen the argument to `run` is a string, an actual shell is created in the subprocess and shell logic can be used.\nFor example, the pipe operator can be used in bash/sh:\n\n```python\n>>> shpyx.run("seq 1 5 | grep \'2\'")\nShellCmdResult(cmd="seq 1 5 | grep \'2\'", stdout=\'2\\n\', stderr=\'\', all_output=\'2\\n\', return_code=0)\n```\n\n### Create a custom runner\n\nUse a custom runner to override the execution defaults, and not have to pass them to every call.\n\nFor example, we can change the default value of `log_cmd`, so that all commands are logged:\n\n```python\n>>> shpyx.run("echo 1")\nShellCmdResult(cmd=\'echo 1\', stdout=\'1\\n\', stderr=\'\', all_output=\'1\\n\', return_code=0)\n\n>>> shpyx.run("echo 1", log_cmd=True)\nRunning: echo 1\nShellCmdResult(cmd=\'echo 1\', stdout=\'1\\n\', stderr=\'\', all_output=\'1\\n\', return_code=0)\n\n>>> runner = shpyx.Runner(log_cmd=True)\n>>> runner.run("echo 1")\nRunning: echo 1\nShellCmdResult(cmd=\'echo 1\', stdout=\'1\\n\', stderr=\'\', all_output=\'1\\n\', return_code=0)\n```\n\n### Propagating terminal control sequences\n\nNote: as of now this is only supported for Unix environments.\n\nSome commands, like `psql`, might output special characters used for terminal management like cursor movement and\ncolors. For example, the `psql` command is used to start an interactive shell against a Postgres DB:\n\n```python\nshpyx.run(f"psql -h {host} -p {port} -U {user} -d {database}", log_output=True)\n```\n\nHowever, the above call will not work as good as running `psql` directly, due to terminal control sequences not being\nproperly propagated. To make it work well, we need to use the [script](https://man7.org/linux/man-pages/man1/script.1.html)\nutility which will properly propagate all control sequences:\n\n```python\n# Linux:\nshpyx.run(f"script -q -c \'psql -h {host} -p {port} -U {user} -d {database}\'", log_output=True)\n# MacOS:\nshpyx.run(f"script -q psql -h {host} -p {port} -U {user} -d {database}", log_output=True)\n\n```\n\nshpyx provides a keyword argument that does this wrapping automatically, `unix_raw`:\n\n```python\nshpyx.run(f"psql -h {host} -p {port} -U {user} -d {database}", log_output=True, unix_raw=True)\n```\n\nThe flag is disabled by default, and should only be used for interactive commands like `psql`.\n\n## API Reference\n\nThe following arguments are supported by `Runner`:\n\n| Name                 | Description                                                                | Default |\n| -------------------- | -------------------------------------------------------------------------- | ------- |\n| `log_cmd`            | Log the executed command.                                                  | `False` |\n| `log_output`         | Log the live output of the command (while it is being executed).           | `False` |\n| `verify_return_code` | Raise an exception if the shell return code of the command is not `0`.     | `True`  |\n| `verify_stderr`      | Raise an exception if anything was written to stderr during the execution. | `False` |\n| `use_signal_names`   | Log the name of the signal corresponding to a non-zero error code.         | `True`  |\n\nThe following arguments are supported by `run`:\n\n| Name                 | Description                                                                | Default                  |\n| -------------------- | -------------------------------------------------------------------------- | ------------------------ |\n| `log_cmd`            | Log the executed command.                                                  | `Runner default`         |\n| `log_output`         | Log the live output of the command (while it is being executed).           | `Runner default`         |\n| `verify_return_code` | Raise an exception if the shell return code of the command is not `0`.     | `Runner default`         |\n| `verify_stderr`      | Raise an exception if anything was written to stderr during the execution. | `Runner default`         |\n| `use_signal_names`   | Log the name of the signal corresponding to a non-zero error code.         | `Runner default`         |\n| `env`                | Environment variables to set during the execution of the command.          | `Same as parent process` |\n| `exec_dir`           | Custom path to execute the command in (defaults to current directory).     | `Same as parent process` |\n| `unix_raw`           | (UNIX ONLY) Whether to use the `script` Unix utility to run the command.   | `False`                  |\n\n## Implementation details\n\n`shpyx` is a wrapper around the excellent [subprocess](https://docs.python.org/3/library/subprocess.html) module, aiming\nto concentrate all the different API functions (`Popen`/`communicate`/`poll`/`wait`) into a single function - `shpyx.run`.\n\nWhile the core API logic is fully supported on both Unix and Windows systems, there is some OS specific code for minor quality-of-life\nimprovements.\nFor example, on non Windows systems, [fcntl](https://docs.python.org/3/library/fcntl.html) is used to configure the subprocess to\nalways be incorruptible (which means one can CTRL-C out of any command).\n\n## Security\n\nThe call to `subprocess.Popen` uses `shell=True` when the input to `run` is a string (to support shell logic like bash piping).\nThis means that an actual system shell is being created, and the subprocess has the permissions of the main Python process.\n\nIt is therefore recommended not pass any untrusted input to `shpyx.run`.\n\nFor more info, see [security considerations](https://docs.python.org/3/library/subprocess.html#security-considerations).\n\n## Useful links\n\nRelevant Python libraries:\n\n- [subprocess](https://docs.python.org/3/library/subprocess.html)\n- [shlex](https://docs.python.org/3/library/shlex.html)\n\nOther user libraries for running shell commands in Python:\n\n- [sarge](https://github.com/vsajip/sarge)\n- [sh](https://github.com/amoffat/sh)\n\n## Contributing\n\nTo contribute simply open a PR with your changes.\n\nTests, linters and type checks are run in CI through GitHub Actions.\n\n### Running checks locally\n\nTo run checks locally, start by installing all the development dependencies:\n\n```shell\npoetry install\n```\n\nTo run the linters use `pre-commit`:\n\n```shell\npre-commit run -a\n```\n\nTo run the unit tests use `pytest`:\n\n```shell\npytest -c tests/pytest.ini tests\n```\n\nTo run type checks use `mypy`:\n\n```shell\nmypy --config-file shpyx/mypy.ini shpyx tests\n```\n\nTo trigger a deployment of a new version upon merge, bump the version number in `pyproject.toml`.\n',
    'author': 'Yossi Rozantsev',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/Apakottur/shpyx',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
